/*
 * drivers/misc/coherent_memalloc.c
 *
 * Copyright (C) 2011 DSP Group
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/ioport.h>
#include <linux/list.h>
#include <linux/sched.h>
#include <linux/dma-mapping.h>
#include <linux/coherent_memalloc.h>

/* module description */
MODULE_LICENSE("GPL2");
MODULE_AUTHOR("DSP Group");
MODULE_DESCRIPTION("Coherent memory allocator");

#define BUFFERS_PER_FILE			64
#define MAX_KMALLOC_BUFFER_SIZE		(16*1024)
#define REALLOC_FACTOR				2

static int memalloc_major = 0;  /* dynamic */

typedef struct _buffer_t
{
	dma_addr_t			phys_addr;
	void				*virt_addr;
	int					size;
	struct list_head	list;
} buffer_t;

typedef struct _filp_private_data
{
	buffer_t			*buffers;
	int					num_buffers;
	spinlock_t			lock;
	struct list_head	empty_list;
	struct list_head	allocated_list;
} filp_private_data;

static int alloc_buffer(buffer_t *buf, int size)
{
	buf->virt_addr = dma_alloc_coherent(NULL, size, &buf->phys_addr, GFP_KERNEL | GFP_DMA);
	if (buf->virt_addr == NULL) {
		return -1;
	}

	buf->size = size;

	return 0;
}

static void free_buffer(buffer_t *buf)
{
	dma_free_coherent(NULL, buf->size, buf->virt_addr, buf->phys_addr);
}

static int realloc_buffers(filp_private_data *private_data)
{
	buffer_t	*realloc_buffers;
	int			num_buffers;
	int			i;

	PDEBUG("realloc_buffers\n");

	/* empty_list is empty - realloc */
	num_buffers = private_data->num_buffers * REALLOC_FACTOR;
	realloc_buffers = krealloc( private_data->buffers,
								sizeof(buffer_t) * num_buffers,
								GFP_KERNEL | GFP_DMA);

	if (!realloc_buffers) {
		return -1;
	}

	/* reinitialise the lists. we don't need their content since we know */
	/* all buffers are currently at the allocated list */
	INIT_LIST_HEAD(&(private_data->empty_list));
	INIT_LIST_HEAD(&(private_data->allocated_list));

	/* add all buffers into allocated list */
	for (i = 0 ; i < private_data->num_buffers ; i++) {
		list_add_tail(&(realloc_buffers[i].list), &(private_data->allocated_list));
	}

	/* add new buffers into empty_list */
	for (i = private_data->num_buffers ; i < num_buffers ; i++) {
		list_add_tail(&(realloc_buffers[i].list), &(private_data->empty_list));
	}		

	/* update private_data fields */
	private_data->buffers = realloc_buffers;
	private_data->num_buffers = num_buffers;

	return 0;
}

static int alloc_memory(unsigned *busaddr, unsigned int size, struct file *filp)
{
	int					ret;
	buffer_t			*buf;
	filp_private_data	*private_data = filp->private_data;

	/* add new buffers using realloc if there are no more empty buffers */
	if (list_empty(&(private_data->empty_list))) {
		ret = realloc_buffers(private_data);

		if (ret) {
			printk("coherent_memalloc: failed to realloc buffers memory\n");
			return ret;
		}
	}

	/* try allocating the memory... */
	buf = list_first_entry(&(private_data->empty_list), buffer_t, list);
	ret = alloc_buffer(buf, size);

	if(ret) {
		printk("coherent_memalloc: Allocation FAILED: size = %d\n", size);
		*busaddr = 0;
	}
	else {
		PDEBUG("coherent_memalloc: Allocation OK: size: %d\n", size);

		/* remove buf from empty_list and insert into allocated_list */
		list_del(&(buf->list));
		list_add_tail(&(buf->list), &(private_data->allocated_list));
		*busaddr = buf->phys_addr;
	}

	return ret;
}

static int free_memory(unsigned long busaddr, struct file *filp)
{
	buffer_t			*buf;
	struct list_head	*pos;
	filp_private_data	*private_data = filp->private_data;
	int					found = 0;

	/* look for buffer_t with the requested busaddr */
	/* although we are deleting entry from the list, we are not using list_for_each_safe() */
	/* function since we are breakig the loop once we delete the entry */
	list_for_each(pos, &(private_data->allocated_list)) {
		buf = list_entry(pos, buffer_t, list);

		if (buf->phys_addr == busaddr) {
			free_buffer(buf);
			found = 1;

			/* remove from allocated_list and add to empty_list*/
			list_del(&(buf->list));
			list_add_tail(&(buf->list), &(private_data->empty_list));
			break;
		}
	}
	
	if (!found)
		printk("coherent_memalloc: error - can't find buffer for deallocation\n");

	return 0;
}

static int memalloc_ioctl(struct inode *inode, struct file *filp,
                          unsigned int cmd, unsigned long arg)
{
	int err = 0;
	int ret;
	long copy_ret;
	filp_private_data *private_data;

	PDEBUG("coherent_memalloc: ioctl cmd 0x%08x\n", cmd);

	if(inode == NULL || filp == NULL || arg == 0) {
		return -EFAULT;
	}
	/*
	* extract the type and number bitfields, and don't decode
	* wrong cmds: return ENOTTY (inappropriate ioctl) before access_ok()
	*/
	if(_IOC_TYPE(cmd) != MEMALLOC_IOC_MAGIC)
		return -ENOTTY;
	if(_IOC_NR(cmd) > MEMALLOC_IOC_MAXNR)
		return -ENOTTY;

	if(_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void *) arg, _IOC_SIZE(cmd));
	else if(_IOC_DIR(cmd) & _IOC_WRITE)
		err = !access_ok(VERIFY_READ, (void *) arg, _IOC_SIZE(cmd));
	if(err)
		return -EFAULT;

	private_data = filp->private_data;

	switch (cmd)
	{
		case MEMALLOC_IOCXGETBUFFER:
		{
			MemallocParams memparams;

			PDEBUG("GETBUFFER\n");
			spin_lock(&private_data->lock);

			copy_ret = __copy_from_user(&memparams, (const void *) arg, sizeof(memparams));
			if (copy_ret) {
				printk("coherent_memalloc: can't copy %ld bytes from user space\n", copy_ret);
				return -1;
			}

			ret = alloc_memory(&memparams.busAddress, memparams.size, filp);

			copy_ret = __copy_to_user((void *) arg, &memparams, sizeof(memparams));
			if (copy_ret) {
				printk("coherent_memalloc: can't copy %ld bytes to user space\n", copy_ret);
				return -1;
			}

			spin_unlock(&private_data->lock);

			return ret;
		}
		case MEMALLOC_IOCSFREEBUFFER:
		{
			unsigned long busaddr;

			PDEBUG("FREEBUFFER\n");
			spin_lock(&private_data->lock);

			copy_ret = __get_user(busaddr, (unsigned long *) arg);
			if (copy_ret) {
				printk("coherent_memalloc: can't get busaddr from user space\n");
				return -1;
			}

			ret = free_memory(busaddr, filp);

			spin_unlock(&private_data->lock);
			return ret;
		}
	}
	return 0;
}

static int memalloc_open(struct inode *inode, struct file *filp)
{
	int i;
	filp_private_data *private_data;
	
	private_data = kmalloc(sizeof(filp_private_data), GFP_KERNEL);

	if (!private_data) {
		printk("coherent_memalloc: can't allocate private_data for filp\n");
		return -1;
	}

	private_data->buffers = kcalloc(BUFFERS_PER_FILE, sizeof(buffer_t), GFP_KERNEL);

	if (!private_data->buffers) {
		kfree(private_data);
		printk("coherent_memalloc: can't allocate buffers for filp\n");
		return -1;
	}

	/* initialize private_data fields */
	private_data->num_buffers = BUFFERS_PER_FILE;
	private_data->lock = SPIN_LOCK_UNLOCKED;
	INIT_LIST_HEAD(&(private_data->empty_list));
	INIT_LIST_HEAD(&(private_data->allocated_list));

	/* insert all buffers into empty_list */
	for (i = 0 ; i < BUFFERS_PER_FILE ; i++) {
		list_add_tail(&(private_data->buffers[i].list), &(private_data->empty_list));
	}

	filp->private_data = private_data;
	
	PDEBUG("coherent_memalloc: opened\n");
	return 0;
}

static int memalloc_release(struct inode *inode, struct file *filp)
{
	struct list_head	*pos;
	buffer_t			*buf;
	filp_private_data	*private_data;

	private_data = filp->private_data;

	/* free all memory allocated using filp */
	list_for_each(pos, &(private_data->allocated_list)){
			buf = list_entry(pos, buffer_t, list);
			PDEBUG("coherent_memalloc: releasing buffer at physical address %x\n", buf->phys_addr);
			free_buffer(buf);
	}

	/* free private_data */
	kfree(private_data->buffers);
	kfree(private_data);
	filp->private_data = NULL;

	PDEBUG("coherent_memalloc: closed\n");
	return 0;
}

/* VFS methods */
static struct file_operations memalloc_fops = {
	open:memalloc_open,
	release:memalloc_release,
	ioctl:memalloc_ioctl,
};

int __init memalloc_init(void)
{
	int result;

	printk("coherent_memalloc: Linear Memory Allocator, %s \n", "$Revision: 1.10 $");

	result = register_chrdev(memalloc_major, "coherent_memalloc", &memalloc_fops);

	if(result < 0) {
		PDEBUG("coherent_memalloc: unable to get major %d\n", memalloc_major);
		goto err;
	}
	else if(result != 0) {		/* this is for dynamic major */
		memalloc_major = result;
	}

	return 0;

err:
	PDEBUG("coherent_memalloc: module not inserted\n");
	unregister_chrdev(memalloc_major, "coherent_memalloc");
	return result;
}

void __exit memalloc_cleanup(void)
{
	unregister_chrdev(memalloc_major, "coherent_memalloc");

	PDEBUG("coherent_memalloc: module removed\n");
	return;
}

module_init(memalloc_init);
module_exit(memalloc_cleanup);

