/*
 * dp52-core.c - Access DP52 through SPI master controllers
 *
 * (C) Copyright 2011, DSP Group
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/debugfs.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/mfd/core.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <linux/uaccess.h>

#include <linux/mfd/dp52/core.h>
#include "dp52.h"

/* Auxilary measuring delay time */
#define AUX_READ_DELAY_TIME_MS 40

/* convert sample to voltage (VBANDGAP is 1v, so we can measure the range 0v..2v) */
#define WINDOW 15
#define SAMPLE_TO_VOLTAGE(sample)	((2000 * (sample)) / ((1 << (WINDOW)) - 1))

#define DEFAULT_CALIBRATION "64:20:0"
static char *calibration = DEFAULT_CALIBRATION;
module_param(calibration, charp, 0444);
MODULE_PARM_DESC(calibration, "AUX ADC calibration");

/* AUX comparator gain steps x4 */
static const int const gain_steps[] = {
	0, 4, 5, 6, 8, 10, 12, 16, 20, 24, 32, 40, 52, 68, 88, 116, 152, 200,
	264, 344, 404
};

/*************************************************************************************************/

void dp52_power_down(struct dp52 *dp52)
{
	mutex_lock(&dp52->mutex_indirect);

	for(;;) {
		/* restore default configuration */
		dp52_write(dp52, DP52_PMU_CTRL, 0x1);
		dp52_write(dp52, DP52_PMU_EN_SW, 0x7a8);
		/* power-off sequence */
		dp52_write(dp52, DP52_PMU_OFF, 0xaaaa);
		dp52_write(dp52, DP52_PMU_OFF, 0x5555);
		dp52_write(dp52, DP52_PMU_OFF, 0xaaaa);
	}

	mutex_unlock(&dp52->mutex_indirect);
}
EXPORT_SYMBOL(dp52_power_down);

int dp52_read(struct dp52 *dp52, int reg)
{
	int ret;

	if (reg >= DP52_INDIRECT_BASE) {
		mutex_lock(&dp52->mutex_indirect);

		if ((ret = dp52_write(dp52, DP52_SPI_INDADD, reg - DP52_INDIRECT_BASE))) {
			mutex_unlock(&dp52->mutex_indirect);
			return ret;
		}
		ret = dp52_read(dp52, DP52_SPI_INDDAT);

		mutex_unlock(&dp52->mutex_indirect);
	} else {
		u8 txbuf[1];
		u8 rxbuf[2];

		txbuf[0] = (reg & 0x7f) | 0x80;
		ret = spi_write_then_read(dp52->spi, txbuf, 1, rxbuf, 2);
		if (ret == 0)
			ret = (((int)rxbuf[0]) << 8 ) | (int)rxbuf[1];
	}

	return ret;
}
EXPORT_SYMBOL(dp52_read);

int dp52_write(struct dp52 *dp52, int reg, int value)
{
	int ret;

	if (reg >= DP52_INDIRECT_BASE) {
		mutex_lock(&dp52->mutex_indirect);

		if ((ret = dp52_write(dp52, DP52_SPI_INDADD, reg - DP52_INDIRECT_BASE))) {
			mutex_unlock(&dp52->mutex_indirect);
			return ret;
		}
		ret = dp52_write(dp52, DP52_SPI_INDDAT, value);

		mutex_unlock(&dp52->mutex_indirect);
	} else {
		u8 bytes[3];

		bytes[0] = reg;
		bytes[1] = value >> 8;
		bytes[2] = value;

		ret = spi_write(dp52->spi, bytes, 3);
	}

	return ret;
}
EXPORT_SYMBOL(dp52_write);

int dp52_update(struct dp52 *dp52, int reg, int mask, int val)
{
	int reg_val, status;

	mutex_lock(&dp52->mutex_rmw);

	reg_val = dp52_read(dp52, reg);
	if (reg_val < 0) {
		mutex_unlock(&dp52->mutex_rmw);
		return reg_val;
	}

	reg_val &= ~mask;
	reg_val |= val & mask;

	status = dp52_write(dp52, reg, reg_val);

	mutex_unlock(&dp52->mutex_rmw);

	return status;
}
EXPORT_SYMBOL(dp52_update);

int dp52_get_chip_version(struct dp52 *dp52)
{
	int val = dp52_read(dp52, DP52_CMU_CSTSR);

	if (val < 0)
		return val;

	return (val & 0x01e0) >> 5;
}
EXPORT_SYMBOL(dp52_get_chip_version);

/******************************************************************************************/

static int dp52_reg_dump_range(struct dp52 *dp52, char *buf, int size, int count, int start, int end)
{
	int i, j;

	for (i = start; i <= end; i += 4) {
		count += snprintf(buf + count, size - count, "%3x:", i);
		if (count >= size - 1)
			break;

		for (j=0; j<4 && j+i <= end; j++) {
			count += snprintf(buf + count, size - count,
					  " %4x", dp52_read(dp52, i+j));
			if (count >= size - 1)
				break;
		}

		count += snprintf(buf + count, size - count, "\n");
		if (count >= size - 1)
			break;
	}

	return count;
}

static ssize_t dp52_reg_dump(struct dp52 *dp52, char *buf, int size)
{
	int count;

	count = dp52_reg_dump_range(dp52, buf, size, 0, 0x000, 0x07f);

	count += snprintf(buf + count, size - count, "\n");
	if (count >= size - 1)
		return count;

	count = dp52_reg_dump_range(dp52, buf, size, count, 0x100, 0x11f);

	count += snprintf(buf + count, size - count, "\n");
	if (count >= size - 1)
		return count;

	count = dp52_reg_dump_range(dp52, buf, size, count, 0x180, 0x18f);
	if (count >= size - 1)
		return count;

	return count;
}

static ssize_t
dp52_reg_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct dp52 *dp52 = dev_get_drvdata(dev);
	return dp52_reg_dump(dp52, buf, PAGE_SIZE);
}

static DEVICE_ATTR(registers, 0444, dp52_reg_show, NULL);

#ifdef CONFIG_DEBUG_FS
static int dp52_reg_open_file(struct inode *inode, struct file *file)
{
	file->private_data = inode->i_private;
	return 0;
}

static ssize_t dp52_reg_read_file(struct file *file, char __user *user_buf,
			       size_t count, loff_t *ppos)
{
	ssize_t ret;
	struct dp52 *dp52 = file->private_data;
	char *buf = kmalloc(PAGE_SIZE, GFP_KERNEL);
	if (!buf)
		return -ENOMEM;
	ret = dp52_reg_dump(dp52, buf, PAGE_SIZE);
	if (ret >= 0)
		ret = simple_read_from_buffer(user_buf, count, ppos, buf, ret);
	kfree(buf);
	return ret;
}

static ssize_t
dp52_reg_write_file(struct file *file, const char __user *user_buf,
                    size_t count, loff_t *ppos)
{
	char buf[32];
	int buf_size, ret;
	char *start = buf;
	unsigned long reg, value;
	struct dp52 *dp52 = file->private_data;

	buf_size = min(count, (sizeof(buf)-1));
	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;
	buf[buf_size] = 0;

	while (*start == ' ')
		start++;
	reg = simple_strtoul(start, &start, 16);
	if (reg > 0x18f)
		return -EINVAL;
	while (*start == ' ')
		start++;
	if (strict_strtoul(start, 16, &value))
		return -EINVAL;

	if ((ret = dp52_write(dp52, reg, value)))
		return ret;

	return buf_size;
}

static const struct file_operations dp52_reg_fops = {
	.open  = dp52_reg_open_file,
	.read  = dp52_reg_read_file,
	.write = dp52_reg_write_file,
};

static void dp52_debugfs_init(struct dp52 *dp52)
{
	char root_name[32];
	int len;

	len = snprintf(root_name, sizeof(root_name)-1, "dp52-%s",
		dev_name(&dp52->spi->dev));
	root_name[len] = 0;

	dp52->debugfs_root = debugfs_create_dir(root_name, NULL);
	if (!dp52->debugfs_root) {
		dev_warn(&dp52->spi->dev, "cannot create debugfs entry\n");
		return;
	}

	dp52->debugfs_reg = debugfs_create_file("regs", 0644, dp52->debugfs_root,
	                                        dp52, &dp52_reg_fops);
	if (!dp52->debugfs_reg)
		dev_warn(&dp52->spi->dev, "cannot create debugfs entry\n");
}

static void dp52_debugfs_exit(struct dp52 *dp52)
{
	if (dp52->debugfs_reg)
		debugfs_remove_recursive(dp52->debugfs_root);
}
#else
static inline void dp52_debugfs_init(struct dp52 *dp52)
{
}

static inline void dp52_debugfs_exit(struct dp52 *dp52)
{
}
#endif

/******************************************************************************************/

void dp52_aux_mutex_lock(struct dp52 *dp52)
{
	mutex_lock(&dp52->mutex_auxilary);
}
EXPORT_SYMBOL(dp52_aux_mutex_lock);

void dp52_aux_mutex_unlock(struct dp52 *dp52)
{
	mutex_unlock(&dp52->mutex_auxilary);
}
EXPORT_SYMBOL(dp52_aux_mutex_unlock);

static int dp52_aux_read(struct dp52 *dp52, unsigned int mux)
{
	int val, ret, automode = 0;

	/* FIXME: Delay until completion of previous measurement. Probably HW
	 * bug, will be simulated */
	msleep(AUX_READ_DELAY_TIME_MS);

	/*
	 * Disable AUTOMODE for the measurement. Remember if it was enabled to
	 * restore later.
	 */
	val = dp52_read(dp52, DP52_TP_CTL2);
	if (val & DP52_TP_CTL2_AUTOMODE) {
		automode = 1;
		val &= ~DP52_TP_CTL2_AUTOMODE;
		dp52_write(dp52, DP52_TP_CTL2, val);
		val &= ~ DP52_TP_RDY;
		dp52_write(dp52, DP52_TP_CTL2, val);
	}

	dp52_write(dp52, DP52_AUX_ADCFG, mux & 0x1f);
	dp52_set_bits(dp52, DP52_AUX_ADCTRL, DP52_AUXADCTRL_SAMPLE);

	/*
	 * Wait for the measurement. The hardware may still be busy sometimes
	 * so wait for it if necessary...
	 */
	msleep(AUX_READ_DELAY_TIME_MS);
	while (dp52_read(dp52, DP52_AUX_ADCTRL) & 0x1)
		msleep(1);

	ret = dp52_read(dp52, DP52_AUX_ADDATA) & 0x7FFF;

	/*
	 * Reset auxadc (prevent touchscreen stuck problem)
	 */
	dp52_set_bits(dp52, DP52_CMU_CCR1, 1<<14);
	msleep(10);
	dp52_clr_bits(dp52, DP52_CMU_CCR1, 1<<14);

	if (automode)
		dp52_set_bits(dp52, DP52_TP_CTL2, DP52_TP_CTL2_AUTOMODE);

	/* FIXME: Delay until completion of previous measurement. Probably HW
	 * bug, will be simulated */
	msleep(AUX_READ_DELAY_TIME_MS);

	return ret;
}

int dp52_auxadc_measure(struct dp52 *dp52, int dcin)
{
	int mux, ret;

	switch (dcin) {
		case 0: mux = DP52_AUX_MUX_DCIN0_16V; break;
		case 1: mux = DP52_AUX_MUX_DCIN1_GAIN; break;
		case 2: mux = DP52_AUX_MUX_DCIN2_16V; break;
		case 3: mux = DP52_AUX_MUX_DCIN3_16V; break;

		default:
			return -EINVAL;
	}

	dp52_aux_mutex_lock(dp52);

	/* do the measurement */
	ret = dp52_aux_read(dp52, mux);

	/* for dcin1 (charing voltage), the values are very small (~30mV) */
	/* so meausing it as-is gives incorrect values (since A2D HW is not acuurate at extreme values) */
	/* thus - we are measuring the voltage AFTER the gain (about 1v), and then we divide by the gain */
	if (mux == DP52_AUX_MUX_DCIN1_GAIN) {
		int gain_index;

		gain_index = dp52_read(dp52, DP52_AUX_DC1GAINCNT1) & 0x1f;
		ret = ret * 4 / gain_steps[gain_index];
	}

	dp52_aux_mutex_unlock(dp52);

	return SAMPLE_TO_VOLTAGE(ret);
}
EXPORT_SYMBOL(dp52_auxadc_measure);

int dp52_auxcmp_calculate(struct dp52 *dp52, int vref, int round, int *gain, int *attn)
{
	/*
	 * Find gain/attenuation setting which converts Vref to 1V BG. First
	 * find the gain which drives the voltage above 1V, then adjust the
	 * attenuation to match 1V exactly.
	 */
	*gain = 1;
	*attn = 63;
	while (vref*gain_steps[*gain]/4 < 1000 && *gain < ARRAY_SIZE(gain_steps))
		(*gain)++;

	if (*gain >= ARRAY_SIZE(gain_steps)) {
		/* Still below 1V. Actual DCIN trigger is therefore above
		 * requested voltage. */
		if (round <= 0)
			return -EINVAL;
		(*gain)--;
	} else {
		while (vref*gain_steps[*gain]*(193+*attn)/(4*256) > 1000 && *attn >= 0)
			(*attn)--;

		if (*attn < 0) {
			/* Still above 1V. Actual DCIN trigger is therefore
			 * below requested voltage. */
			if (round < 0) {
				/* round down */
				*attn = 0;
			} else if (round > 0) {
				/* round up */
				if (*gain == 1)
					return -EINVAL;
				*attn = 63;
				(*gain)--;
			} else
				return -EINVAL;
		}
	}
	return 0;
}
EXPORT_SYMBOL(dp52_auxcmp_calculate);

int dp52_auxcmp_get_calibrated_values(struct dp52 *dp52, int dcin, int *gain, int *attn)
{
	if (dcin == 3) {
		*gain = dp52->dcin3_gain;
		*attn = dp52->dcin3_attn;
		return 0;
	}

	/* error */
	return -1;
}
EXPORT_SYMBOL(dp52_auxcmp_get_calibrated_values);

int dp52_auxcmp_enable(struct dp52 *dp52, int dcin, int gain, int attn)
{
	dp52_aux_mutex_lock(dp52);
	dp52_write(dp52, DP52_AUX_DC0GAINCNT1+dcin, gain | (attn << 5));
	dp52_set_bits(dp52, DP52_AUX_EN, 1 << (4+dcin));
	dp52_aux_mutex_unlock(dp52);

	/* return actual trigger voltage */
	return 1000 * 256 * 4 / (193+attn) / gain_steps[gain];
}
EXPORT_SYMBOL(dp52_auxcmp_enable);

int dp52_auxcmp_disable(struct dp52 *dp52, int dcin)
{
	return dp52_clr_bits(dp52, DP52_AUX_EN, 1 << (4+dcin));
}
EXPORT_SYMBOL(dp52_auxcmp_disable);

int dp52_auxcmp_get(struct dp52 *dp52, int dcin)
{
	return !!(dp52_read(dp52, DP52_AUX_RAWPMINTSTAT) & (1 << dcin));
}
EXPORT_SYMBOL(dp52_auxcmp_get);

/******************************************************************************************/

static void dp52_irq_lock(struct irq_data *d)
{
	struct dp52 *dp52 = d->chip_data;

	mutex_lock(&dp52->irq_lock);
}

static void dp52_irq_sync_unlock(struct irq_data *d)
{
	struct dp52 *dp52 = d->chip_data;

	if (dp52->irq_enable_cur != dp52->irq_enable_cache) {
		dp52_write(dp52, DP52_ICU_MASK, dp52->irq_enable_cur);
		dp52->irq_enable_cache = dp52->irq_enable_cur;
	}

	mutex_unlock(&dp52->irq_lock);
}

static void dp52_irq_mask(struct irq_data *d)
{
	struct dp52 *dp52 = d->chip_data;

	dp52->irq_enable_cur &= ~(1 << (d->irq - dp52->irq_base));
}

static void dp52_irq_unmask(struct irq_data *d)
{
	struct dp52 *dp52 = d->chip_data;

	dp52->irq_enable_cur |= 1 << (d->irq - dp52->irq_base);
}

static int dp52_set_wake(struct irq_data *d, unsigned int on)
{
	struct dp52 *dp52 = d->chip_data;

	if (on)
		dp52->irq_enable_wake |= 1 << (d->irq - dp52->irq_base);
	else
		dp52->irq_enable_wake &= ~(1 << (d->irq - dp52->irq_base));

	return 0;
}

static struct irq_chip dp52_irq_chip = {
	.name = "dp52",
	.irq_bus_lock = dp52_irq_lock,
	.irq_bus_sync_unlock = dp52_irq_sync_unlock,
	.irq_mask = dp52_irq_mask,
	.irq_unmask = dp52_irq_unmask,
	.irq_set_wake = dp52_set_wake,
};

static irqreturn_t dp52_irq_thread(int irq, void *priv)
{
	struct dp52 *dp52 = priv;
	int i, pending, irq_base;

	irq_base = dp52->irq_base;
	pending = dp52_read(dp52, DP52_ICU_STAT);
	if (!pending)
		return IRQ_NONE;

	for (i=0; pending != 0; i++, pending >>= 1) {
		if (!(pending & 1))
			continue;

		/* Check the mask again. Another handler/thread might have
		 * changed it in between. */
		if (dp52->irq_enable_cur & (1 << i))
			handle_nested_irq(irq_base + i);
	}

	return IRQ_HANDLED;
}

static int __devinit dp52_parse_calibration(struct dp52 *dp52)
{
	char *pos;

	if (!strlen(calibration))
		calibration = DEFAULT_CALIBRATION;

	dp52->aux_bgp = simple_strtoul(calibration, &pos, 10);
	if (dp52->aux_bgp <= 0 || dp52->aux_bgp > 255)
		return -EINVAL;

	if (*pos != ':')
		return -EINVAL;

	dp52->dcin3_gain = simple_strtoul(pos+1, &pos, 10);
	if (dp52->dcin3_gain < 0 || dp52->dcin3_gain >= ARRAY_SIZE(gain_steps))
		return -EINVAL;

	if (*pos != ':')
		return -EINVAL;

	dp52->dcin3_attn = simple_strtoul(pos+1, &pos, 10);
	if (dp52->dcin3_attn < 0 || dp52->dcin3_attn > 63)
		return -EINVAL;

	if (*pos != 0)
		return -EINVAL;

	return 0;
}

static int __devinit dp52_probe(struct spi_device *spi)
{
	struct dp52_platform_data *pdata = spi->dev.platform_data;
	struct dp52 *dp52;
	int i;
	int ret = -ENOMEM;

	if (!pdata) {
		dev_err(&spi->dev, "no platform data?\n");
		return -EINVAL;
	}

	spi->mode = SPI_MODE_1; /* clk active high */
	spi->bits_per_word = 8;

	ret = spi_setup(spi);
	if (ret < 0)
		return ret;

	dp52 = kzalloc(sizeof(*dp52), GFP_KERNEL);
	if (!dp52)
		return -ENOMEM;

	dev_set_drvdata(&spi->dev, dp52);
	dp52->spi = spi;
	dp52->irq = pdata->irq;
	dp52->irq_base = pdata->irq_base;

	mutex_init(&dp52->irq_lock);
	mutex_init(&dp52->mutex_auxilary);
	mutex_init(&dp52->mutex_indirect);
	mutex_init(&dp52->mutex_rmw);

	/*
	 * Parse calibration data
	 */
	if ((ret = dp52_parse_calibration(dp52))) {
		dev_err(&spi->dev, "invalid calibration data!\n");
		goto err_free;
	}

	/*
	 * Setup IRQ's
	 */
	dp52_write(dp52, DP52_ICU_MASK, 0x0000);
	dp52_write(dp52, DP52_ICU_STAT, 0x0000);
	dp52_write(dp52, DP52_AUX_INTMSKN, 0x0000);
	dp52_write(dp52, DP52_AUX_INTSTAT, 0x0000);
	dp52_write(dp52, DP52_PMU_INTMSK, 0x0000);
	dp52_write(dp52, DP52_PMU_STAT, 0x0000);

	for (i = pdata->irq_base; i <pdata->irq_base + DP52_NUM_IRQS; i++) {
		irq_set_chip_data(i, dp52);
		irq_set_chip_and_handler(i, &dp52_irq_chip, handle_level_irq);
		irq_set_nested_thread(i, 1);
		set_irq_flags(i, IRQF_VALID);
	}

	ret = request_threaded_irq(pdata->irq, NULL, dp52_irq_thread,
	                           IRQF_TRIGGER_HIGH | IRQF_ONESHOT, "dp52", dp52);
	if (ret) {
		dev_err(&spi->dev, "Failed to request IRQ %d: %d\n", pdata->irq, ret);
		goto err_free;
	}

	/*
	 * Permanently enable the 4MHz oscillator (at 4MHz) and use it for PWM0
	 * and the AUX ADC (divided by 4).
	 *
	 * FIXME: enable on demand to save power!
	 */
	dp52_update(dp52, DP52_CMU_CCR2, (1 << 7) | (1 << 6) | 3,
		(1 << 7) | (1 << 6) | 1);
	msleep(1);
	dp52_update(dp52, DP52_CMU_CCR0, (3 << 10) | (7 << 5), (2 << 10) | (3 << 5));

	/* Set AUX ADC bandgap, configure sample window to 15 bit */
	dp52_update(dp52, DP52_AUX_ADCTRL, 0x0f << 1, 7 << 1);
	dp52_update(dp52, DP52_AUX_ADCTRL, 0xff << 8, dp52->aux_bgp << 8);

	/* Enable AUX ADC and Bandgap */
	dp52_set_bits(dp52, DP52_AUX_EN, 0x03);

	/* Initialize regulators */
	if ((ret = dp52_pmu_probe(&spi->dev, pdata))) {
		dev_err(&spi->dev, "PMU init failed");
		goto err_free_irq;
	}

	/* Register child devices */
	ret = mfd_add_devices(&spi->dev, -1, pdata->cells, pdata->num_cells,
	                      NULL, pdata->irq_base);
	if (ret) {
		dev_err(&spi->dev, "failed to add children: %d\n", ret);
		goto err_pmu_rem;
	}

	ret = device_create_file(&spi->dev, &dev_attr_registers);
	if (ret < 0)
		dev_warn(&spi->dev, "failed to add sysfs regs file\n");

	dp52_debugfs_init(dp52);

	dev_info(&spi->dev, "core driver initialized\n");
	return 0;

err_pmu_rem:
	dp52_pmu_remove(&spi->dev);
err_free_irq:
	free_irq(pdata->irq, dp52);
err_free:
	kfree(dp52);
	return ret;
}

static int __devexit dp52_remove(struct spi_device *spi)
{
	struct device *dev = &spi->dev;
	struct dp52 *dp52 = dev_get_drvdata(dev);

	dp52_debugfs_exit(dp52);
	mfd_remove_devices(dev);
	dp52_pmu_remove(dev);
	free_irq(dp52->irq, dp52);
	kfree(dp52);

	return 0;
}

#ifdef CONFIG_PM
static int dp52_suspend(struct spi_device *spi, pm_message_t mesg)
{
	struct dp52 *dp52 = dev_get_drvdata(&spi->dev);

	/* Only leave interrupts enabled which are marked as wakeup */
	dp52_write(dp52, DP52_ICU_MASK,
	           dp52->irq_enable_wake & dp52->irq_enable_cur);

	if (dp52->irq_enable_wake)
		irq_set_irq_wake(dp52->irq, 1);

	return 0;
}

static int dp52_resume(struct spi_device *spi)
{
	struct dp52 *dp52 = dev_get_drvdata(&spi->dev);

	if (dp52->irq_enable_wake)
		irq_set_irq_wake(dp52->irq, 0);

	/* Restore interrupt mask */
	dp52_write(dp52, DP52_ICU_MASK, dp52->irq_enable_cur);

	return 0;
}
#else
#define dp52_suspend NULL
#define dp52_resume NULL
#endif

static struct spi_driver dp52_driver = {
	.driver = {
		.name  = "dp52",
		.bus   = &spi_bus_type,
		.owner = THIS_MODULE,
	},
	.probe	 = dp52_probe,
	.remove  = __devexit_p(dp52_remove),
	.suspend = dp52_suspend,
	.resume  = dp52_resume,
};

static int __init dp52_init(void)
{
	return spi_register_driver(&dp52_driver);
}

static void __exit dp52_exit(void)
{
	spi_unregister_driver(&dp52_driver);
}

module_init(dp52_init);
module_exit(dp52_exit);

MODULE_AUTHOR("DSPG Technologies GmbH");
MODULE_DESCRIPTION("DP52 core driver");
MODULE_LICENSE("GPL");

