/*
 * drivers/gpio/dmw-gpio.c
 *
 *  Copyright (c) 2012 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <asm-generic/gpio.h>
#include <linux/dmw-gpio.h>
#include <linux/errno.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include <linux/types.h>

struct dmw_gpio {
	struct gpio_chip chip;
	void *regs;
	u32 *nx_masks;
	struct resource *mem_res;
};

#define GET_REG(priv, reg, gpio) ( \
	__raw_readl((priv)->regs + (__BANK(gpio) * 0x60) + (reg)) & (1 << __PIN(gpio)) \
	)

#define SET_REG(priv, reg, gpio) \
	__raw_writel(1 << __PIN(gpio), (priv)->regs + (__BANK(gpio) * 0x60) + (reg))


static int gpio_get_pull_enable(struct dmw_gpio *dmw_gpio, unsigned gpio)
{
	return !!GET_REG(dmw_gpio, XGP_PULL_CTRL, gpio);
}

/*
 * Returns: 0: Output, 1: Input
 */
static int gpio_get_direction(struct dmw_gpio *dmw_gpio, unsigned gpio)
{
	return !!GET_REG(dmw_gpio, XGP_DIR, gpio);
}

static void dmw_gpio_set_value(struct dmw_gpio *dmw_gpio, unsigned gpio, int value)
{
	if (value)
		SET_REG(dmw_gpio, XGP_DATA_SET, gpio);
	else
		SET_REG(dmw_gpio, XGP_DATA_CLR, gpio);
}

static int gpio_get_enable(struct dmw_gpio *dmw_gpio, unsigned gpio)
{
	return !!GET_REG(dmw_gpio, XGP_EN, gpio);
}

static int gpio_get_opendrain(struct dmw_gpio *dmw_gpio, unsigned gpio)
{
	return !!GET_REG(dmw_gpio, XGP_OD, gpio);
}

static int gpio_get_keeper(struct dmw_gpio *dmw_gpio, unsigned gpio)
{
	return !!GET_REG(dmw_gpio, XGP_KEEP, gpio);
}

/*
 * Returns: 0: pull down, 1: pull up
 */
static int gpio_get_pull_selection(struct dmw_gpio *dmw_gpio, unsigned gpio)
{
	return !!GET_REG(dmw_gpio, XGP_PULL_TYPE, gpio);
}

/*
 * Tests whether the Pull-Up/Dn internal resistor can be disconnected, in order
 * to save power.
 */
static void tst_pull_disconnect(struct dmw_gpio *dmw_gpio, unsigned gpio)
{
	if (!gpio_get_pull_enable(dmw_gpio, gpio))
		return;

	if (gpio_get_direction(dmw_gpio, gpio) == 0) {
		/* Output gpio */

		/* Note:
		 * "Wired-OR" is achived by enabling "Open-Drain" and
		 * No Pull-Down resistor, but the user may connect
		 * a Pull-Up resistor, if no such External resistor
		 * exists. So "open-drain" must be excluded !
		 */
		if (!gpio_get_opendrain(dmw_gpio, gpio))
			SET_REG(dmw_gpio, XGP_PULL_DIS, gpio);
	} else {
		/* Input gpio */

		if (gpio_get_keeper(dmw_gpio, gpio))
			SET_REG(dmw_gpio, XGP_PULL_DIS, gpio);
	}
}

static int
dmw_gpiolib_request(struct gpio_chip *chip, unsigned offset)
{
	struct dmw_gpio *dmw_gpio = container_of(chip, struct dmw_gpio, chip);

	if (dmw_gpio->nx_masks[__BANK(offset)] & (1 << __PIN(offset)))
		return -ENODEV;
	if (!gpio_get_enable(dmw_gpio, offset))
		return -ENODEV;

	return 0;
}

static int
dmw_gpiolib_direction_input(struct gpio_chip *chip, unsigned offset)
{
	struct dmw_gpio *dmw_gpio = container_of(chip, struct dmw_gpio, chip);

	SET_REG(dmw_gpio, XGP_DIR_IN, offset);
	tst_pull_disconnect(dmw_gpio, offset);

	return 0;
}

static int
dmw_gpiolib_direction_output(struct gpio_chip *chip, unsigned offset, int value)
{
	struct dmw_gpio *dmw_gpio = container_of(chip, struct dmw_gpio, chip);

	dmw_gpio_set_value(dmw_gpio, offset, value);
	SET_REG(dmw_gpio, XGP_DIR_OUT, offset);
	tst_pull_disconnect(dmw_gpio, offset);

	return 0;
}

static int
dmw_gpiolib_get(struct gpio_chip *chip, unsigned offset)
{
	struct dmw_gpio *dmw_gpio = container_of(chip, struct dmw_gpio, chip);

	return !!GET_REG(dmw_gpio, XGP_DATA, offset);
}

static void
dmw_gpiolib_set(struct gpio_chip *chip, unsigned offset, int value)
{
	struct dmw_gpio *dmw_gpio = container_of(chip, struct dmw_gpio, chip);

	dmw_gpio_set_value(dmw_gpio, offset, value);
}

static void
dmw_gpiolib_show(struct seq_file *s, struct gpio_chip *chip)
{
	struct dmw_gpio *dmw_gpio = container_of(chip, struct dmw_gpio, chip);
	const char *label;
	unsigned i;

	for (i = 0; i < chip->ngpio; i++) {
		if (!(label = gpiochip_is_requested(chip, i)))
			continue;

		seq_printf(s, " %cGPIO%-2d (%-20.20s) %s %s%s",
			'A' + __BANK(i), __PIN(i), label,
			gpio_get_direction(dmw_gpio, i) ? "in " : "out",
			dmw_gpiolib_get(chip, i) ? "hi" : "lo",
			gpio_get_pull_enable(dmw_gpio, i)
				? (gpio_get_pull_selection(dmw_gpio, i) ? " pu" : " pd")
				: ""
			);
		seq_printf(s, "\n");
	}
}

static int __init dmw_gpio_probe(struct platform_device *pdev)
{
	struct dmw_gpio_pdata *pdata = pdev->dev.platform_data;
	struct dmw_gpio *dmw_gpio;
	int len, ret = 0;

	if (!pdata) {
		dev_err(&pdev->dev, "missing platform data!\n");
		return -EINVAL;
	}

	dmw_gpio = kzalloc(sizeof(*dmw_gpio), GFP_KERNEL);
	if (!dmw_gpio)
		return -ENOMEM;

	dmw_gpio->nx_masks = pdata->nx_masks;
	dmw_gpio->mem_res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!dmw_gpio->mem_res) {
		ret = -EINVAL;
		goto err_free;
	}

	len = resource_size(dmw_gpio->mem_res);
	if (!request_mem_region(dmw_gpio->mem_res->start, len, "dmw-tdm")) {
		ret = -EBUSY;
		goto err_free;
	}

	dmw_gpio->regs = ioremap_nocache(dmw_gpio->mem_res->start, len);
	if (!dmw_gpio->regs) {
		ret = -ENOMEM;
		goto err_release;
	}

	dmw_gpio->chip.label		= "dmw-gpio";
	dmw_gpio->chip.owner		= THIS_MODULE;
	dmw_gpio->chip.request		= dmw_gpiolib_request;
	dmw_gpio->chip.direction_input	= dmw_gpiolib_direction_input;
	dmw_gpio->chip.get		= dmw_gpiolib_get;
	dmw_gpio->chip.direction_output	= dmw_gpiolib_direction_output;
	dmw_gpio->chip.set		= dmw_gpiolib_set;
	dmw_gpio->chip.dbg_show		= dmw_gpiolib_show;
	dmw_gpio->chip.ngpio		= pdata->num_banks * 32;

	ret = gpiochip_add(&dmw_gpio->chip);
	if (ret)
		goto err_unmap;

	dev_info(&pdev->dev, "%d banks\n", pdata->num_banks);

	platform_set_drvdata(pdev, dmw_gpio);
	return 0;

err_unmap:
	iounmap(dmw_gpio->regs);
err_release:
	release_mem_region(dmw_gpio->mem_res->start, len);
err_free:
	kfree(dmw_gpio);
	return ret;
}

static int __devexit dmw_gpio_remove(struct platform_device *pdev)
{
	struct dmw_gpio *dmw_gpio = platform_get_drvdata(pdev);
	int ret;

	ret = gpiochip_remove(&dmw_gpio->chip);
	if (ret)
		return ret;

	iounmap(dmw_gpio->regs);
	release_mem_region(dmw_gpio->mem_res->start,
			   resource_size(dmw_gpio->mem_res));
	kfree(dmw_gpio);

	return 0;
}

static struct platform_driver dmw_gpio_driver = {
	.driver		= {
		.name	= "dmw-gpio",
		.owner	= THIS_MODULE,
	},
	.remove		= __devexit_p(dmw_gpio_remove),
};

static int __init dmw_gpio_init(void)
{
	return platform_driver_probe(&dmw_gpio_driver, dmw_gpio_probe);
}
subsys_initcall(dmw_gpio_init);
