/*
 *  arch/arm/mach-dmw/powerdomain.c
 *
 *  Copyright (C) 2011 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/init.h>
#include <linux/io.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/pm_runtime.h>

#include <mach/hardware.h>
#include <mach/platform.h>
#include <mach/powerdomain.h>

static int dmw_pd_probe(struct platform_device *pdev)
{
	struct dmw_pwrdom_pdata *pdata = pdev->dev.platform_data;
	struct device *dev = &pdev->dev;

	if (!pdata) {
		dev_err(dev, "no device data specified\n");
		return -ENOENT;
	}

	dmw_pwrdom_enable(pdata->domain);
	pm_runtime_set_active(dev);
	pm_runtime_enable(dev);

	dev_info(dev, "domain '%s' registered\n", pdata->name);

	return 0;
}

static int __devexit dmw_pd_remove(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct dmw_pwrdom_pdata *pdata = dev->platform_data;

	pm_runtime_disable(dev);
	dmw_pwrdom_disable(pdata->domain);

	return 0;
}

static int dmw_pd_runtime_suspend(struct device *dev)
{
	struct dmw_pwrdom_pdata *pdata = dev->platform_data;

	dmw_pwrdom_disable(pdata->domain);

	dev_dbg(dev, "domain '%s' disabled\n", pdata->name);

	return 0;
}

static int dmw_pd_runtime_resume(struct device *dev)
{
	struct dmw_pwrdom_pdata *pdata = dev->platform_data;

	dmw_pwrdom_enable(pdata->domain);

	dev_dbg(dev, "domain '%s' enabled\n", pdata->name);

	return 0;
}

static const struct dev_pm_ops dmw_pd_pm_ops = {
	.runtime_suspend	= dmw_pd_runtime_suspend,
	.runtime_resume		= dmw_pd_runtime_resume,
};

static struct platform_driver dmw_pd_driver = {
	.driver		= {
		.name		= "dmw-powerdomain",
		.owner		= THIS_MODULE,
		.pm		= &dmw_pd_pm_ops,
	},
	.probe		= dmw_pd_probe,
	.remove		= __devexit_p(dmw_pd_remove),
};


static int __init dmw_powerdomain_init(void)
{
	int ret;

	ret = platform_driver_register(&dmw_pd_driver);
	if (ret)
		printk(KERN_ERR "%s: failed to add PD driver\n", __func__);

	return ret;
}

arch_initcall(dmw_powerdomain_init);
