/*
 * arch/arm/mach-dmw/css/coma-alsa.c - cordless TDM kernel interface
 *
 * Using interface to Linux kernel is implemented that
 * allows sending ALSA PCM messages
 *
 * Copyright (C) 2007 NXP Semiconductors
 * Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#define pr_fmt(fmt) KBUILD_MODNAME ": " fmt

#include <linux/kthread.h>
#include "coma.h"
#include "coma-service.h"
#include <mach/coma-alsa.h>
#include <linux/delay.h>


#define SERVICE_NAME "coma-alsa"
#define SERVICE_ID   (COMA_SERVICE_ALSA) 
#define CFIFO_SIZE   1024

#define COMA_ALSA_TIMEOUT (1 * HZ) /* 1 sec. */



enum alsa_pcm_service_protocol_errors {
	ALSA_PCM_SERVICE_NO_ERROR      =  0,
	ALSA_PCM_SERVICE_INVALID_PARAM = -1,
	ALSA_PCM_SERVICE_NO_MEMORY     = -2,
	ALSA_PCM_SERVICE_DEVICE_BUSY   = -3,
	ALSA_PCM_SERVICE_DEVICE_ERROR  = -4,
	ALSA_PCM_SERVICE_LAST          = -5,
};

#define ALSA_PCM_FLAG_RX               (1 << 0)
#define ALSA_PCM_FLAG_TX               (1 << 1)

struct alsa_msg_open {
	uint8_t type;
	uint8_t id;              /* id of the ALSA PCM device */
	uint32_t cookie;         /* this cookie will be used for responses */
	uint32_t rate;           /* rate in HZ */
	uint32_t sample_size;    /* size of a sample in bits */
	uint32_t period_size;    /* number of frames in a period, 0 if no elapsed message required */
	uint32_t channels;       /* all channels are one frame */
	void    *rx_buffer_addr; /* physical address of rx sample buffer */
	uint32_t rx_buffer_size; /*
	                          * size of rx buffer in bytes (should be a multiple of
	                          * channels * sample_size (sample_size in bytes))
	                          * including the 4 bytes for the progress index
	                          */
	void    *tx_buffer_addr; /* physical address of tx sample buffer */
	uint32_t tx_buffer_size; /*
	                          * size of tx buffer in bytes (should be a multiple of
	                          * channels * sample_size (sample_size in bytes))
	                          * including the 4 bytes for the progress index
	                          */
} __attribute__ ((__packed__));

struct alsa_any_msg {
	uint8_t type;
};

struct alsa_msg_start {
	uint8_t type;
	uint8_t id;
	uint16_t flags;
} __attribute__ ((__packed__));

struct alsa_msg_stop {
	uint8_t type;
	uint8_t id;
	uint16_t flags;
} __attribute__ ((__packed__));

struct alsa_msg_puase {
	uint8_t type;
	uint8_t id;
} __attribute__ ((__packed__));

struct alsa_msg_reset {
	uint8_t type;
	uint8_t id;
	uint16_t flags;
	uint32_t cookie;
} __attribute__ ((__packed__));

struct alsa_msg_close {
	uint8_t type;
	uint8_t id;
	uint32_t cookie;
} __attribute__ ((__packed__));

struct alsa_elapsed {
	uint8_t type;
	uint8_t id;
} __attribute__ ((__packed__));


struct alsa_msg_ack {
	uint8_t type;
	uint32_t cookie;
} __attribute__ ((__packed__));


struct alsa_msg_nack {
	uint8_t type;
	uint32_t cookie;
	int32_t  reason;
}__attribute__ ((__packed__));


enum   alsa_pcm_messge_status {
        WAIT_FOR_ANSWER = 0 ,
        ANSWERED=1,
} __attribute__ ((__packed__));

struct alsa_reply {
        enum  alsa_pcm_messge_status  done;
        union msg_res {
                struct alsa_msg_nack nack;
                struct alsa_msg_ack  ack;
                struct alsa_elapsed  elapsed;
        } msg ;
} __attribute__ ((__packed__));

enum alsa_service_message_types {
	ALSA_PCM_OPEN           = 0,
	ALSA_PCM_CLOSE          = 1,
	ALSA_PCM_START          = 2,
	ALSA_PCM_STOP           = 3,
	ALSA_PCM_RESET          = 4,
	ALSA_PCM_ELAPSED        = 5,
	ALSA_PCM_QUERY          = 6,
	ALSA_PCM_QUERY_RESPONSE = 7,
	ALSA_PCM_ACK            = 8,
	ALSA_PCM_NACK           = 9,
};

struct elapsed_service_t {
        unsigned int id;
        enum coma_alsa_restart (* function)(unsigned long int);
        unsigned long int data; 
};

static DECLARE_WAIT_QUEUE_HEAD(coma_alsa_reply);
static DEFINE_MUTEX(coma_alsa_mutex);

static int initialized = 0;
static int cfifo_size;
static struct cfifo *l2c, *c2l;
static unsigned int mem_handle;

static struct elapsed_service_t elapsed_service[3] ;


int coma_alsa_register_elapsed_cb(unsigned int id ,enum coma_alsa_restart (* function)(unsigned long int) ,unsigned long int data)
{
 	pr_debug("coma_alsa_register_elapsed_cb id=%d",id);
        
        elapsed_service[id].id = id;
        elapsed_service[id].function = function;
        elapsed_service[id].data = data;
        return 0;
}

int coma_alsa_deregister_elapsed_cb(unsigned int id )
{
        elapsed_service[id].id = 0;
        elapsed_service[id].function = NULL;
        elapsed_service[id].data = 0;
        return 0;
}




static int coma_alsa_send_synced(void *buf, size_t size,enum  alsa_pcm_messge_status *done)
{
	int ret;

	mutex_lock(&coma_alsa_mutex);

	ret = coma_create_message(SERVICE_ID, SERVICE_ID, NULL, 0,
	                          (unsigned char *)buf, size);
	if (ret < 0) {
		pr_err("%s: failed sending message (type=%u)\n",
		       SERVICE_NAME, *(uint32_t *)buf);
		mutex_unlock(&coma_alsa_mutex);
		return ret;
	}
	coma_signal(SERVICE_ID);
	mutex_unlock(&coma_alsa_mutex);

	ret = wait_event_timeout(coma_alsa_reply,*done != WAIT_FOR_ANSWER, COMA_ALSA_TIMEOUT);
	if (ret == 0){
	
		pr_err("%s: ack expected, but timed out\n", SERVICE_NAME);
		ret = -ETIMEDOUT ;
	} else
		ret = 0;

	return ret;
}

static int coma_alsa_send_unsynced(void *buf, size_t size)
{
	int ret = 0;

	mutex_lock(&coma_alsa_mutex);

	ret = coma_create_message(SERVICE_ID, SERVICE_ID, NULL, 0,
	                          (unsigned char *)buf, size);
	if (ret < 0) {
		pr_err("%s: failed sending message (type=%u)\n",
		       SERVICE_NAME, *(uint32_t *)buf);
		mutex_unlock(&coma_alsa_mutex);
		return ret;

	}
	coma_signal(SERVICE_ID);

	mutex_unlock(&coma_alsa_mutex);
	return ret;
}


int coma_alsa_open(uint8_t id ,
                        uint32_t rate,
	                uint32_t sample_size,    /* size of a sample in bits */
	                uint32_t period_size,    /* number of frames in a period, 0 if no elapsed message required */
	                uint32_t channels,       /* all channels are one frame */
	                void    *rx_buffer_addr, /* physical address of rx sample buffer */
	                uint32_t rx_buffer_size, /*
	                                          * size of rx buffer in bytes (should be a multiple of
	                                          * channels * sample_size (sample_size in bytes))
	                                          * including the 4 bytes for the progress index
	                                          */
	                void    *tx_buffer_addr,  /* physical address of tx sample buffer */
	                uint32_t tx_buffer_size   /*
	                                          * size of tx buffer in bytes (should be a multiple of
	                                          * channels * sample_size (sample_size in bytes))
	                                          * including the 4 bytes for the progress index
	                                          */
	                )
{
	int ret;
	struct alsa_msg_open msg;
	struct alsa_reply reply;

	if (!initialized) {
		pr_err("%s: open but not initialized\n", SERVICE_NAME);
		return -EFAULT;
	}

	msg.type = ALSA_PCM_OPEN;
	msg.id =  id;              /* id of the ALSA PCM device */
	msg.rate= rate;                   /* rate in HZ */
	msg.sample_size = sample_size ;     /* size of a sample in bits */
	msg.period_size = period_size ;    /* number of frames in a period, 0 if no elapsed message required */
	msg.channels = channels;       /* all channels are one frame */
	msg.rx_buffer_addr = rx_buffer_addr ; /* physical address of rx sample buffer */
	msg.rx_buffer_size = rx_buffer_size ; /*
	                          * size of rx buffer in bytes (should be a multiple of
	                          * channels * sample_size (sample_size in bytes))
	                          * including the 4 bytes for the progress index
	                          */
	msg.tx_buffer_addr = tx_buffer_addr ; /* physical address of tx sample buffer */
	msg.tx_buffer_size = tx_buffer_size ; /*
	                          * size of tx buffer in bytes (should be a multiple of
	                          * channels * sample_size (sample_size in bytes))
	                          * including the 4 bytes for the progress index
	                          */


        pr_debug("msg open  id=%d rate=%d sample_size=%d periodsize=%d channels=%d rx_buf=%p rx_buf_size=%d tx_buf=%p,tx_buf_size=%d\n",
                 			 msg.id,msg.rate,msg.sample_size,msg.period_size,msg.channels,msg.rx_buffer_addr,msg.rx_buffer_size,msg.tx_buffer_addr,msg.tx_buffer_size );
        	
	/* clear the reply message */
	memset((void *)&reply, 0, sizeof(reply));
        reply.done = WAIT_FOR_ANSWER;
	
	msg.cookie = (uint32_t) &reply;

	pr_debug("%s: openning alsa %u cookie = %p \n", SERVICE_NAME, id,(void *)msg.cookie);

	ret = coma_alsa_send_synced(&msg, sizeof(msg),&(reply.done));
 	if (ret < 0) {
		pr_err("%s: open alsa %u failed on timeout(%d)\n",
                      SERVICE_NAME, id, ret);
	        return ret;
		       
	}
	
	BUG_ON(msg.cookie != reply.msg.ack.cookie);
	
	if (reply.msg.nack.reason != ALSA_PCM_SERVICE_NO_ERROR ){
	        ret = reply.msg.nack.reason;
		pr_err("%s: open alsa %u failed on (%d)\n",
		       SERVICE_NAME, id, ret);

	}
 	       
	return ret;
}
EXPORT_SYMBOL(coma_alsa_open);

int coma_alsa_close(unsigned int id)
{
	int ret;
	struct alsa_msg_close msg;
	struct alsa_reply reply;
        /* just delay to ensure that previos messages allready handled by CSS*/
        udelay(1000);

	if (!initialized) {
		pr_err("%s: closing but not initialized\n", SERVICE_NAME);
		return -EFAULT;
	}

	msg.type = ALSA_PCM_CLOSE;
	msg.id = id;

	pr_debug("%s: closing alsa %u\n", SERVICE_NAME, id);
	/* clear the reply message */
	memset((void *)&reply, 0, sizeof(reply));
        reply.done = WAIT_FOR_ANSWER;
	
	msg.cookie = (uint32_t) &reply;

	pr_debug("%s: closing alsa %u\n", SERVICE_NAME, id);

	ret = coma_alsa_send_synced(&msg, sizeof(msg),&(reply.done));
 	if (ret < 0) {
		pr_err("%s: close alsa %u failed on timeout(%d)\n",
                      SERVICE_NAME, id, ret);
	        return ret;
		       
	}
	
	BUG_ON(msg.cookie != reply.msg.ack.cookie);
	
	if (reply.msg.nack.reason != ALSA_PCM_SERVICE_NO_ERROR ){
	        ret = reply.msg.nack.reason;
		pr_err("%s: close alsa %u failed on (%d)\n",
		       SERVICE_NAME, id, ret);
		if (ret == -3)
		{
		 ret = ALSA_PCM_SERVICE_NO_ERROR;
			pr_err("%s: close alsa %u failed on (%d)\n",
				SERVICE_NAME, id, ret);
		}       

	}
 	       
	return ret;
}
EXPORT_SYMBOL(coma_alsa_close);


int coma_alsa_start(unsigned int id, uint16_t flags)
{
	int ret;
	struct alsa_msg_start msg;

	if (!initialized) {
		pr_err("%s: starting but not initialized\n", SERVICE_NAME);
		return -EFAULT;
	} 

	msg.type = ALSA_PCM_START;
	msg.id = id;
	msg.flags = 0;
        if ( flags == 0)
                msg.flags |= ALSA_PCM_FLAG_TX ;
        else 
               msg.flags |= ALSA_PCM_FLAG_RX ;

	pr_debug("%s: starting alsa id= %u direction =%d \n", SERVICE_NAME, id,msg.flags);

	ret = coma_alsa_send_unsynced(&msg, sizeof(msg));
	if (ret < 0)
		pr_err("%s: start alsa %u failed (%d)\n",
		       SERVICE_NAME, id, ret);
	return ret;
}
EXPORT_SYMBOL(coma_alsa_start);

int coma_alsa_stop(unsigned int id, uint16_t flags)
{
	int ret;
	struct alsa_msg_stop msg;

	if (!initialized) {
		pr_err("%s: stop but not initialized\n", SERVICE_NAME);
		return -EFAULT;
	}

	msg.type = ALSA_PCM_STOP;
	msg.id = id;
	msg.flags = 0;
        if ( flags == 0)
                msg.flags |= ALSA_PCM_FLAG_TX ;
        else 
               msg.flags |= ALSA_PCM_FLAG_RX ;

	pr_debug("%s: stoping alsa id= %u direction =%d \n", SERVICE_NAME, id,msg.flags);

	ret = coma_alsa_send_unsynced(&msg, sizeof(msg));
	if (ret < 0)
		pr_err("%s: stop alsa %u failed (%d)\n",
		       SERVICE_NAME, id, ret);
	return ret;
}
EXPORT_SYMBOL(coma_alsa_stop);

int coma_alsa_reset(unsigned int id,uint16_t flags)
{
	int ret;
	struct alsa_msg_reset msg;
	struct alsa_reply reply;
 
	if (!initialized) {
		pr_err("%s: puase but not initialized\n", SERVICE_NAME);
		return -EFAULT;
	}

	msg.type = ALSA_PCM_RESET;
	msg.id = id;
	msg.flags = 0;
        if ( flags == 0)
               msg.flags |= ALSA_PCM_FLAG_TX ;
        else 
               msg.flags |= ALSA_PCM_FLAG_RX ;

	pr_debug("%s: reseting alsa id= %u direction =%d \n", SERVICE_NAME, id,msg.flags);
	memset((void *)&reply, 0, sizeof(reply));
        reply.done = WAIT_FOR_ANSWER;
	
	msg.cookie = (uint32_t) &reply;

	pr_err("%s: reseting alsa %u\n", SERVICE_NAME, id);

	ret = coma_alsa_send_synced(&msg, sizeof(msg),&(reply.done));
 	if (ret < 0) {
		pr_err("%s: reseting alsa %u failed on timeout(%d)\n",
                      SERVICE_NAME, id, ret);
	        return ret;
		       
	}
	
	BUG_ON(msg.cookie != reply.msg.ack.cookie);
	
	if (reply.msg.nack.reason != ALSA_PCM_SERVICE_NO_ERROR ){
	        ret = reply.msg.nack.reason;
		pr_err("%s: reseting alsa %u failed on (%d)\n",
		       SERVICE_NAME, id, ret);

	}
 	       
	return ret;
}
EXPORT_SYMBOL(coma_alsa_reset);


static int coma_alsa_process_message(struct cmsg *cmsg)
{
        struct alsa_reply   *reply;
	struct alsa_any_msg  *result = cmsg->payload;
 		
	pr_debug("%s: coma_alsa_process_message (%d)\n",
		       SERVICE_NAME, result->type);

  
	BUG_ON(cmsg->type != SERVICE_ID);
	
	switch (result->type)
	{
               case ALSA_PCM_NACK:
               {
	                struct alsa_msg_nack *result_nack = cmsg->payload;
                        pr_debug("coma_alsa_process_message %x\n",result_nack->cookie );
                        
                        reply = (struct alsa_reply *)result_nack->cookie;
	                
 	                reply->msg.nack.reason = result_nack->reason;
 	       }
 	       case ALSA_PCM_ACK:
 	       {
 	       
 	                struct alsa_msg_ack *result_ack = cmsg->payload;
                        
                        reply = (struct alsa_reply *) result_ack->cookie;
        
                        pr_debug("coma_alsa_process_message %x\n",result_ack->cookie );
                 
                        reply->msg.ack.type = result_ack->type;
                        reply->msg.ack.cookie = result_ack->cookie;
                        reply->done = ANSWERED;	                
			wake_up(&coma_alsa_reply);
	       }
	       break;
	       case ALSA_PCM_ELAPSED:
	       {
	                struct alsa_elapsed *elapsed = cmsg->payload;
	                elapsed_service[elapsed->id].function ( elapsed_service[elapsed->id].data);
	       }
	       break;
	       default : 
                        BUG_ON(1);
	       break ;       
	}
	
	return 0;
}

static int coma_alsa_setup(void)
{
	pr_debug("%s()\n", __func__);
	initialized = 1;
	return 0;
}

static void coma_alsa_remove(void)
{
	pr_debug("%s()\n", __func__);
	initialized = 0;
}

void alsa_service_init(void)
{
	int ret;

	if (!l2c) {
                pr_err("%s() allocating  fifos\n", __func__);
	
		cfifo_size =
			cfifo_alloc(CFIFO_SIZE, &l2c, CFIFO_SIZE, &c2l, &mem_handle);
		if (cfifo_size < 0) {
			pr_err("%s: error: cannot alloc cfifos\n", SERVICE_NAME);
			return;
		}
	} else {
                pr_err("%s()   fifos allready allocated clear\n", __func__);
		cfifo_reset(l2c);
		cfifo_reset(c2l);
	}

	ret = coma_register(SERVICE_ID, SERVICE_NAME, l2c, c2l,
	                    coma_alsa_process_message,
	                    coma_alsa_setup,
	                    coma_alsa_remove);
	if (ret < 0) {
		pr_err("%s: error: cannot register service\n", SERVICE_NAME);
		cfifo_free((char *)l2c, cfifo_size, mem_handle);
	}

	mutex_init(&coma_alsa_mutex);

}

void alsa_service_exit(void)
{
	coma_deregister(SERVICE_ID);
//	cfifo_free((char *)l2c, cfifo_size, mem_handle);
//	cfifo_free((char *)c2l, cfifo_size, mem_handle);
}

