/*
 *  linux/arch/arm/mach-dmw/camera.h
 *
 *  Copyright (C) 2012 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 * DMW96 camera enumerator. used to associate front/back camera ports,
 * which share the same CIU, with individual camera sensors
 *
 */

#include <linux/errno.h>
#include <linux/platform_device.h>
#include <media/dmw96ciu.h>
#include "include/mach/camera.h"

/************************
 * interfaces
 ************************/

/* port power control, called by sensors */
int dmw_camera_port_power_ctl( dmw_camera_port_t * port,
							   enum v4l2_power	   power )
{
	return port->power_ctl( port,
					 	    power,
					 	    port->sensor );
}

/* port clocking control, called by sensors */
int dmw_camera_port_clock_ctl( dmw_camera_port_t * port,
							   unsigned long 	   xclkfreq )
{
	return port->clock_ctl( port,
					 	    xclkfreq );
}

/* start using i2c */
struct i2c_client * dmw_camera_get_i2c( dmw_camera_port_t * 	   port,
										dmw_camera_sensor_desc_t * sensor_desc )
{
	struct i2c_adapter * adap;

	BUG_ON( !port );
	BUG_ON( !sensor_desc );

	/* if someone has the i2c, return */
	if ( port->i2c_ref ) {
		/* we don't support simultaneous accesses from multiple sensors */
		BUG_ON( ((port->sensor) && (port->sensor != sensor_desc)) );

		/* update refcount */
		port->i2c_ref++;

		return port->i2c_client;
	}

	/* lookup the i2c adapter */
	adap = i2c_get_adapter( port->i2c_adapter_num );

	// TODO DEBUG
	BUG_ON( !adap );

	if ( !adap ) {
		return NULL;
	}

	/* update refcount */
	port->i2c_ref++;

	/* set i2c addr, name to match probed sensor */
	strcpy( port->i2c_board_info.type,
			sensor_desc->name );

	port->i2c_board_info.addr = sensor_desc->i2c_addr;

	/* create a new client for this sensor */
	port->i2c_client = i2c_new_device( adap,
									   &(port->i2c_board_info) );

	/* release the adapter */
	i2c_put_adapter( adap );

	return port->i2c_client;
}

/* stop using i2c */
void dmw_camera_put_i2c( dmw_camera_port_t * port )
{
	BUG_ON( !port->i2c_client );

	/* update refcount */
	--port->i2c_ref;

	/* someone still uses the i2c, return */
	if ( port->i2c_ref ) {
		return;
	}

	/* release the client */
	i2c_unregister_device( port->i2c_client );

	/* reset */
	port->i2c_client = NULL;
}

/* associate sensor with port. called after sensor is up */
void dmw_camera_associate_sensor_with_port( dmw_camera_port_t * 	   port,
											dmw_camera_sensor_desc_t * sensor_desc )
{
	/* can't have an existing sensor */
	BUG_ON( port->sensor );

	printk( "sensor %s detected on port %s\n",
			sensor_desc->name,
			port->name );

	/* associate */
	port->sensor = sensor_desc;
}

/* initialize port */
int dmw_camera_initialize_port( dmw_camera_port_t * port )
{
	int 					   			i;
	int 					   			ret;
	dmw_camera_sensor_desc_t * 			sensor_desc;
	dmw_camera_sensor_platform_data_t * sensor_pdata;

	ret = -ENODEV;

	port->i2c_client = NULL;
	port->i2c_ref  	 = 0;
	port->sensor 	 = NULL;

	printk( "initializing camera port %s, i2c adapter id %d\n",
			port->name,
			port->i2c_adapter_num );

	for ( i = 0; i < port->num_sensors; i++ ) {

		/* get the sensor platform data as coupled to the sensor platform device */
		sensor_pdata = (dmw_camera_sensor_platform_data_t *) platform_get_drvdata( port->sensor_list[i] );

		/* if no driver is associated with this sensor, skip */
		if ( !sensor_pdata ) {
			continue;
		}

		/* extract the sensor descriptor */
		sensor_desc = sensor_pdata->__sensor_desc;

		BUG_ON( !sensor_desc );

		printk( "\t-> probing: %s (at 0x%x)\n",
				sensor_desc->name,
				sensor_desc->i2c_addr );

		/* invoke probing */
		sensor_desc->probe( sensor_desc,
						    port );
	}

	return ret;
}
