#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2009 Free Software Foundation
#
# FILE:
# GRParser.py
#
# DESCRIPTION:
"""
 Class that contains a SAX-based xml processor for GNUe Reports
"""
# NOTES:
#


from gnue.common.datasources import GDataSource
from gnue.common.formatting import GTypecast
from gnue.common.definitions import GParser
from gnue.common.logic import GTrigger
import copy, types






########
########  Please keep this file neat !!!
########







#######################################################
# This method loads a report from an XML file and returns
# a GRReport object.  If initialize is 1 (default), then
# the report is initialized and ready to go.
#######################################################

def loadFile(buffer, connections=None, initialize=1):
  return GParser.loadXMLObject (buffer, xmlReportHandler, 'GRReport', 'report',
           initialize, attributes={"_connections": connections})



xmlElements = None


def getXMLelements():

  global xmlElements

  # Here due to some nasty recursive imports
  from gnue.reports.base import GRReport, GRSources, GRParameters, GRSortOptions
  from gnue.reports.base import layout as GRLayout

  if xmlElements == None:

    #
    #
    xmlElements = {
      'report':       {
         'BaseClass': GRReport.GRReport,
         'Required': 1,
         'SingleInstance': 1,
         'Attributes':  {
            'title':       {
               'Typecast': GTypecast.text },
            'author':       {
               'Typecast': GTypecast.text },
            'version':       {
               'Typecast': GTypecast.name },
            'description':       {
               'Typecast': GTypecast.text } } ,
         'ParentTags':  None },

      'parameters':   {
         'BaseClass': GRParameters.GRParameters ,
         'SingleInstance': 1,
         'ParentTags':  ('report',)},

      'parameter':    {
         'BaseClass': GRParameters.GRParameter,
         'Attributes': {
            'name':          {
               'Required': 1,
               'Unique': 1,
               'Typecast': GTypecast.name },
            'required':    {
               'Typecast': GTypecast.boolean,
               'Default': 0 },
            'limited':     {
               'Typecast': GTypecast.boolean,
               'Default': 0 },
            'default':     {
               'Typecast': GTypecast.text },
            'description': {
               'Required': 1,
               'Typecast': GTypecast.text },
            'source':      {
               'Typecast': GTypecast.name },
            'type':        {
               'Typecast': GTypecast.name,
               'Default': "char" } },
         'ParentTags':  ('parameters',) },

      'sortoptions':  {
         'BaseClass': GRSortOptions.GRSortOptions,
         'SingleInstance': 1,
         'ParentTags':  ('report',) },

      'sortoption':   {
         'BaseClass': GRSortOptions.GRSortOption,
         'Attributes': {
            'id':          {
               'Required': 1,
               'Unique': 1,
               'Typecast': GTypecast.name },
            'description': {
               'Required': 1,
               'Typecast': GTypecast.name },
            'default':     {
               'Typecast': GTypecast.boolean,
               'Default': 0 } },
         'ParentTags':  ('sortoptions',) },

      'sortcolumn':   {
         'BaseClass': GRSortOptions.GRSortColumn,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name } },
         'ParentTags':  ('sortoption',) },

      'layout':       {
         'BaseClass': GRLayout.GRLayout,
         'Required': 1,
         'SingleInstance': 1,
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'Attributes': {
            'format':      {
               'Typecast': GTypecast.text },
            'suppressGNUeTags': {
               'Typecast': GTypecast.boolean } },
         'ParentTags':  ('report',) },

      'section':      {
         'BaseClass': GRLayout.GRSection,
         'Required': 1,
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'Attributes': {
            'name':        {
               'Typecast': GTypecast.name },
            'source':      {
               'Typecast': GTypecast.name } },
         'ParentTags':  ('layout',),
         'UsableBySiblings': True },

      'default':      {
         'BaseClass': GRLayout.GRDefault,
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'ParentTags':  ('section',),
         'UsableBySiblings': True },

      'firstrow':      {
         'BaseClass': GRLayout.GRFirstRow,
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'ParentTags':  ('section',),
         'UsableBySiblings': True },

      'notfirstrow':      {
         'BaseClass': GRLayout.GRNotFirstRow,
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'ParentTags':  ('section',),
         'UsableBySiblings': True },

      'lastrow':      {
         'BaseClass': GRLayout.GRLastRow,
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'ParentTags':  ('section',),
         'UsableBySiblings': True },

      'notlastrow':      {
         'BaseClass': GRLayout.GRNotLastRow,
         'MixedContent': 1,
         'KeepWhitespace': 1,
         'ParentTags':  ('section',),
         'UsableBySiblings': True },

      'field':        {
         'BaseClass': GRLayout.GRField,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name },
            'source':      {
               'Typecast': GTypecast.name },
            'section':      {
               'Typecast': GTypecast.name },
            'format':      {
               'Typecast': GTypecast.name } },
         'ParentTags':  ('section',),
         'UsableBySiblings': True },

      'summ':         {
         'BaseClass': GRLayout.GRSumm,
         'Attributes': {
            'name':        {
               'Typecast': GTypecast.name },
            'function':    {
               'Typecast': GTypecast.name,
               'Default': "count"},
            'field':      {
               'Typecast': GTypecast.name,
               'Default': None},
            'formula':      {
               'Typecast': GTypecast.name,
               'Default': None},
            'section':     {
               'Typecast': GTypecast.name },
            'format':      {
               'Typecast': GTypecast.name } },
         'ParentTags':  ('layout',),
         'UsableBySiblings': True },

      'formula':         {
         'BaseClass': GRLayout.GRFormula,
         'Attributes': {
            'name':        {
               'Typecast': GTypecast.name },
            'format':      {
               'Typecast': GTypecast.name } },
         'ParentTags':  ('layout',),
         'UsableBySiblings': True },

      'param':         {
         'BaseClass': GRLayout.GRParam,
         'Attributes': {
            'name':        {
               'Required': 1,
               'Typecast': GTypecast.name },
            'format':      {
               'Typecast': GTypecast.name } },
         'ParentTags':  ('layout',),
         'UsableBySiblings': True },

      'sources':      {
         'BaseClass': GRSources.GRSources,
         'SingleInstance': 1,
         'ParentTags':  ('report',) },
    }

    #
    # Add DataSource elements
    #
    xmlElements.update(
       GDataSource.getXMLelements(
          {'datasource':   {
               'BaseClass': GRSources.GRDataSource,
               'ParentTags':  ('sources',) },
           'cparam': {
               'BaseClass': GRParameters.GRConditionParam }
          }))

    #
    # Add trigger elements
    #
    xmlElements.update(
      GTrigger.getXMLelements(
        updates={'trigger':{
                   'ParentTags': ('report',)},
                }))



  return GParser.buildImportableTags('report',xmlElements)


#######################################################
#
# xmlReportHandler
#
# This class is called by the XML parser to
# process the xml file.
#
#######################################################

class xmlReportHandler (GParser.xmlHandler):
  def __init__(self):
    GParser.xmlHandler.__init__(self)

    self.xmlElements = getXMLelements()

    #
    # Any namespace-qualified tags will be Pass-Thru tags..
    # The reporting engine actually doesn't care about these.
    #
    # TODO: This is a hack and won't work if we proxy
    # TODO: layout events and create layout engines. ??
    #
    from gnue.reports.base.layout import GRPassThru
    self.xmlMasqueradeNamespaceElements = GRPassThru

    #
    self.xmlNamespaceAttributesAsPrefixes = 1

