# GNU Enterprise Common Library - PostgreSQL database driver using PyGreSQL
#
# Copyright 2000-2007 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: pygresqldrv.py 9222 2007-01-08 13:02:49Z johannes $

"""
Database driver plugin for PostgreSQL backends using the PyGreSQL DBSIG2
module.
"""

__all__ = ['Connection']

__pluginalias__ = ['pygresql']

from gnue.common.datasources.drivers.sql.postgresql import Base


# =============================================================================
# Test if plugin is functional
# =============================================================================

def __initplugin__ ():
  from gnue.common.datasources import GConnections
  try:
    import pgdb
  except ImportError:
    raise GConnections.DependencyError, ('pygresql', None)


# =============================================================================
# Driver info
# =============================================================================

class DriverInfo:

  name = "PyGreSQL"

  url = "http://druid.net/pygresql/"

  doc = """
Description
-----------
PyGreSQL is the PostgreSQL-Python driver available via PostgreSQL's CVS tree.
It is compliant to Python's DB-API 2.0.

Support
-------
Supported Platforms:

  - Linux/BSD

Connection Properties
---------------------
* dbname -- This is the database to use (required)
* host  -- This is the name of the database host, or, on Linux,
    directory containing the network socket (optional)
* port -- Port that PostgreSQL is running on (defaults to 5432) (optional)
* use_oid -- if set to any value, the driver uses Postgres' OID field as an
    implicit primary key for all tables. OID fields were generated by default
    for all tables for Postgres 7.x and are not generated by default for
    Postgres 8.x.

Examples
--------
  [myconn]
  provider=pygresql       # Use the pygresql adapter
  host=localhost          # The hostname/IP of the postgresql host
  dbname=mydb             # The name of the pg database

Notes
-----
1. The developers of GNUe have had mixed results with the PygreSQL driver
   and generally recommend one of the others.
"""


# =============================================================================
# Connection class
# =============================================================================

class Connection (Base.Connection):
  """
  Connection class for PostgreSQL backends using the PyGreSQL DBSIG2 module.
  """

  _drivername_ = 'pgdb'
  _need_rollback_after_exception_ = True


  # ---------------------------------------------------------------------------
  # Create an apropriate time object for the given values
  # ---------------------------------------------------------------------------

  def _createTime_ (self, hour, minute, second, msec = 0):
    """
    PygreSQL time handling is broken, we need to use the Timestamp constructor
    instead of the Time constructor
    """

    psec = second + float (msec) / 1000000

    return self._driver.Timestamp (1970, 1, 1, hour, minute, psec)
