/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Copyright (C) 2010 University of Waikato, Hamilton, NZ
 */

package weka.core;

import java.util.ArrayList;
import java.util.Arrays;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import junit.textui.TestRunner;
import weka.core.converters.ConverterUtils.DataSource;

/**
 * Tests Instances. Run from the command line with:<p/>
 * java weka.core.InstancesTest
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 11450 $
 */
public class InstancesTest
  extends TestCase {
  
  private static final double TOLERANCE = 1e-14;
  
  /** the test instances to work with. */
  protected Instances m_Instances;
  
  /**
   * Constructs the <code>InstancesTest</code>.
   *
   * @param name 	the name of the test
   */
  public InstancesTest(String name) { 
    super(name); 
  }
  
  /**
   * Called by JUnit before each test method.
   *
   * @throws Exception 	if an error occurs
   */
  protected void setUp() throws Exception {
    super.setUp();
    
    m_Instances = DataSource.read(ClassLoader.getSystemResourceAsStream("weka/core/data/InstancesTest.arff"));
  }

  /**
   * Called by JUnit after each test method.
   *
   * @throws Exception 	if an error occurs
   */
  protected void tearDown() throws Exception {
    m_Instances = null;

    super.tearDown();
  }

  /**
   * Returns the test suite.
   * 
   * @return		the test suite
   */
  public static Test suite() {
    return new TestSuite(InstancesTest.class);
  }

  /**
   * Tests the behavior of the {@link Instances.variance} and
   * {@link Instances.variances} methods.
   */
  public void testVarianceBehavior() {
    Instances inst;
    
    // weight <= 1 leads to NaN variance:
    inst = getNumericInstances();
    addWeightedValues(inst, 100, 0.5);

    assertTrue("Instances.variance not NaN for weight <= 1!",
        Double.isNaN(inst.variance(0)));
    assertTrue("Instances.variance not NaN for weight <= 1!",
        Double.isNaN(inst.variances()[0]));

    // even if count > 1, weight <= 1 leads to NaN variance:
    inst = getNumericInstances();
    addWeightedValues(inst, 100, 0.5, 100, 0.3);

    assertTrue("Instances.variance not NaN for weight <= 1!",
        Double.isNaN(inst.variance(0)));
    assertTrue("Instances.variance not NaN for weight <= 1!",
        Double.isNaN(inst.variances()[0]));
    
  }
  
  /**
   * Tests the accuracy of the {@link Instances.variance} and
   * {@link Instances.variances} methods.
   */
  public void testVarianceAccuracy() {
    checkVariance(generatedWeightedValues1, generatedVar1);
    checkVariance(generatedWeightedValues2, generatedVar2);
    checkVariance(generatedWeightedValues3, generatedVar3);
    checkVariance(generatedWeightedValues4, generatedVar4);
    checkVariance(generatedWeightedValues5, generatedVar5, 1e-2);
    checkVariance(generatedWeightedValues6, generatedVar6, 2);
    checkVariance(generatedWeightedValues7, generatedVar7);
    checkVariance(generatedWeightedValues8, generatedVar8);
    checkVariance(generatedWeightedValues9, generatedVar9);
    checkVariance(generatedWeightedValues10, generatedVar10);
    checkVariance(generatedWeightedValues11, generatedVar11);
    checkVariance(generatedWeightedValues12, generatedVar12);
    checkVariance(generatedWeightedValues13, generatedVar13);
    checkVariance(generatedWeightedValues14, generatedVar14, 3);
    checkVariance(generatedWeightedValues15, generatedVar15);
    checkVariance(generatedWeightedValues16, generatedVar16, 2);
    checkVariance(generatedWeightedValues17, generatedVar17);
    checkVariance(generatedWeightedValues18, generatedVar18, 2);
    checkVariance(generatedWeightedValues19, generatedVar19);
    checkVariance(generatedWeightedValues20, generatedVar20, 2);
    checkVariance(generatedWeightedValues21, generatedVar21);
    checkVariance(generatedWeightedValues22, generatedVar22);
  }
  
  private Instances getNumericInstances() {
    ArrayList<Attribute> atts = new ArrayList<Attribute>(1);
    atts.add(new Attribute("data points"));
    return new Instances("VarianceTest", atts, 0);
  }
  
  private void addWeightedValues(Instances inst, double... weightedValues) {
    assert weightedValues.length % 2 == 0;

    for (int i = 0; i < weightedValues.length; i += 2) {
      inst.add(new DenseInstance(
          weightedValues[i + 1],
          new double[]{weightedValues[i]}
          ));
    }
  }
  
  private void checkVariance(long[] weightedValues, long var) {
    checkVariance(weightedValues, var, TOLERANCE); }
  
  private void checkVariance(long[] weightedValues, long var, double tol) {
    double varDouble = Double.longBitsToDouble(var);

    Instances inst = getNumericInstances();
    for (int i = 0; i < weightedValues.length; i += 2) {
      inst.add(new DenseInstance(
          Double.longBitsToDouble(weightedValues[i + 1]),
          new double[]{Double.longBitsToDouble(weightedValues[i])}
          ));
    }
    
    assertEquals("Inaccurate variance calculation!",
        varDouble, inst.variance(0), Math.abs(varDouble*tol));
    assertEquals("Inaccurate variance calculation!",
        varDouble, inst.variances()[0], Math.abs(varDouble*tol));
  }
  
  /**
   * Tests the creation of a dataset (unique attribute names).
   * 
   * @see Instances#Instances(String, ArrayList, int)
   */
  public void testCreationUnique() {
    Instances			data;
    ArrayList<Attribute>	atts;
    String			relName;
    
    relName = "testCreationUnique";
    atts    = new ArrayList<Attribute>();
    atts.add(new Attribute("att-numeric_1"));
    atts.add(new Attribute("att-numeric_2"));
    atts.add(new Attribute("att-data_1", "yyyy-MM-dd HH:mm"));
    atts.add(new Attribute("att-nominal_1", new ArrayList<String>(Arrays.asList(new String[]{"1", "2", "3"}))));
    atts.add(new Attribute("att-nominal_2", new ArrayList<String>(Arrays.asList(new String[]{"yes", "no"}))));
    atts.add(new Attribute("att-string_1", (ArrayList<String>) null));
    data = new Instances(relName, atts, 0);
    
    assertEquals("relation name differs", relName, data.relationName());
    assertEquals("# of attributes differ", atts.size(), data.numAttributes());
  }
  
  /**
   * Tests the creation of a dataset (ambiguous attribute names).
   * 
   * @see Instances#Instances(String, ArrayList, int)
   */
  public void testCreationAmbiguous() {
    Instances			data;
    ArrayList<Attribute>	atts;
    String			relName;
    
    relName = "testCreationAmbiguous";
    atts    = new ArrayList<Attribute>();
    atts.add(new Attribute("att-numeric_1"));
    atts.add(new Attribute("att-numeric_1"));
    atts.add(new Attribute("att-data_1", "yyyy-MM-dd HH:mm"));
    atts.add(new Attribute("att-nominal_1", new ArrayList<String>(Arrays.asList(new String[]{"1", "2", "3"}))));
    atts.add(new Attribute("att-nominal_1", new ArrayList<String>(Arrays.asList(new String[]{"yes", "no"}))));
    atts.add(new Attribute("att-string_1", (ArrayList<String>) null));
    
    try {
      data = new Instances(relName, atts, 0);
    }
    catch (IllegalArgumentException e) {
      data = null;
    }
    assertNull("dataset created with ambiguous attribute names", data);
  }
  
  /**
   * Tests the copying of the header of a dataset.
   * 
   * @see Instances#Instances(Instances, int)
   */
  public void testHeaderCopy() {
    Instances 	data;
    
    data = new Instances(m_Instances, 0);
    assertEquals("# of attributes differ", m_Instances.numAttributes(), data.numAttributes());
    assertEquals("class index differs", m_Instances.classIndex(), data.classIndex());
    assertEquals("Unexpected instances", 0, data.numInstances());

    m_Instances.setClassIndex(m_Instances.numAttributes() - 1);
    data = new Instances(m_Instances, 0);
    assertEquals("class index differs", m_Instances.classIndex(), data.classIndex());
  }
  
  /**
   * Tests the full copy of a dataset.
   * 
   * @see Instances#Instances(Instances)
   */
  public void testFullCopy() {
    Instances data;
    
    data = new Instances(m_Instances);
    assertEquals("# of attributes differ", m_Instances.numAttributes(), data.numAttributes());
    assertEquals("class index differs", m_Instances.classIndex(), data.classIndex());
    assertEquals("# of instances differ", m_Instances.numInstances(), data.numInstances());

    m_Instances.setClassIndex(m_Instances.numAttributes() - 1);
    data = new Instances(m_Instances);
    assertEquals("class index differs", m_Instances.classIndex(), data.classIndex());
  }
  
  /**
   * Tests the partial copy of a dataset.
   * 
   * @see Instances#Instances(Instances, int, int)
   */
  public void testPartialCopy() {
    Instances data;
    
    data = new Instances(m_Instances, 0, m_Instances.numInstances());
    assertEquals("# of instances differ", m_Instances.numInstances(), data.numInstances());

    data = new Instances(m_Instances, 5, 10);
    assertEquals("# of instances differ", 10, data.numInstances());
  }

  /**
   * Executes the test from command-line.
   * 
   * @param args	ignored
   */
  public static void main(String[] args){
    TestRunner.run(suite());
  }

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 0.000000e+00; weights_ordering = Random
  private static final long generatedVar1 = 4846484824648898872L; // approx ~ 1.023045e+16
  private static final long[] generatedWeightedValues1 = {4728882238542208474L, 4607182418800017408L, 4724297175877502601L, 4607182418800017408L, 4731876825100680126L, 4607182418800017408L, 4729943263241482912L, 4607182418800017408L, 4725557600258270159L, 4607182418800017408L, 4729345099550339492L, 4607182418800017408L, -4511529973594946376L, 4607182418800017408L, -4499776894199643076L, 4607182418800017408L, 4725493748071799552L, 4607182418800017408L, -4510366045959106760L, 4607182418800017408L, -4505288058990558360L, 4607182418800017408L, 4727779447587440858L, 4607182418800017408L, -4511015767980080824L, 4607182418800017408L, 4733212210121780867L, 4607182418800017408L, 4729772619875419738L, 4607182418800017408L, 4732168811792281457L, 4607182418800017408L, 4722686605778703701L, 4607182418800017408L, 4728535520138398304L, 4607182418800017408L, 4731477632314922232L, 4607182418800017408L, 4730977381367954693L, 4607182418800017408L, 4731754764860750329L, 4607182418800017408L, 4729112738205149132L, 4607182418800017408L, 4713445635474141744L, 4607182418800017408L, 4732046769387429390L, 4607182418800017408L, 4730376610848151648L, 4607182418800017408L, 4719794279002592627L, 4607182418800017408L, 4724522920327633506L, 4607182418800017408L, 4732498947896005933L, 4607182418800017408L, -4500865876371234980L, 4607182418800017408L, 4732002611912376005L, 4607182418800017408L, 4733341815312740015L, 4607182418800017408L, 4727671273135572123L, 4607182418800017408L, -4498690684394321004L, 4607182418800017408L, 4718992664958101708L, 4607182418800017408L, 4730178627501669707L, 4607182418800017408L, -4500525510087317668L, 4607182418800017408L, 4730481312031545740L, 4607182418800017408L, 4724808925415164141L, 4607182418800017408L, -4502372516229779408L, 4607182418800017408L, -4505310934268662536L, 4607182418800017408L, 4721858224388437273L, 4607182418800017408L, 4727093387733937480L, 4607182418800017408L, -4508507260000721928L, 4607182418800017408L, -4499059661902671520L, 4607182418800017408L, 4732644973750086170L, 4607182418800017408L, 4729366653279193268L, 4607182418800017408L, 4728840096080803372L, 4607182418800017408L, -4502265166720238012L, 4607182418800017408L, 4716014656055770004L, 4607182418800017408L, 4730288480517034618L, 4607182418800017408L, 4725814338898514444L, 4607182418800017408L, 4729715543396825008L, 4607182418800017408L, -4499910072273722208L, 4607182418800017408L, 4731173392608590006L, 4607182418800017408L, 4731739814882602966L, 4607182418800017408L, 4732439262319897621L, 4607182418800017408L, 4728454366754411248L, 4607182418800017408L, 4726762614886416308L, 4607182418800017408L, 4729754882936938574L, 4607182418800017408L, 4731893206463466506L, 4607182418800017408L, 4716849310072850312L, 4607182418800017408L, 4728326322834546592L, 4607182418800017408L, -4506209325053939412L, 4607182418800017408L, 4721561304413860391L, 4607182418800017408L, -4500535973156195220L, 4607182418800017408L, -4503466462706944028L, 4607182418800017408L, 4724016589220206171L, 4607182418800017408L, -4517053138999825104L, 4607182418800017408L, 4731378274237806490L, 4607182418800017408L, 4723351267368183576L, 4607182418800017408L, 4733343606537957742L, 4607182418800017408L, 4715352326303886072L, 4607182418800017408L, 4731074504410326908L, 4607182418800017408L, 4725678574159331048L, 4607182418800017408L, 4709025026149596256L, 4607182418800017408L, 4732326896774211134L, 4607182418800017408L, 4715765061828556204L, 4607182418800017408L, 4720300547219232870L, 4607182418800017408L, -4507656352905579236L, 4607182418800017408L, -4499877223662369516L, 4607182418800017408L, 4729068780272619998L, 4607182418800017408L, 4718177709378698588L, 4607182418800017408L, 4730416812677164759L, 4607182418800017408L, 4717027386500823440L, 4607182418800017408L, 4731090415843949842L, 4607182418800017408L, 4730411586870855780L, 4607182418800017408L, 4726692326825475826L, 4607182418800017408L, 4725042519138069515L, 4607182418800017408L, 4730179878966512002L, 4607182418800017408L, 4732659188735280944L, 4607182418800017408L, 4721848950156368874L, 4607182418800017408L, -4500890900926467748L, 4607182418800017408L, 4727946932764405435L, 4607182418800017408L, -4500451955247714496L, 4607182418800017408L, 4728976659903966833L, 4607182418800017408L, 4724985947847046225L, 4607182418800017408L, 4727507719439666307L, 4607182418800017408L, 4731103733958097378L, 4607182418800017408L, -4500305217662075524L, 4607182418800017408L, 4724656629577169039L, 4607182418800017408L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Random
  private static final long generatedVar2 = 4845954430503588592L; // approx ~ 9.169662e+15
  private static final long[] generatedWeightedValues2 = {4729323142217213443L, 4605349566362591809L, 4723711352455312527L, 4608741619390252714L, 4728990318963065423L, 4610136171090288780L, 4721116397571532503L, 4605494327174332376L, 4725561774399326372L, 4609118199992285378L, 4733133218988950512L, 4599582752191746802L, 4728717972209676131L, 4600495886246374956L, -4503355289129338376L, 4607752204505722147L, 4722937846164705009L, 4602775687168185486L, 4730845896038768151L, 4606544843064828417L, -4515302148183552336L, 4602861177544798805L, -4512799545563857344L, 4601175698007518104L, 4728839035053154855L, 4607546656429425316L, 4711172051843903712L, 4608442448985989521L, 4730755128938045054L, 4599897120962617194L, 4729774319541858423L, 4610292538501614046L, 4729237761845654189L, 4603738251862352977L, 4724354511411158084L, 4605031778010038744L, 4726216715330493969L, 4607593851510042434L, -4504649376391565840L, 4605181358163508686L, 4717142813434071964L, 4600375650667615206L, 4731680676622582388L, 4604721608646180503L, -4506839905284996668L, 4605517277443121944L, 4726777915123297186L, 4607273527020543456L, 4733150174898800734L, 4609669186547087138L, -4523777924287131712L, 4604018576914713099L, -4503047894453526912L, 4609905520183638028L, 4712165044358495728L, 4607542801916422748L, 4724483563381356876L, 4607678814832292858L, 4731351104236074039L, 4603734178368533620L, 4716287140294550660L, 4600429095312247572L, 4720822351512492625L, 4609567825253094284L, -4502266694540626488L, 4606415361265462859L, 4716710961732499616L, 4608645487321721108L, 4726991150548469990L, 4605975359788831737L, -4501160088090119436L, 4603813239160915724L, 4732142190855722617L, 4608981868968230918L, -4510953787167818672L, 4609984517679141844L, 4729924082213789928L, 4609375020298367138L, -4499650522493266528L, 4608592320274912175L, 4727843351372385061L, 4607586337779928489L, 4697399131833226944L, 4606933783547003096L, 4725614007082931272L, 4610066217903023065L, 4728893289901446390L, 4607447166573470862L, -4504530597144079812L, 4607406831574100140L, 4715698834495128544L, 4600449791806109292L, -4510012488496334360L, 4606717543089984883L, -4498818226827455316L, 4604061913479926772L, -4506336980161079528L, 4600805428581838210L, 4714075034615894704L, 4608867700102960436L, 4717748773408751772L, 4607969388985041110L, -4513103756713801264L, 4609133150218321044L, 4731410321606255017L, 4604997604747097110L, 4722976679495668386L, 4607699346594155219L, 4725106840181560110L, 4605257820345552733L, 4731060621314621535L, 4607137405689407482L, 4732696504941884940L, 4603271508458733492L, 4732404724810770798L, 4606665381343112300L, 4731779740686030836L, 4605751086963215209L, 4718492705442193692L, 4607405499830459479L, 4727243205912488161L, 4605986396397296532L, 4726882120742769758L, 4608386476760210401L, 4729493024529038628L, 4603791172805876017L, 4721750309392461055L, 4603659737062210664L, 4730793564842068850L, 4606764981105852940L, 4730216978761876931L, 4608486382850291770L, 4727812669594304630L, 4600241367692853238L, 4718419588878748136L, 4609154920009688040L, 4725577539621193516L, 4609235332740895570L, 4726150803812253957L, 4608548562406353616L, 4700278833559888960L, 4606149757199762008L, 4727968108170018476L, 4605701697289708391L, -4504371317152599404L, 4602922293494588532L, -4502132259779425048L, 4610298446653392404L, -4499609127281450972L, 4608089223030871424L, -4507525592779163236L, 4607499037206621727L, 4707401292046610880L, 4608319736939095553L, 4731310976034623974L, 4605925797299060243L, 4727282024084847291L, 4605478192941036933L, 4728053324579319809L, 4608021725613577939L, 4723307003270298768L, 4602675137180989078L, 4732680339043480952L, 4605385855777971385L, 4732946536135350428L, 4603898083822778132L, 4729121590079028547L, 4608632647375475289L, -4499171697447510328L, 4602022277129090188L, 4730753094966916448L, 4609895239697573080L, 4731424306336134033L, 4607235371541135717L, 4732983462459749488L, 4606196530269673318L, 4717467404942736168L, 4604337770891144908L, 4728208002503526990L, 4606873716890193972L, -4504715182509145888L, 4610110173097513412L, 4733310579387719538L, 4605741479834720833L, 4720860585794290590L, 4608165048911644733L, 4731208196595247920L, 4608825134401668841L, 4727432854223417146L, 4609462559924503025L, -4506886325157586948L, 4604652260683971855L, 4732982862733901127L, 4606152604114328696L, 4730393276059900889L, 4604410386509615726L, 4730585719807556805L, 4600368806109740738L, 4729468363117879402L, 4605018263396474446L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Random
  //weights_mean = 1.500000e+01; weights_stdDev = 7.000000e+00; weights_ordering = Random
  private static final long generatedVar3 = 4846557962988966698L; // approx ~ 1.037673e+16
  private static final long[] generatedWeightedValues3 = {-4503807140920556460L, 4622755662931528716L, 4731725093772953266L, 4625589980208343423L, 4696610537905121664L, 4625920092409723279L, 4720580071055854319L, 4616997743744594858L, -4500714132950646128L, 4617117908242328910L, 4729686229769871496L, 4625301702191936772L, 4724477618105401801L, 4624698925222337134L, 4725960563863096454L, 4625587525999524032L, 4729689489488353917L, 4627856996645153365L, 4715624699361848004L, 4627656919349902018L, -4503390555721937696L, 4622608132948180144L, -4501811965634370968L, 4627844716192901380L, -4501213352937808592L, 4625264387863764717L, 4726943041268491804L, 4622438128733822184L, 4729835511574953206L, 4620787799765423913L, 4721250632922291656L, 4626108944843160419L, 4727708835291137449L, 4626343847328499756L, 4706434092057878464L, 4621262698856917788L, 4725970252303357468L, 4617030210635416638L, -4512382468320528776L, 4628170216644696760L, -4511848721358463824L, 4615151139320241816L, 4729173846240056022L, 4619832016315588480L, 4731851428972243016L, 4621274762364150296L, 4727527116131208882L, 4628142877556181848L, 4727082279302528794L, 4624051179168377637L, 4728367725311964173L, 4621391348873615596L, 4730404011801043330L, 4626944545747511307L, 4733000418925767107L, 4628285756977047092L, 4730374095942428388L, 4626618388719275209L, 4729670502781499850L, 4626034532415703078L, 4719602641933674210L, 4625603565000998112L, -4502025711537363148L, 4628161229549447224L, 4727710469679897610L, 4628106255457782336L, 4730614713997737299L, 4625135514464093512L, 4730499322129870239L, 4628179786461828527L, -4502768644638287132L, 4623223329876407329L, 4731202964612314648L, 4627511587802424220L, 4730116615807410741L, 4623554831093989807L, 4729529748217405363L, 4621813123700564520L, -4499658134143641896L, 4627117758955391152L, 4731953398328628342L, 4626704458994472891L, 4730879956888117461L, 4618346420040478578L, 4725021279138180442L, 4621585210918622576L, 4724315930038865178L, 4620746881832112893L, 4731395414861825603L, 4626754062533883499L, 4731945456149368592L, 4626711582125973256L, -4510903260305167984L, 4625553374630814540L, 4731022945368236839L, 4620332075291314426L, -4499030147977413840L, 4619826163192389505L, -4501305844413614472L, 4624193966636017118L, -4502757629749995508L, 4623168081206919661L, 4730047612085276335L, 4616350146978350012L, 4732531030828527498L, 4626444866957609096L, 4731911838244311930L, 4614315496604759400L, -4500227287680357992L, 4626854081426751169L, 4729268477207751146L, 4623241452507396180L, 4729170539081458956L, 4626626618078568403L, 4725775579682665887L, 4625306798941172942L, 4716915612021267484L, 4623141689611282608L, 4724754119127655592L, 4623809245165458435L, 4732157397804825171L, 4627998414138367104L, 4728833948096107646L, 4624580350466907778L, -4522077383439774400L, 4621636374112328926L, 4731414579488453418L, 4627076498923262181L, 4724006875314378220L, 4618246540694127084L, 4732319823626169020L, 4625311184128144258L, 4725140527632323945L, 4626866046190402324L, 4721994364754391466L, 4625609972677067994L, 4730856382117688154L, 4627059205056505892L, 4731179546076526416L, 4627020875318580521L, 4713906189135966384L, 4618338201819725436L, 4725361469767029866L, 4625998139027584558L, -4504093373603755360L, 4625627960511897015L, 4732549616536280319L, 4627392561489574364L, 4722815807481848856L, 4627723793739434930L, 4728171004114269974L, 4626481947590115701L, -4499313440786226696L, 4628223220520175336L, 4732781845493101940L, 4620177148327968265L, 4720946049969394581L, 4623512094879015598L, 4729622612673622691L, 4622106668139611984L, 4728286063516207848L, 4626677261521509189L, -4510099456535796504L, 4626299568708607579L, 4720651985240030249L, 4623318747390079402L, 4708154108480548368L, 4626447590855549404L, 4720685041167866529L, 4613661668678298232L, 4696590718866276736L, 4622112363974370658L, 4731460140835947569L, 4625092466350277899L, 4725532365358532260L, 4617615990965369816L, -4508959374376543328L, 4619180098341064268L, -4502918547619211744L, 4626825284915292815L, -4500719360129695332L, 4622348976205463240L, 4720134910109599196L, 4614521389216880320L, -4498801657710353894L, 4626536926079802233L, 4730873177739765262L, 4624945692572496919L, 4730884474664669988L, 4627046880010461974L, 4728115130648371681L, 4616524494714125620L, 4729077246472497954L, 4627664416003112650L, 4731336584050946475L, 4623383611634500142L, -4505482039752264668L, 4617612174124753980L, 4710975049907830344L, 4624739804675826272L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 0.000000e+00; weights_ordering = Random
  private static final long generatedVar4 = 4369264983939939690L; // approx ~ 1.300867e-16
  private static final long[] generatedWeightedValues4 = {4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L, 4726483295884279809L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279807L, 4607182418800017408L, 4726483295884279808L, 4607182418800017408L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Random
  private static final long generatedVar5 = 4368905548946497962L; // approx ~ 1.212260e-16
  private static final long[] generatedWeightedValues5 = {4726483295884279808L, 4600431312770412264L, 4726483295884279808L, 4607279135448364769L, 4726483295884279808L, 4608588527848664547L, 4726483295884279808L, 4608068012528948212L, 4726483295884279809L, 4604578803762545462L, 4726483295884279809L, 4605592314566982932L, 4726483295884279808L, 4607002884196660424L, 4726483295884279808L, 4608575180871752286L, 4726483295884279807L, 4609364048958247002L, 4726483295884279808L, 4610148728324462154L, 4726483295884279808L, 4609857732061380102L, 4726483295884279809L, 4609318756732074410L, 4726483295884279809L, 4605669036273719471L, 4726483295884279808L, 4610111405478421494L, 4726483295884279807L, 4608486690467642000L, 4726483295884279808L, 4606103341967605120L, 4726483295884279808L, 4604240197286029585L, 4726483295884279809L, 4601567297241886052L, 4726483295884279807L, 4608118713379084174L, 4726483295884279809L, 4602880482501098654L, 4726483295884279807L, 4610293652630883681L, 4726483295884279809L, 4608528372100448295L, 4726483295884279809L, 4610251652212954192L, 4726483295884279808L, 4599659203468028186L, 4726483295884279807L, 4609397903669915324L, 4726483295884279809L, 4608002904991011455L, 4726483295884279808L, 4610254150106833374L, 4726483295884279808L, 4608541558370557556L, 4726483295884279807L, 4603489809815963352L, 4726483295884279809L, 4609828438367150848L, 4726483295884279809L, 4603506790428007642L, 4726483295884279808L, 4608152266551353529L, 4726483295884279808L, 4607705147113854371L, 4726483295884279808L, 4609220036712100580L, 4726483295884279808L, 4603353204282295716L, 4726483295884279809L, 4608866620393666700L, 4726483295884279807L, 4608139275950940571L, 4726483295884279808L, 4609659144212186920L, 4726483295884279808L, 4608425735129336683L, 4726483295884279807L, 4601760296562304570L, 4726483295884279808L, 4600659550216345112L, 4726483295884279809L, 4608084992162437952L, 4726483295884279807L, 4603532565741934930L, 4726483295884279808L, 4608360190037793996L, 4726483295884279809L, 4606862101396289433L, 4726483295884279808L, 4607032725364008499L, 4726483295884279807L, 4607304423804839806L, 4726483295884279808L, 4608996150055354190L, 4726483295884279808L, 4608264934115704568L, 4726483295884279807L, 4600935985160089030L, 4726483295884279809L, 4604343524365080631L, 4726483295884279808L, 4609673501989627034L, 4726483295884279807L, 4607152770705972236L, 4726483295884279809L, 4607000178813859774L, 4726483295884279808L, 4606628176398338672L, 4726483295884279807L, 4609922684312541034L, 4726483295884279808L, 4608889724010418761L, 4726483295884279809L, 4605804439129312958L, 4726483295884279809L, 4609025113654992944L, 4726483295884279809L, 4608433385387922297L, 4726483295884279807L, 4605192169442565815L, 4726483295884279807L, 4603341137102513612L, 4726483295884279808L, 4608401238762623948L, 4726483295884279808L, 4609348055025461856L, 4726483295884279809L, 4603857666912013670L, 4726483295884279808L, 4608895557284752268L, 4726483295884279809L, 4604677966661927724L, 4726483295884279809L, 4607619787351616298L, 4726483295884279809L, 4605047468338094326L, 4726483295884279807L, 4602423924201681948L, 4726483295884279807L, 4607928355493196372L, 4726483295884279809L, 4609778500544856096L, 4726483295884279808L, 4609623906275287708L, 4726483295884279808L, 4608306495204142464L, 4726483295884279809L, 4605060992477389170L, 4726483295884279808L, 4609484929242974653L, 4726483295884279807L, 4609960268254101109L, 4726483295884279807L, 4603388435687760420L, 4726483295884279808L, 4609799322316926021L, 4726483295884279807L, 4607701004035791009L, 4726483295884279807L, 4605482761107495589L, 4726483295884279808L, 4599288967561151106L, 4726483295884279807L, 4602978612007557032L, 4726483295884279808L, 4601766178232241234L, 4726483295884279808L, 4600930157308315470L, 4726483295884279809L, 4605416219685750849L, 4726483295884279809L, 4607345801526403179L, 4726483295884279808L, 4608139897704193919L, 4726483295884279807L, 4608566612084122850L, 4726483295884279808L, 4605109363219474656L, 4726483295884279807L, 4602165864998752136L, 4726483295884279807L, 4604151805710554014L, 4726483295884279808L, 4607665283024084424L, 4726483295884279807L, 4607558288295725657L, 4726483295884279809L, 4608706405472931418L, 4726483295884279809L, 4602155725633041738L, 4726483295884279809L, 4606571798066919854L, 4726483295884279809L, 4600253024781698330L, 4726483295884279808L, 4602890051281796674L, 4726483295884279809L, 4602438718425481848L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Random
  //weights_mean = 1.500000e+01; weights_stdDev = 7.000000e+00; weights_ordering = Random
  private static final long generatedVar6 = 4368620151765967979L; // approx ~ 1.141904e-16
  private static final long[] generatedWeightedValues6 = {4726483295884279807L, 4626860419283271715L, 4726483295884279808L, 4618352511393057780L, 4726483295884279808L, 4620821065319975374L, 4726483295884279809L, 4619211772574084658L, 4726483295884279809L, 4622544275888422389L, 4726483295884279807L, 4618670391639971602L, 4726483295884279808L, 4628158508841338124L, 4726483295884279809L, 4626903145783690845L, 4726483295884279809L, 4627805916478421460L, 4726483295884279808L, 4624282084245060042L, 4726483295884279809L, 4625630220925397353L, 4726483295884279808L, 4618830017489266130L, 4726483295884279808L, 4624476911666067169L, 4726483295884279808L, 4624942709069254360L, 4726483295884279807L, 4621165921871494442L, 4726483295884279808L, 4628182806042460893L, 4726483295884279808L, 4622353845288811002L, 4726483295884279808L, 4628076558403891397L, 4726483295884279807L, 4628247110649166208L, 4726483295884279808L, 4627459525282145296L, 4726483295884279808L, 4627731180413882050L, 4726483295884279809L, 4616439428704081594L, 4726483295884279809L, 4628272839604389580L, 4726483295884279808L, 4627388843226164801L, 4726483295884279809L, 4621326185817887764L, 4726483295884279807L, 4622726883322470966L, 4726483295884279808L, 4624456146392032807L, 4726483295884279809L, 4618419906202802476L, 4726483295884279807L, 4624303585558844402L, 4726483295884279807L, 4624155781951419160L, 4726483295884279808L, 4625470807785783477L, 4726483295884279807L, 4626099219000799230L, 4726483295884279808L, 4620046080949526525L, 4726483295884279808L, 4623108471174709125L, 4726483295884279807L, 4623055890140603438L, 4726483295884279807L, 4625498454517916753L, 4726483295884279807L, 4626041056079222605L, 4726483295884279808L, 4627076376189172234L, 4726483295884279808L, 4627524144545340044L, 4726483295884279808L, 4627478839544697571L, 4726483295884279808L, 4621651238704607751L, 4726483295884279808L, 4624021043452045098L, 4726483295884279808L, 4627312452430555010L, 4726483295884279809L, 4626593888185276953L, 4726483295884279808L, 4621970924673337684L, 4726483295884279809L, 4627730223436298114L, 4726483295884279809L, 4620741237133102864L, 4726483295884279807L, 4625027989170590185L, 4726483295884279809L, 4626459680219390781L, 4726483295884279809L, 4622588217881999938L, 4726483295884279808L, 4626049553185537385L, 4726483295884279807L, 4616402477865728772L, 4726483295884279807L, 4625679588077723382L, 4726483295884279807L, 4622433036957955590L, 4726483295884279809L, 4618063231242100544L, 4726483295884279808L, 4627771958226471372L, 4726483295884279807L, 4621491943287264068L, 4726483295884279807L, 4623758999229986092L, 4726483295884279808L, 4627814874395123825L, 4726483295884279807L, 4621907252857212966L, 4726483295884279807L, 4623365133087232988L, 4726483295884279809L, 4619701029989430825L, 4726483295884279808L, 4623264490508272534L, 4726483295884279808L, 4624983643854489115L, 4726483295884279808L, 4626836496943635196L, 4726483295884279807L, 4624525082464104009L, 4726483295884279808L, 4626383338829665583L, 4726483295884279807L, 4620904855044816261L, 4726483295884279808L, 4625132264361901118L, 4726483295884279807L, 4625459477488702879L, 4726483295884279808L, 4619573369824502134L, 4726483295884279809L, 4627142687733066448L, 4726483295884279808L, 4626410265067686535L, 4726483295884279808L, 4627469381917230176L, 4726483295884279809L, 4620861558684840646L, 4726483295884279808L, 4627225965737580780L, 4726483295884279809L, 4616570131695882380L, 4726483295884279807L, 4626231916218017787L, 4726483295884279807L, 4617447820154657794L, 4726483295884279809L, 4615992757684428136L, 4726483295884279809L, 4617318677208895626L, 4726483295884279807L, 4620883915261522496L, 4726483295884279807L, 4623527933530638566L, 4726483295884279808L, 4627728094995411151L, 4726483295884279807L, 4626860391190151001L, 4726483295884279807L, 4627614933502591460L, 4726483295884279807L, 4624542624639495018L, 4726483295884279808L, 4626282131296046097L, 4726483295884279808L, 4625477681583155276L, 4726483295884279807L, 4626551481972057121L, 4726483295884279809L, 4622422784134209768L, 4726483295884279809L, 4624697671634713829L, 4726483295884279808L, 4614018900736769148L, 4726483295884279809L, 4626433549691543118L, 4726483295884279807L, 4622565064273528613L, 4726483295884279808L, 4624092248139890089L, 4726483295884279807L, 4621153995946878268L, 4726483295884279807L, 4622029684531356316L, 4726483295884279808L, 4622349094915308566L, 4726483295884279807L, 4627917737937153671L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e+08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 0.000000e+00; weights_ordering = Random
  private static final long generatedVar7 = 4846475463431897850L; // approx ~ 1.021173e+16
  private static final long[] generatedWeightedValues7 = {4723380808749750449L, 4607182418800017408L, 4717274636089219782L, 4607182418800017408L, 4721515109291976821L, 4607182418800017408L, 4719187803559483255L, 4607182418800017408L, -4498869951761561925L, 4607182418800017408L, 4725808410171062700L, 4607182418800017408L, 4728095722587025482L, 4607182418800017408L, -4496818894745567134L, 4607182418800017408L, -4501001764281711357L, 4607182418800017408L, -4501096198809497114L, 4607182418800017408L, -4508843425626235112L, 4607182418800017408L, -4496686165911267802L, 4607182418800017408L, 4726879849456150893L, 4607182418800017408L, -4501733744402439947L, 4607182418800017408L, 4729852764393944513L, 4607182418800017408L, -4515615498808769891L, 4607182418800017408L, -4499990864595710804L, 4607182418800017408L, 4723891741144278490L, 4607182418800017408L, 4728836570746226270L, 4607182418800017408L, 4727285770574652916L, 4607182418800017408L, 4728245916289743821L, 4607182418800017408L, 4727835990662513513L, 4607182418800017408L, -4493367473549737088L, 4607182418800017408L, 4726287489344654559L, 4607182418800017408L, 4726941339913924114L, 4607182418800017408L, -4494570793763365313L, 4607182418800017408L, 4713676821518164437L, 4607182418800017408L, 4715774685693860323L, 4607182418800017408L, -4493357383857229297L, 4607182418800017408L, -4498348558217528833L, 4607182418800017408L, -4496113181079050893L, 4607182418800017408L, 4727362478662385529L, 4607182418800017408L, 4721889751417693917L, 4607182418800017408L, -4493971017676528025L, 4607182418800017408L, -4506768259758603572L, 4607182418800017408L, -4495652721784272189L, 4607182418800017408L, 4720464628307755164L, 4607182418800017408L, 4727173919179725205L, 4607182418800017408L, 4725391254120626542L, 4607182418800017408L, -4499808673580342254L, 4607182418800017408L, 4727987020285596379L, 4607182418800017408L, 4725249553333689111L, 4607182418800017408L, 4729856923519117321L, 4607182418800017408L, -4493416584925386352L, 4607182418800017408L, -4494269331614318926L, 4607182418800017408L, 4728171013324054739L, 4607182418800017408L, 4716079154156527503L, 4607182418800017408L, 4725588637494370138L, 4607182418800017408L, 4713277380955204503L, 4607182418800017408L, -4493973128001754375L, 4607182418800017408L, -4493821149959470450L, 4607182418800017408L, -4494300850068304068L, 4607182418800017408L, -4498464794804257155L, 4607182418800017408L, -4493869261964912239L, 4607182418800017408L, -4495249985491541512L, 4607182418800017408L, -4501607480278071311L, 4607182418800017408L, 4715293965251657546L, 4607182418800017408L, 4724698721923438698L, 4607182418800017408L, 4726812765606133118L, 4607182418800017408L, -4494832518817408977L, 4607182418800017408L, 4729298949754641804L, 4607182418800017408L, 4728372408911211568L, 4607182418800017408L, -4525752678416587458L, 4607182418800017408L, 4724724831274960683L, 4607182418800017408L, -4495110649754717866L, 4607182418800017408L, 4729315490348836900L, 4607182418800017408L, -4493838081962091894L, 4607182418800017408L, 4724938682648170600L, 4607182418800017408L, 4723318175115991896L, 4607182418800017408L, 4709385678299830532L, 4607182418800017408L, 4727888825554340155L, 4607182418800017408L, -4497675943152307835L, 4607182418800017408L, 4729975109423287302L, 4607182418800017408L, 4722407840998503208L, 4607182418800017408L, -4501379380486485092L, 4607182418800017408L, -4495608771594706590L, 4607182418800017408L, -4498416265946661401L, 4607182418800017408L, 4726907411304310213L, 4607182418800017408L, -4503079461152886000L, 4607182418800017408L, -4497426524495551540L, 4607182418800017408L, -4496669617101270049L, 4607182418800017408L, -4494312123455014777L, 4607182418800017408L, 4723760174863718190L, 4607182418800017408L, 4723058950050585192L, 4607182418800017408L, -4501731803731864255L, 4607182418800017408L, -4493381566458592518L, 4607182418800017408L, 4716573312519425767L, 4607182418800017408L, 4722787315638015179L, 4607182418800017408L, 4728981109804570430L, 4607182418800017408L, -4494675368350780954L, 4607182418800017408L, 4724313063764946106L, 4607182418800017408L, 4708918699786166767L, 4607182418800017408L, 4722347486283001084L, 4607182418800017408L, -4511643428362952586L, 4607182418800017408L, -4502430609645587638L, 4607182418800017408L, -4500910425219908234L, 4607182418800017408L, -4498936155645736381L, 4607182418800017408L, -4500221175158952782L, 4607182418800017408L, 4727737016066925858L, 4607182418800017408L, 4727029525057232037L, 4607182418800017408L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e+08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Random
  private static final long generatedVar8 = 4845998362872549746L; // approx ~ 9.257527e+15
  private static final long[] generatedWeightedValues8 = {4713385523502522613L, 4608967063214761044L, 4718211017911287898L, 4601758817497516394L, -4510436887808089019L, 4607411795634549289L, 4703633611265388142L, 4603564735936222234L, -4503104855422849883L, 4609455726789806510L, -4512039026399867123L, 4608756791564734789L, -4506544592687652643L, 4603277224555390007L, 4709596051362743703L, 4607714260415050262L, 4729639691532909403L, 4606577228960169505L, 4729416010285731653L, 4604881751610213832L, 4729025127808004421L, 4602547546635595160L, 4729437263504642236L, 4609077089379815770L, -4536073643805637857L, 4599301639769858488L, -4494710882238530774L, 4605836985524226534L, 4728848127798522741L, 4603725428744422866L, 4716788870799775392L, 4607130043079568372L, 4715521004185609363L, 4599467664925625424L, -4493592201885589817L, 4607645438919804302L, 4700843484386899312L, 4600883111274754454L, -4504395411582851319L, 4600919379734481066L, 4698572781568271068L, 4608667478547062780L, 4727809679918863180L, 4607839659552684762L, 4727644406970369568L, 4607909851540508843L, -4504642164452507951L, 4608783926039041964L, 4724294305937520864L, 4609418952448132714L, -4495045193038353145L, 4609399879539950425L, 4728957486635533585L, 4600917425906188436L, 4724691353094859841L, 4608014918853822618L, -4499959595675349438L, 4606272319656681817L, 4728744696946390996L, 4600295406223338096L, 4705059763585838115L, 4607545194033286978L, 4729817548341561878L, 4604921091126683500L, 4729977522807448510L, 4604408480976662238L, -4501334599909159602L, 4606253858191205217L, 4728353001235937460L, 4610290919915991497L, 4724667335970435150L, 4604343843624226120L, 4725483510165923063L, 4609264912582072150L, 4710185166289500762L, 4599707332929043418L, 4728814789131712126L, 4607542652099238693L, -4494304741891477692L, 4607982262098957252L, -4495960873202645358L, 4604132241284007882L, -4507628994860678785L, 4610291510111408797L, -4505224623818258842L, 4609555540752799512L, 4724431951770514984L, 4610050237288747201L, 4713214212763751358L, 4605742134293630526L, -4495291990384791057L, 4608018155736635129L, 4726918190664599515L, 4607771071104986477L, 4729787771338112321L, 4602816054416644274L, -4496350583218070089L, 4603155002227040046L, -4494869217234778770L, 4602617402805923396L, -4494436785143506488L, 4600946146438252766L, -4498521414912286146L, 4607946372620547269L, -4494912372589512482L, 4605722193191096864L, -4494856350127378339L, 4604311847813435418L, -4502585084224343890L, 4609500406259460544L, 4701903121176464910L, 4609124564845529209L, -4493505866352070724L, 4603997085642016823L, -4494317912052056047L, 4608436588908803949L, 4729104268107376186L, 4599341174691191282L, -4493908858752412719L, 4607744017314545314L, 4728182593488049715L, 4602669358666998454L, 4723223451562602219L, 4608138912045277828L, 4730064751157109831L, 4603183532456047846L, -4495190820846763925L, 4608995472454278543L, -4495392958046447468L, 4609230780455479609L, -4496289589079172935L, 4608706029778614750L, -4498710514163786000L, 4610034618825340932L, -4508357555094868708L, 4605976016417786088L, 4714792509535454340L, 4602271975438954708L, -4496973066159747393L, 4599635915458206014L, 4726356896843503387L, 4601801288922613890L, -4498058834738999560L, 4609688817948601326L, -4500002837237579093L, 4607716168022615417L, -4503518987818713726L, 4606072458978628383L, -4514315299236374365L, 4609600803882601402L, 4728677473860646488L, 4608218776362161471L, -4501045225780611691L, 4608992808161631820L, 4728089171083649091L, 4606383431972966368L, -4498589188160481042L, 4609852623582774242L, -4513352707347735299L, 4608052085454862598L, 4722627661247084241L, 4603875114884018710L, -4494720464998575123L, 4608540078307179460L, 4726600876777330286L, 4600022057719725140L, 4719243654814539027L, 4603984179560797748L, -4501260415257851895L, 4599744326341444518L, -4503327357182512632L, 4604627686437860666L, 4729219604598047925L, 4605564179443696302L, 4711187689008149297L, 4607605672501492524L, 4717318014914402912L, 4607403581307276467L, 4727144441634294141L, 4609204655991810764L, 4699297430210420757L, 4603164353636349060L, -4497111230524935519L, 4604813802987833178L, 4708661606544406757L, 4607933489348982297L, -4500366704001892020L, 4605757500032459867L, -4494493755644407761L, 4607944046485063990L, 4727744651418057056L, 4608134384774402230L, -4498499994771173632L, 4609426020072971840L, 4723991777411576798L, 4609003497421982849L, -4494667627095927872L, 4609611523574982439L, 4727929201370447354L, 4605140536706343079L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e+08; values_ordering = Random
  //weights_mean = 1.500000e+01; weights_stdDev = 7.000000e+00; weights_ordering = Random
  private static final long generatedVar9 = 4846444118099410636L; // approx ~ 1.014904e+16
  private static final long[] generatedWeightedValues9 = {-4501827054114478480L, 4626368957387632414L, -4498790238130739780L, 4627340198811911145L, -4504380724099075169L, 4618112161774403850L, 4727920458177834578L, 4618423949531819862L, 4729112508835169395L, 4626248775309296640L, 4727423522788618230L, 4625959398825387816L, 4729537303807769731L, 4619553910618923810L, -4506529547291655944L, 4623833846962429886L, -4493771404207553876L, 4618916581444029948L, 4726718425441082164L, 4626138703037307317L, 4723502696902180187L, 4621899597609496538L, -4495011809856385741L, 4622895317659182352L, -4509029875319883107L, 4617162356873454744L, 4728209216867444169L, 4616821744645119852L, 4729129971273548013L, 4628214603309632512L, -4497212334874890821L, 4626840663800147175L, 4726829256288166947L, 4627509581063575615L, -4494155113389501477L, 4624987012063478337L, -4493777526206257886L, 4627445309688132517L, -4499371719988949116L, 4616652853687142392L, -4494464388910068295L, 4614433281920045348L, 4699085490309269175L, 4623597793862572314L, 4728298682722643184L, 4625351460586403316L, 4696693912564603281L, 4626463223650603090L, 4716386461791799706L, 4621896792780454642L, 4728799183385518122L, 4614222516164588224L, -4493347055365870467L, 4625749744364198735L, 4725621219742823480L, 4626431448087182402L, -4494344103176291720L, 4624949066508030168L, 4728228762117761357L, 4627283917688383448L, 4718411637419882183L, 4627246270063120982L, 4719740289218948525L, 4628163627215888116L, 4727267091757443039L, 4618956568258363418L, -4507227162729630562L, 4620727748141704842L, 4707852878769281537L, 4624948885193716849L, -4493390780189604886L, 4616520365419205284L, -4500144668220120354L, 4627853503311134412L, 4722825674467446418L, 4620035302536495733L, 4729505903060568358L, 4626273851689715985L, -4493732265857681405L, 4628009894185470373L, -4494207163058858582L, 4628215258724594674L, 4724059573541925741L, 4620808730081661436L, -4498593208789950622L, 4616232576284340612L, -4499466247139587334L, 4623058071916538283L, 4722522578487079357L, 4626435282245232388L, -4497826418715856168L, 4627397731947898250L, 4718063868828414168L, 4621432321471804993L, -4505257388697312668L, 4621252876105143297L, 4726629435635865851L, 4626470840841464471L, 4729350066523650555L, 4626687792091926390L, 4725212582438103417L, 4625942457181247402L, 4718805630429318930L, 4624647699020701647L, 4721478606434197210L, 4628214978481756828L, 4720416740309731593L, 4625112241910228174L, -4494317103922271723L, 4625134002457507258L, -4497484862686058120L, 4622339472950727584L, -4512620791838342932L, 4622785470865284423L, -4506953329658660592L, 4628159705625801813L, 4724000669078542544L, 4625235540268520645L, -4501033286765670026L, 4628045221090746935L, -4494813983639297853L, 4626896754849007462L, -4494487058325066719L, 4623625334342595702L, 4729464844147038251L, 4624711398621520367L, -4502344623746626801L, 4624912365284891234L, -4493588108268288009L, 4617066121026275996L, -4496910272634351999L, 4627299268628125450L, 4730051737217080512L, 4625300257024043094L, 4721335936362694143L, 4617744618935994666L, -4499963465768533981L, 4628095555875363117L, 4728584249341373859L, 4622012693303221271L, 4728525019722503937L, 4627353939356225691L, 4729724496484427394L, 4628038992624044144L, 4727684490210177084L, 4616294987945003986L, -4495475477499171692L, 4625993785440235484L, 4729788930932816701L, 4624311760442998365L, 4725893664821645813L, 4625216624728419551L, -4495044310355255353L, 4616824963145310508L, -4501084228055251923L, 4627881890123905046L, -4497976211968003156L, 4625134205637416500L, -4500543148835017350L, 4626121522794974288L, 4725413731964815228L, 4621079250027540738L, -4496065424285157161L, 4625781346428809516L, 4722423594378253866L, 4626691032135896795L, 4720546280284031271L, 4626213305333684797L, 4726235814389053820L, 4626779332817343454L, 4726824010042915040L, 4625093981587637066L, -4525530197896616029L, 4624983659984819329L, 4719927730504248097L, 4621152063240546337L, -4498574802625592811L, 4619480506330727676L, 4726992491356857824L, 4621683581687663994L, 4727153590453708884L, 4625962095757150160L, -4496837820104750629L, 4627706928787930950L, -4506352876611746827L, 4626207725146862334L, 4727990610886979268L, 4626264395287572245L, 4728089569928837311L, 4617828136393547752L, -4498269521756063168L, 4627019669892742152L, 4718240927449826814L, 4626110618427291710L, 4724815332590321837L, 4626982713240138878L, -4504596073355689234L, 4623112822648107050L, -4496273477839798219L, 4624823098424589595L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e-08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 0.000000e+00; weights_ordering = Random
  private static final long generatedVar10 = 4366306305509334870L; // approx ~ 8.408599e-17
  private static final long[] generatedWeightedValues10 = {-4734470680404634706L, 4607182418800017408L, 4470421850251685888L, 4607182418800017408L, -4735839770538920040L, 4607182418800017408L, -4741420583765536421L, 4607182418800017408L, -4742297585609213690L, 4607182418800017408L, -4738493286222727298L, 4607182418800017408L, -4733531223394231344L, 4607182418800017408L, -4731428303820524911L, 4607182418800017408L, -4733234807930234341L, 4607182418800017408L, -4729542985398075592L, 4607182418800017408L, -4741688533281633313L, 4607182418800017408L, -4738554068048185795L, 4607182418800017408L, -4732365815970712562L, 4607182418800017408L, -4732067974837085806L, 4607182418800017408L, 4468952519180008512L, 4607182418800017408L, -4734154553658941542L, 4607182418800017408L, -4739400531641392284L, 4607182418800017408L, 4480166308909675748L, 4607182418800017408L, -4731682532630403438L, 4607182418800017408L, -4731477935819315042L, 4607182418800017408L, 4482311875910341154L, 4607182418800017408L, -4743727277103360758L, 4607182418800017408L, -4731567277876783188L, 4607182418800017408L, -4731319079404763954L, 4607182418800017408L, -4748546021788478072L, 4607182418800017408L, 4480721208288947824L, 4607182418800017408L, -4732390603383770735L, 4607182418800017408L, -4731628356359234803L, 4607182418800017408L, -4731999895800650724L, 4607182418800017408L, -4735217744414882752L, 4607182418800017408L, -4736286265063847040L, 4607182418800017408L, -4733822315543990122L, 4607182418800017408L, -4738750516297841812L, 4607182418800017408L, -4746344185188802632L, 4607182418800017408L, -4735387870146781620L, 4607182418800017408L, -4730786159296083420L, 4607182418800017408L, -4736667636450203178L, 4607182418800017408L, -4739614030893726785L, 4607182418800017408L, -4731181816433321522L, 4607182418800017408L, -4743565746215771436L, 4607182418800017408L, -4735863682497362253L, 4607182418800017408L, -4734936708568080788L, 4607182418800017408L, -4751415884392227984L, 4607182418800017408L, -4731790350103800050L, 4607182418800017408L, -4731663414794234656L, 4607182418800017408L, 4479768676209473192L, 4607182418800017408L, -4767399404796903296L, 4607182418800017408L, -4748652898587532264L, 4607182418800017408L, -4731037533310083016L, 4607182418800017408L, 4484470428174576836L, 4607182418800017408L, 4474870180312055352L, 4607182418800017408L, -4732833690983241863L, 4607182418800017408L, -4735670910079618201L, 4607182418800017408L, -4736202934210237459L, 4607182418800017408L, -4736604852591279957L, 4607182418800017408L, -4746203299657626076L, 4607182418800017408L, -4734666139891573756L, 4607182418800017408L, -4729637485203068769L, 4607182418800017408L, -4730114711158138216L, 4607182418800017408L, 4484682922185399528L, 4607182418800017408L, -4737342877206727193L, 4607182418800017408L, -4741919642492234792L, 4607182418800017408L, 4485224462538088444L, 4607182418800017408L, -4736201107387176024L, 4607182418800017408L, -4736226841563101172L, 4607182418800017408L, 4463648199704893536L, 4607182418800017408L, -4732022207288347022L, 4607182418800017408L, 4481124870227549162L, 4607182418800017408L, -4735703360203571097L, 4607182418800017408L, -4735083811731209636L, 4607182418800017408L, -4730990940627757486L, 4607182418800017408L, -4741645909871546275L, 4607182418800017408L, 4470505742642918144L, 4607182418800017408L, 4481528359695945080L, 4607182418800017408L, -4732193001868211272L, 4607182418800017408L, -4732132381294282214L, 4607182418800017408L, -4734600685195865629L, 4607182418800017408L, -4734989568457807184L, 4607182418800017408L, 4481244596866278768L, 4607182418800017408L, -4743139837418128704L, 4607182418800017408L, -4737405404523005272L, 4607182418800017408L, -4730006985618266343L, 4607182418800017408L, -4731356894767046668L, 4607182418800017408L, 4476581988999620376L, 4607182418800017408L, 4482870572081512124L, 4607182418800017408L, -4746613690997047588L, 4607182418800017408L, -4732949734311926670L, 4607182418800017408L, -4733477637778226638L, 4607182418800017408L, 4480467639219926028L, 4607182418800017408L, 4466644309210762688L, 4607182418800017408L, -4734954540609170404L, 4607182418800017408L, -4732582867747677936L, 4607182418800017408L, -4740140746724170120L, 4607182418800017408L, -4739052280109910586L, 4607182418800017408L, -4746810785036588128L, 4607182418800017408L, -4730990753244214000L, 4607182418800017408L, -4737495976981546596L, 4607182418800017408L, -4731204776447341409L, 4607182418800017408L, 4471561436269231888L, 4607182418800017408L, -4734554522650514217L, 4607182418800017408L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e-08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Random
  private static final long generatedVar11 = 4368048854900483305L; // approx ~ 1.055646e-16
  private static final long[] generatedWeightedValues11 = {-4748272242485446040L, 4607267569512654533L, -4732489666166860638L, 4606910796826798325L, -4732105378533347622L, 4604977292629317518L, -4737662374907954758L, 4606609735562023923L, -4731727301945234953L, 4603484673305666809L, -4731398331517472488L, 4609934129226065800L, -4746026410513310592L, 4605308311416711278L, 4483104349889490836L, 4609135152037319976L, -4757676981042362400L, 4610124263388674430L, -4738470004378752285L, 4602789486788585666L, -4738594479392728937L, 4607192181432454793L, -4731656951371842832L, 4608226588085334127L, -4731560227815568490L, 4609971415572181825L, -4731919767639128971L, 4606263444768024390L, -4744381470610909134L, 4599430466951886390L, -4745022936590761554L, 4606800929187158910L, -4737143164460571818L, 4607296471412641646L, -4731898002549979762L, 4610039422327319996L, -4731755535863011642L, 4608896493350607505L, -4738543339934250864L, 4609189166278793828L, -4734521827753864535L, 4606916248227234223L, -4735852116304946699L, 4607038584720438081L, -4737489921557973110L, 4603410313605153790L, -4744377235072322698L, 4602919951439546595L, -4735830148126441514L, 4602441205902023586L, 4484517348216051076L, 4603683242288907742L, -4749440539031954800L, 4603936285298417136L, 4482837119959427108L, 4605178755096263080L, -4731642708198254275L, 4599841985558185446L, -4730283580562202996L, 4602161888439683438L, -4734118333357250496L, 4605229938231837500L, -4729887154804753135L, 4610214525957427308L, 4484570474558627196L, 4606697111111402605L, -4730374381299739574L, 4607763524426065444L, 4484562669801526092L, 4607783714722579014L, 4476204361204022064L, 4605302511515986044L, -4731337057837262912L, 4609917617822738756L, -4735059065132621614L, 4604696727099866120L, 4463919633950362848L, 4607063636783145595L, -4732696546059535462L, 4609576506127094390L, -4732138313914443893L, 4605288378939168566L, -4746466164559596744L, 4609776052816141492L, -4732459061112145960L, 4601416933284442232L, -4729808189098266582L, 4608134549239968216L, -4735237107479785180L, 4610075840372676842L, -4731915956843284453L, 4605268097847853214L, -4748578402217016608L, 4605219350081341842L, -4730799028716068294L, 4609054017315441872L, -4732480166729960609L, 4604227838162760639L, -4740763639956328491L, 4605191972693485978L, -4732295247912898866L, 4599361496349435124L, 4465255825460001920L, 4608072082956158663L, -4733153410762913546L, 4609246221031541789L, -4729799527530929056L, 4606862359068734376L, -4731604257242624777L, 4605647044891220054L, -4730623187107594083L, 4605326633505392747L, -4740533032587748222L, 4603083777493102500L, -4735334610053057759L, 4609074798043820645L, -4738345475687415688L, 4605854202902751346L, -4730000815775914806L, 4607986671819063857L, -4731964272089467072L, 4602191744420804916L, 4485001587814772912L, 4605484008734806642L, -4729947914127519884L, 4603519697791333258L, 4478215200443784140L, 4608741476327763240L, -4736870797842118256L, 4604971936634463392L, -4730988845206054202L, 4608779501804444957L, -4736643231236794217L, 4606602889956813423L, -4732326867763562622L, 4609949189152611538L, 4476972022096625180L, 4602679155465817731L, -4730728099537201394L, 4603403473427116498L, -4732479737338524846L, 4605419565821100798L, -4731797742860403528L, 4608966156331732080L, -4740743372376139625L, 4600513136593714726L, -4737737190017773955L, 4599997219635816060L, -4732106016401995788L, 4607858376855034288L, -4737520277367885687L, 4607277878620551290L, 4479487386858864644L, 4609279018748170959L, 4485398979391111492L, 4599779699183735566L, -4738514893419233090L, 4605204145012528697L, -4751456834189873824L, 4599869922176195544L, 4474051532325312560L, 4608374442193853948L, -4730427661894577901L, 4605172979271548819L, -4730539878013166235L, 4604899192673666768L, 4485060990009405988L, 4607129684018750216L, -4733078954637966948L, 4609795691261792454L, -4731879467719866610L, 4599403928417045988L, -4731928117217663522L, 4606401130078442687L, -4738447552013801716L, 4600716416567064322L, -4733201758940416598L, 4600038505157926436L, -4731863597876755595L, 4605770238643385333L, -4730590439326651300L, 4609542543019347004L, -4741273448173436958L, 4607227067537754629L, -4737402486951983224L, 4605069311133407810L, -4737256377658971080L, 4609093408795724774L, 4479653418001283180L, 4608742570718772635L, -4738144968124795528L, 4601615689239303260L, 4471296705327559952L, 4605295260060146874L, -4730252185730931637L, 4605545121118163029L, -4729812538635804754L, 4607677059437574634L, -4730212371920278772L, 4607206498399352270L};

  //Generated values with parameters:
  //Count = 100
  //values_mean = -1.000000e-08; values_stdDev = 1.000000e-08; values_ordering = Random
  //weights_mean = 1.500000e+01; weights_stdDev = 7.000000e+00; weights_ordering = Random
  private static final long generatedVar12 = 4367339858964729950L; // approx ~ 9.682552e-17
  private static final long[] generatedWeightedValues12 = {4475380041062980480L, 4624192845509667953L, -4731694657061295628L, 4627833033437992878L, -4740394882451618560L, 4627159440107616827L, -4734750608881492599L, 4619957384730289040L, 4481663716025306382L, 4624751652003471943L, -4744861609719666580L, 4625427642828507359L, -4732413128867167655L, 4621456214221964318L, -4730869166019130686L, 4626238840924820069L, -4732157618362898750L, 4627452146834169489L, 4483397530383288264L, 4623212649214895045L, 4468650756158429312L, 4626412915663261402L, 4477705891415163152L, 4626434750038904502L, -4752110088666799648L, 4616375256979614290L, -4732122361182877934L, 4623970947594337184L, -4739429224362007666L, 4625847609896911932L, 4473303361089029304L, 4627084867527351883L, -4732728352493713976L, 4616017563869312284L, -4729890950882994951L, 4616971857998364656L, -4733241559653952325L, 4627939647559812550L, -4740013593632745526L, 4625347685491024632L, -4734910562453121108L, 4621439539639514636L, -4732502289843927718L, 4621738710826610089L, -4731050287330572036L, 4615827416560191204L, -4731089162749331742L, 4622374762080088750L, -4734463440628722059L, 4627380367794392002L, 4483084110782989688L, 4627991989954442468L, 4483809232187463032L, 4626660260694948782L, -4737816784667809023L, 4627949085052089670L, -4743005847764184716L, 4623214860471887462L, -4738293914140715316L, 4620018832727201353L, 4484887524897204592L, 4627086533874003478L, -4737602169232844941L, 4627543572335017822L, -4737990994109582800L, 4622158544248749408L, -4732759166397284020L, 4626883101525507040L, 4484305524584703432L, 4626138654625634786L, -4732151364031419184L, 4621589543308412194L, -4731446009957510728L, 4620744764535727406L, -4731605408097938570L, 4627405667669990614L, -4741991445342458218L, 4626629693195967389L, -4732144808895369814L, 4621386903811592980L, 4478463045779254232L, 4620254372295755723L, -4730800774542215086L, 4626047619545973657L, -4734573635688952121L, 4626552586156341692L, -4738516336134642644L, 4626768607398336721L, -4730803930967337678L, 4617167843494518956L, 4482681177282887640L, 4619485471606629224L, -4739887723205307361L, 4626692142563585316L, -4741353330308000532L, 4617212063448206368L, 4481962086063282290L, 4626238161707687105L, -4731941842679005971L, 4627451650790171529L, -4737451737009156016L, 4619007742414850566L, -4740851041074294546L, 4622001197368316120L, 4475080383463770752L, 4620353561012886721L, -4742729602112209566L, 4621668596837641196L, -4729833826943297896L, 4625904156751038010L, -4736946387325181041L, 4622562937092209404L, -4731970204621943454L, 4623437009729531592L, -4740256298359531567L, 4627211033940266046L, -4741647759393910359L, 4627778041984086559L, 4482828501280999176L, 4623877475487963698L, -4736096013101598007L, 4621913539491507764L, -4739023773702434982L, 4626863235025785371L, -4759789603654218048L, 4626881148496416652L, -4731269636999858749L, 4625395912507896348L, -4745796623958127328L, 4627829853901995768L, -4731374338616973030L, 4625098432320519353L, -4730630769467007788L, 4625782804541849292L, -4743741646282833966L, 4625406525211539646L, -4742350421858829198L, 4627695999791355399L, -4731191682658851884L, 4624237696253991336L, -4733619598366387006L, 4625180830969137487L, -4737479592008780281L, 4621106732752958286L, -4730504369038892532L, 4622791870111003432L, -4747542347508857096L, 4622457002048802846L, -4731207403269208838L, 4627247185185150374L, -4740655851774052142L, 4624705486554974981L, 4478483890363296284L, 4624437965123033994L, -4739375647871449778L, 4626784203668656612L, -4734955947891966760L, 4627175191066047278L, -4733102721497598082L, 4617019262730837584L, -4736862149380473147L, 4624603971155709634L, -4733774468849340484L, 4624600070127764582L, 4474450880154363768L, 4626275664197206092L, 4485050196667168004L, 4625662880194794506L, -4738457663692036868L, 4624231288965420822L, -4736117302450810223L, 4620528836118513735L, 4480980418615901096L, 4622909920176533312L, -4729846170699655378L, 4627069147183051676L, -4729776599407986311L, 4623198110976226082L, 4483432942302445484L, 4627739129733551294L, 4476421689612987752L, 4627578560181757677L, -4732701942722396635L, 4620901975526437290L, -4732202592358589740L, 4623598725945299993L, -4736242439320583566L, 4626297377560595109L, -4732283779544056895L, 4623768738547336915L, -4730603737156837420L, 4624314455637657250L, -4731252694525474117L, 4626467626515169601L, -4739588661764389525L, 4614483536343590860L, -4735497316437732373L, 4625364868639602218L, 4476779488488290164L, 4622411275073079269L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Sorted
  private static final long generatedVar13 = 4844821157515779861L; // approx ~ 7.955158e+15
  private static final long[] generatedWeightedValues13 = {-4498883019239542574L, 4599762071914296286L, -4498978555365506826L, 4600471180802359400L, -4499586097762367668L, 4600994883538193028L, -4499932489866951500L, 4601042100249543730L, -4500880364156292796L, 4601063545649384718L, -4501421802722738900L, 4601158376237996866L, -4502145465288932188L, 4601205663605296758L, -4511097297484762640L, 4601957950187486994L, -4520378742112579808L, 4602183439196657190L, 4708893181266563072L, 4603043530776003301L, 4715813587437350832L, 4603274038760662486L, 4716005464232775056L, 4603404135253672582L, 4716621512141638924L, 4603478146684436612L, 4717648384386760780L, 4603908297692232348L, 4720603547868885092L, 4604002771957090813L, 4721142407424888830L, 4604436322021761076L, 4722321182459063778L, 4604564530648932766L, 4723272673892031971L, 4604872025051496897L, 4723890434465216637L, 4605034737009704045L, 4724424887076874158L, 4605222711276922795L, 4725405509413910078L, 4605506758048250429L, 4725595770504557978L, 4605568711544604524L, 4725739847903259004L, 4605754550686160830L, 4725910168916213754L, 4606032623757571525L, 4727292969949122528L, 4606848563275000891L, 4727349014234537856L, 4606977984451851891L, 4727446905989801411L, 4607234278168981998L, 4727734364345937736L, 4607410111950839904L, 4728233739143318164L, 4607449944781359413L, 4728373152937783292L, 4607645085601368594L, 4729244259768555394L, 4607662796312866981L, 4729338178809021442L, 4608144351998681186L, 4729394871380088624L, 4608208949100379322L, 4729707466896960095L, 4608275525406731479L, 4730000890845347572L, 4608326807193950882L, 4730776071962740582L, 4608367158298019440L, 4730871689098819520L, 4608621334400358825L, 4730919583469013786L, 4608672867606178141L, 4731200279567444494L, 4608712366267430892L, 4731205898937935294L, 4608973234598411704L, 4731227134797483984L, 4608983961907626248L, 4731547718224341562L, 4609299431601842318L, 4731901758426793670L, 4609444234982760226L, 4731956451424512704L, 4609468273954926363L, 4732044926608962711L, 4609625931089400024L, 4732160967566632302L, 4609668864071152252L, 4732176778841544026L, 4609743295468240366L, 4732230951653308884L, 4609849450018886979L, 4732736838842077684L, 4609879786158557908L, 4732864178942078116L, 4610086511701437336L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = -1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Sorted
  private static final long generatedVar14 = 4369145203569842926L; // approx ~ 1.271339e-16
  private static final long[] generatedWeightedValues14 = {-4496888740970495999L, 4600136333207654516L, -4496888740970495999L, 4600347555785356556L, -4496888740970495999L, 4602691292181123420L, -4496888740970495999L, 4602752911082273434L, -4496888740970495999L, 4602807069850087895L, -4496888740970495999L, 4602886910490872448L, -4496888740970495999L, 4603262955852143357L, -4496888740970495999L, 4604211562645496686L, -4496888740970495999L, 4604363060764191210L, -4496888740970495999L, 4604644228617891693L, -4496888740970495999L, 4604917066377702641L, -4496888740970495999L, 4604938060219553774L, -4496888740970495999L, 4605493466927389444L, -4496888740970495999L, 4605535030741080916L, -4496888740970495999L, 4605872756864297847L, -4496888740970495999L, 4605942164981480098L, -4496888740970495999L, 4606063200494148852L, -4496888740970496000L, 4606101161147798462L, -4496888740970496000L, 4606431815276823945L, -4496888740970496000L, 4606786456493093484L, -4496888740970496000L, 4606858273547565016L, -4496888740970496000L, 4607021019500636987L, -4496888740970496000L, 4607183544993522364L, -4496888740970496000L, 4607213873487445713L, -4496888740970496000L, 4607217372610357607L, -4496888740970496000L, 4607303065325460394L, -4496888740970496000L, 4607536801697501400L, -4496888740970496000L, 4607647193108114131L, -4496888740970496000L, 4607697821775953564L, -4496888740970496000L, 4607884902991884841L, -4496888740970496000L, 4607913725708211436L, -4496888740970496000L, 4607921043644349743L, -4496888740970496000L, 4607989497777089676L, -4496888740970496000L, 4608059402356565062L, -4496888740970496000L, 4608128615605056847L, -4496888740970496000L, 4608319478587850079L, -4496888740970496000L, 4608495272979479798L, -4496888740970496001L, 4608887230000504792L, -4496888740970496001L, 4609049095358931370L, -4496888740970496001L, 4609139614700986211L, -4496888740970496001L, 4609208594905773342L, -4496888740970496001L, 4609332265322190362L, -4496888740970496001L, 4609688909012684870L, -4496888740970496001L, 4609885056444510054L, -4496888740970496001L, 4609927463338719716L, -4496888740970496001L, 4609929577315829826L, -4496888740970496001L, 4609930717570924316L, -4496888740970496001L, 4609955330741397468L, -4496888740970496001L, 4609958009635036394L, -4496888740970496001L, 4609999955213756326L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Reverse Sorted
  private static final long generatedVar15 = 4846525634760287623L; // approx ~ 1.031207e+16
  private static final long[] generatedWeightedValues15 = {-4498724717905826950L, 4610231149223484458L, -4498873186599743426L, 4610199253416945813L, -4500156193889798808L, 4610184461267133136L, -4500197995497711192L, 4610158471636300224L, -4500488628278123652L, 4610151534809649957L, -4501347315239159832L, 4610137368255168637L, -4501659816142108040L, 4609970686694132645L, -4501668843745720440L, 4609785923821508556L, -4501891047775573396L, 4609777098017713944L, -4506292013854532276L, 4609547115613693506L, -4510280007178061024L, 4609526661846758148L, 4703809685822988320L, 4609502490578148483L, 4711231740197478208L, 4609193431441541474L, 4712704472965687224L, 4609131560917836541L, 4715850014453468680L, 4609025880730286900L, 4717798080369273496L, 4608954236011396626L, 4718607186132186664L, 4608746620718135241L, 4720158891009281126L, 4608648890006967644L, 4720482630903176799L, 4608629161168977554L, 4723363081155274507L, 4608464156698545085L, 4724236563303935974L, 4608363360753217056L, 4725731730499824383L, 4608310690417446494L, 4726073916071323714L, 4608255980783072883L, 4727009695106922353L, 4607717739594307970L, 4727134288230401644L, 4607651234833308655L, 4727395796000997277L, 4607382449621708854L, 4727934678211540875L, 4607367200521007707L, 4728581508218413222L, 4607295964010947507L, 4729012718048978167L, 4607038717616771341L, 4729064655717120497L, 4607023829743318587L, 4729158737645173178L, 4606596014963602066L, 4729173851315647256L, 4606353538685492988L, 4729293691710713416L, 4605993908393294310L, 4729983522699911066L, 4605967451835749774L, 4730062739943786584L, 4605854539361707269L, 4730354079348301742L, 4605802471749361477L, 4730376385578524455L, 4605696841298632305L, 4730633545610187340L, 4605534832523331589L, 4730646009715410126L, 4605371850970393947L, 4731316803439917395L, 4605230056600763558L, 4731529732748967230L, 4604699538859415954L, 4731714681119033521L, 4604691392080454986L, 4731856220174935388L, 4604167685234204470L, 4732021069082141418L, 4604085454102898610L, 4732163090315594040L, 4604038340117360974L, 4732389141592254963L, 4602966033532596513L, 4732519017311623742L, 4601963324146402374L, 4732535978859612041L, 4601738423981440292L, 4733327718332130292L, 4601314556968065828L, 4733352781297470850L, 4599818981672537218L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = -1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Reverse Sorted
  private static final long generatedVar16 = 4368434882069979115L; // approx ~ 1.103227e-16
  private static final long[] generatedWeightedValues16 = {-4496888740970495999L, 4610041747259297550L, -4496888740970495999L, 4610019060282025792L, -4496888740970495999L, 4610013382394183768L, -4496888740970495999L, 4609920251904726550L, -4496888740970495999L, 4609712533146400942L, -4496888740970495999L, 4609700512626245687L, -4496888740970495999L, 4609651246693139426L, -4496888740970495999L, 4609553372533567575L, -4496888740970495999L, 4609497713991681662L, -4496888740970495999L, 4609315422314239198L, -4496888740970495999L, 4609032775427501254L, -4496888740970495999L, 4608692751974420262L, -4496888740970496000L, 4608415460290654302L, -4496888740970496000L, 4608163259974913671L, -4496888740970496000L, 4608144851881721605L, -4496888740970496000L, 4608024555763581957L, -4496888740970496000L, 4607840834411286842L, -4496888740970496000L, 4607732040894599399L, -4496888740970496000L, 4607662480825968236L, -4496888740970496000L, 4607231340087721699L, -4496888740970496000L, 4607212565751311570L, -4496888740970496000L, 4607021400436026942L, -4496888740970496000L, 4606492664047508261L, -4496888740970496000L, 4606432054960742174L, -4496888740970496000L, 4606383284317416951L, -4496888740970496000L, 4606237052941692857L, -4496888740970496000L, 4605879257723952905L, -4496888740970496000L, 4605602074550391608L, -4496888740970496000L, 4605279364297824812L, -4496888740970496000L, 4604984586038122372L, -4496888740970496000L, 4604480852952101936L, -4496888740970496000L, 4604431629519150300L, -4496888740970496000L, 4604216819244122986L, -4496888740970496000L, 4604212788928430620L, -4496888740970496000L, 4603932888525994372L, -4496888740970496001L, 4603566552711005113L, -4496888740970496001L, 4603341746900074290L, -4496888740970496001L, 4603242464992983609L, -4496888740970496001L, 4603193794726693614L, -4496888740970496001L, 4603182272870035231L, -4496888740970496001L, 4602885083496942426L, -4496888740970496001L, 4602702576531913483L, -4496888740970496001L, 4602487111808346590L, -4496888740970496001L, 4602394564023843662L, -4496888740970496001L, 4602147690820891998L, -4496888740970496001L, 4600855545153271318L, -4496888740970496001L, 4600706650820285908L, -4496888740970496001L, 4600697203032838624L, -4496888740970496001L, 4600183272625108486L, -4496888740970496001L, 4599326087327807190L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Reverse Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Sorted
  private static final long generatedVar17 = 4845990686280770488L; // approx ~ 9.242174e+15
  private static final long[] generatedWeightedValues17 = {4733348455146219176L, 4600017354668893968L, 4733302569523637416L, 4601017568369491810L, 4732883582143443500L, 4601077293157283378L, 4732315848589630716L, 4601793587724131200L, 4732259464080565824L, 4601799468563441854L, 4731774802681219744L, 4601952792582303800L, 4731768225259534704L, 4602250854831685962L, 4731384339978885316L, 4602261372852218244L, 4731272256137843319L, 4603490217062266874L, 4730782021947291157L, 4603666427339162087L, 4730675345121087404L, 4603730109284838850L, 4730468417874014777L, 4604081392621969154L, 4730425486376885315L, 4604229510433550975L, 4730007600383355488L, 4604305952540801210L, 4729920389279247644L, 4604496525676086672L, 4729529988745622006L, 4605498045099555342L, 4729299217320798658L, 4605612698510040284L, 4729250262785220614L, 4605648631257015219L, 4729108458247597228L, 4605665295402139496L, 4728232252985153804L, 4606065125280279852L, 4728133079396658540L, 4607272576797741966L, 4728053220062059061L, 4607321272257106179L, 4727726477237671800L, 4607604343013716200L, 4727705441006940588L, 4607691679234223049L, 4727453546667923126L, 4607743978412464311L, 4727333156209802774L, 4607810785659365952L, 4727135606254335036L, 4607895112713196719L, 4725056171769112280L, 4608190464928552531L, 4724871715779098942L, 4608194236834648990L, 4723901460274577610L, 4608262266548711450L, 4723261808493760874L, 4608326975442308087L, 4722560269754673271L, 4608417523219037205L, 4721153375940423854L, 4608649373953443190L, 4717738876147666840L, 4609160379253121273L, 4716289641047958688L, 4609319609890895967L, -4529852164277123584L, 4609473329169356416L, -4506644609985765924L, 4609552748323017080L, -4505483756403146456L, 4609680569205372845L, -4504482233492284172L, 4609683330555393540L, -4503416039875140328L, 4609774223465092493L, -4503080534638873756L, 4609783952096385446L, -4502800801801694876L, 4609809190386015713L, -4502285848118748960L, 4609841584046007144L, -4502278277593549152L, 4609907118647032228L, -4502262408532601236L, 4609929212392464158L, -4501888743237089060L, 4609996813539230966L, -4500959835545757448L, 4610161296427433527L, -4499939399959270976L, 4610161984715654808L, -4499699752207236660L, 4610175349868810125L, -4499560514574980988L, 4610254240675203604L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = -1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Reverse Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Sorted
  private static final long generatedVar18 = 4369265738749491374L; // approx ~ 1.301053e-16
  private static final long[] generatedWeightedValues18 = {-4496888740970496001L, 4599777922561164184L, -4496888740970496001L, 4600126623314876738L, -4496888740970496001L, 4600510552783539902L, -4496888740970496001L, 4601581132532965328L, -4496888740970496001L, 4601934977550306712L, -4496888740970496001L, 4603203216132367912L, -4496888740970496001L, 4603250657597095300L, -4496888740970496001L, 4603698983223634659L, -4496888740970496001L, 4603851895651959710L, -4496888740970496001L, 4603865259065979381L, -4496888740970496001L, 4603872609787085912L, -4496888740970496001L, 4604056889360093068L, -4496888740970496001L, 4604256880570954190L, -4496888740970496001L, 4604339461735098907L, -4496888740970496001L, 4604452439054755964L, -4496888740970496001L, 4604507647540100234L, -4496888740970496001L, 4605165015216691073L, -4496888740970496001L, 4605271156329063806L, -4496888740970496001L, 4605422538400089988L, -4496888740970496001L, 4605807337446960386L, -4496888740970496001L, 4606224359311284830L, -4496888740970496000L, 4606304606754865563L, -4496888740970496000L, 4606459059395121652L, -4496888740970496000L, 4606499411264680201L, -4496888740970496000L, 4606590751163640987L, -4496888740970496000L, 4606737361832730358L, -4496888740970496000L, 4607086007005073365L, -4496888740970496000L, 4607305478478654703L, -4496888740970496000L, 4607485975191788571L, -4496888740970496000L, 4607672295387921342L, -4496888740970496000L, 4607825727604131836L, -4496888740970496000L, 4607876590712989821L, -4496888740970496000L, 4607894164388571044L, -4496888740970496000L, 4608007395018589809L, -4496888740970496000L, 4608030859297328066L, -4496888740970496000L, 4608033310587552645L, -4496888740970496000L, 4608287166342518595L, -4496888740970496000L, 4608395630777749294L, -4496888740970496000L, 4608547514461062746L, -4496888740970496000L, 4608573917987692474L, -4496888740970495999L, 4609004815687260454L, -4496888740970495999L, 4609322263832902347L, -4496888740970495999L, 4609335340644829321L, -4496888740970495999L, 4609378582119512454L, -4496888740970495999L, 4609410285946745534L, -4496888740970495999L, 4609502969673886554L, -4496888740970495999L, 4609506069959105358L, -4496888740970495999L, 4610134729103016212L, -4496888740970495999L, 4610138224941819353L, -4496888740970495999L, 4610302033526263826L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Reverse Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Reverse Sorted
  private static final long generatedVar19 = 4843847101279961366L; // approx ~ 6.981101e+15
  private static final long[] generatedWeightedValues19 = {4733215792649303372L, 4610224878390929378L, 4733165869683471121L, 4610201498364516272L, 4733094991764008129L, 4610108809015856854L, 4732914097338246530L, 4609727071115227777L, 4732877984296468086L, 4609658481090494248L, 4732766235258132259L, 4609537288946441173L, 4732576204255661903L, 4609436430014193003L, 4732471032131935976L, 4609230219512695754L, 4732213770897531367L, 4609197161463249773L, 4731917155466307932L, 4609092209083273137L, 4731292026975711474L, 4609008628811607516L, 4731284249878599229L, 4608830839458365535L, 4731276054643276614L, 4608658829502841489L, 4731261710057371500L, 4608658504911803171L, 4731161410372359760L, 4608607988086432220L, 4731147263578562654L, 4608577365543460497L, 4730993721144584654L, 4608195579659972043L, 4730749883535450687L, 4608117532610008854L, 4730671921612210793L, 4608057073070176976L, 4730013781226596513L, 4608052263935331188L, 4729867778585458285L, 4608001608807938766L, 4729752690734914667L, 4607961884150647803L, 4729417125066863256L, 4607925613827794822L, 4729265549233780001L, 4607701915543963122L, 4729207225402473849L, 4607460095696593120L, 4728926264213373048L, 4607409313757963092L, 4728883080451689830L, 4607311157751531638L, 4728624971395744793L, 4607217331567065423L, 4728483277609315644L, 4607207310922250450L, 4728269208816837524L, 4606439883098350006L, 4728202230238991670L, 4606326054389354065L, 4727885790583467661L, 4605780550987240539L, 4725018313348215212L, 4605580159885207848L, 4724336614940255546L, 4605281979274535974L, 4723316643791900070L, 4604791401871683064L, 4723265257900991807L, 4604717791961875529L, 4722838874849298104L, 4604593789217505512L, 4722334574482716152L, 4604561423978787458L, 4722317850176138487L, 4604205622711843228L, 4720170707814683220L, 4604059441452027018L, 4718785865115108200L, 4603826220532592226L, 4709320022365802032L, 4603629336823222840L, 4708470759292251920L, 4603293687254260589L, 4708147462600116736L, 4602950866130187844L, 4705598169737367840L, 4602886583250126597L, -4514740532362205024L, 4602005478281236118L, -4513691232687468704L, 4600976756229478912L, -4507609806185082840L, 4600930274644037850L, -4504920597314939140L, 4600107532071144210L, -4499256415221439240L, 4599842209439517370L};

  //Generated values with parameters:
  //Count = 50
  //values_mean = -1.000000e+08; values_stdDev = 1.000000e-08; values_ordering = Reverse Sorted
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Reverse Sorted
  private static final long generatedVar20 = 4368380259183964524L; // approx ~ 1.096494e-16
  private static final long[] generatedWeightedValues20 = {-4496888740970496001L, 4610200098788707190L, -4496888740970496001L, 4609976622176319806L, -4496888740970496001L, 4609618669619515097L, -4496888740970496001L, 4609604352942819337L, -4496888740970496001L, 4609530938290044310L, -4496888740970496001L, 4609421126416208718L, -4496888740970496001L, 4609301711316637745L, -4496888740970496001L, 4609265824218981038L, -4496888740970496001L, 4609241955693705163L, -4496888740970496001L, 4609153135437953844L, -4496888740970496001L, 4609121991161261205L, -4496888740970496001L, 4609058353723179804L, -4496888740970496001L, 4608985498580364081L, -4496888740970496000L, 4608843152476958607L, -4496888740970496000L, 4608508293355138106L, -4496888740970496000L, 4608225731625900906L, -4496888740970496000L, 4608205811448917528L, -4496888740970496000L, 4608163837845676971L, -4496888740970496000L, 4608121226599152495L, -4496888740970496000L, 4607915689650830052L, -4496888740970496000L, 4607822192079745658L, -4496888740970496000L, 4607319950483567705L, -4496888740970496000L, 4607263669842505493L, -4496888740970496000L, 4607241728035266015L, -4496888740970496000L, 4605978210160816381L, -4496888740970496000L, 4605659679377328860L, -4496888740970496000L, 4605319899537762158L, -4496888740970496000L, 4605250888497060912L, -4496888740970496000L, 4605103386003520370L, -4496888740970496000L, 4604184386162901333L, -4496888740970496000L, 4604175709952800680L, -4496888740970496000L, 4604076555759513422L, -4496888740970496000L, 4603972970573577872L, -4496888740970496000L, 4603668112076900909L, -4496888740970496000L, 4603521724087036032L, -4496888740970495999L, 4603424999353625099L, -4496888740970495999L, 4603099374337307708L, -4496888740970495999L, 4602856024416086724L, -4496888740970495999L, 4602810042119298791L, -4496888740970495999L, 4602427680883279282L, -4496888740970495999L, 4602361781612924878L, -4496888740970495999L, 4602304554010379948L, -4496888740970495999L, 4601987698389818698L, -4496888740970495999L, 4601361409322333814L, -4496888740970495999L, 4601315095927541240L, -4496888740970495999L, 4600901418182846126L, -4496888740970495999L, 4600086769613710690L, -4496888740970495999L, 4599961967104683538L, -4496888740970495999L, 4599933982999727822L, -4496888740970495999L, 4599481050556144784L};

  //Generated values with parameters:
  //Count = 1000
  //values_mean = 1.000000e-08; values_stdDev = 1.000000e-08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Random
  private static final long generatedVar21 = 4367598905343741854L; // approx ~ 1.000185e-16
  private static final long[] generatedWeightedValues21 = {4488743049630693075L, 4607305823710211642L, 4487307971345557119L, 4603764078621426908L, 4472999674452488864L, 4603967553946901814L, 4493355961375496259L, 4609480740024975842L, 4478026818376583444L, 4607132556956522785L, -4741659785428333988L, 4604544034270670415L, -4738722752356417552L, 4605645569635190680L, 4478048396676643420L, 4604668997042941618L, 4491687339141284124L, 4599980822795187918L, 4491770345591021836L, 4608895770810305771L, -4751589381616707376L, 4602734657239699242L, 4490542209299119608L, 4607666568664347004L, 4481126682518037156L, 4607182862365836782L, -4738496870679219504L, 4607597322997810737L, -4755031709578428320L, 4604305896360070330L, -4759298801420947968L, 4609623203270158163L, 4485132917444910997L, 4603882924777621758L, 4486685266305339575L, 4609583063890547983L, 4489320526233086154L, 4609787581394586544L, 4492183240762524353L, 4609990370705824941L, 4482726050685279740L, 4609096272253787504L, 4485867317526776879L, 4601389052670907274L, 4486853070828314495L, 4608141449438137752L, 4484632217600691948L, 4603250544068940552L, 4485281785489558152L, 4606264530522934428L, 4493364832896911870L, 4604559691025231924L, 4486375014185643043L, 4603446141472184288L, 4493816994226388282L, 4602950280677170114L, 4482946805070716772L, 4601549386809758422L, -4740657916813903420L, 4607002308722696814L, 4489510763519462096L, 4607876977209474343L, 4490400747114037882L, 4600135754102872102L, 4476054872237023800L, 4601480555475945292L, 4486189048453753970L, 4607982912335053962L, -4741130319022590768L, 4607099865352377637L, 4487298829763596404L, 4609617187973213235L, 4490410451090106440L, 4609679247466532124L, -4741353998616029042L, 4607824883944098782L, 4492129160277929400L, 4609329173960541287L, 4489835660975464473L, 4607436389133036777L, 4486134052023780747L, 4601123863385848902L, 4462049449440901760L, 4609880029237331872L, 4491811586772007986L, 4599375249133852422L, -4756189959878942656L, 4602286955014152592L, 4491421852361609256L, 4606486160291849352L, -4747926122870980984L, 4608773992296162117L, 4491784292952918771L, 4602222799518213492L, 4491186577286792198L, 4602153715055279298L, -4740296690675315120L, 4599966711660950980L, 4461687231172385024L, 4609318132688914167L, -4740982388659419580L, 4608993755421419707L, 4490825156786700092L, 4607860206676826049L, -4739172032107968204L, 4608532918345134708L, -4739560223048149800L, 4609104421536278468L, 4490063972637157502L, 4606325404903152386L, 4447038148904330752L, 4610055335638991014L, 4490594460755734372L, 4608059704345021129L, 4486374234913245652L, 4608801513755884210L, 4493385973428315418L, 4608652363637879110L, 4483434187405794774L, 4610072855908830836L, 4493077869735911448L, 4607762146309991463L, -4749350129270106608L, 4605231396781252539L, -4738041332403146692L, 4609984497893704322L, 4480468596734464960L, 4605211762337575278L, 4493484158376218274L, 4608687783386407032L, 4483949026282877360L, 4609903195805928943L, 4491884965459476040L, 4607640407889985829L, 4474958485765524368L, 4599965783609731132L, 4485678845136852850L, 4599314163366300136L, 4486327985169277031L, 4609059519674087979L, -4763928844414805696L, 4601472710772397332L, -4747067183995623360L, 4605815572048468864L, 4478714614607047818L, 4603011710044121252L, -4748186877203526528L, 4603197789030780386L, 4482165792865709082L, 4606927338241559001L, 4482225409807290026L, 4607864919059938567L, 4492459997014723550L, 4603485215713951100L, 4485463511893611434L, 4608418888770280600L, 4483582696506339569L, 4603541489539883826L, 4493765675531719079L, 4605716101251400169L, 4484803686545775607L, 4604409775619663954L, 4491391432905330781L, 4610263292385555662L, 4492612449603407062L, 4610102220542963424L, -4740495779232138928L, 4600280079588660156L, -4738885824943933928L, 4603607378865609354L, 4491875270693118298L, 4605934799359865434L, 4493111246239387871L, 4609393664917900121L, 4487429706799530234L, 4606044103788859304L, 4489171013192017842L, 4607852134611820188L, 4481780643090935144L, 4602271403765342978L, 4493303278283495659L, 4603081420213017997L, 4491611106302414432L, 4609295469764686076L, 4491768755015895483L, 4603148588583511019L, 4463103218455277472L, 4607649658961207438L, 4486339172040592284L, 4604087951400618997L, 4491100403194172089L, 4608228456433561799L, -4740815978887480952L, 4608396485355217058L, -4741632855028371622L, 4608657908687080802L, 4486596132974851715L, 4599246640899175472L, 4492117615143445200L, 4607475229127017683L, -4758157700814617824L, 4604109500496329811L, 4488781187773412416L, 4605973746496371030L, -4746009851163343736L, 4607633658793110156L, -4743497534448700048L, 4606428352645064225L, 4493670946555054532L, 4609470753784885769L, 4483490074717399838L, 4606306563311169292L, -4743116729237173300L, 4610264409955128120L, 4481818057097345777L, 4610097676551650279L, 4492608053010772469L, 4605988183093536129L, 4490723923107431151L, 4609906490766272068L, 4477393268982794276L, 4608703030158887031L, -4739908541394133192L, 4606798124752785627L, 4492724172955317325L, 4609503563976528848L, 4491938764698401650L, 4606990189899370688L, 4472777164276251344L, 4609110798803175773L, 4487451113284726724L, 4609889330922717447L, 4491786496137959670L, 4608172849811193023L, 4492781930670141138L, 4608074281115862848L, 4489723275689081200L, 4603454149741719417L, 4493507254858543047L, 4607099735869708312L, -4750919773082146376L, 4605058152224166430L, 4491996545858465713L, 4603344907099817096L, 4491690216251830622L, 4608060177147051364L, -4740141085439990300L, 4599369203639062720L, 4487195211627250274L, 4607506325386057658L, -4749540767430580008L, 4609568518508561168L, 4475758104088430832L, 4604351162935278076L, -4745881804444097800L, 4599910279227757050L, 4491383153784892837L, 4606956548631686800L, -4738307847660329124L, 4607062018197159823L, 4491289862500843896L, 4603482199271117368L, 4490154425647028598L, 4599299836133520000L, 4492848037499484491L, 4603377203058156082L, 4489968448601396849L, 4602737766454550703L, 4475817972812371256L, 4609231799152339330L, -4742902445890467384L, 4604735640909929310L, -4754121655849892016L, 4601156641327824170L, 4493782769812372818L, 4609945461721313219L, 4488209192671344888L, 4609030473581580002L, 4480164606562197424L, 4607913798195794523L, 4490199374034624594L, 4599847182882359348L, 4490298731597996921L, 4602872662624122588L, 4493179922698929452L, 4600738845590028682L, 4493281346552526690L, 4609993229357278726L, 4491168683066168650L, 4600805996044695058L, 4492420347717884961L, 4608180146672289630L, 4482384192444104438L, 4608662542697189153L, 4493801505595867371L, 4609474001896903700L, 4490657231090732435L, 4604167428364145673L, 4490540994765865669L, 4608822811840239347L, 4479153772020122498L, 4606667578139064864L, -4738819580006377452L, 4607946494821261866L, 4487318893722335835L, 4601584655326041908L, -4745239603251602652L, 4609738073108750930L, 4485776597490928610L, 4605049458049080528L, 4488237634673158658L, 4605985664938317866L, 4490658652538903718L, 4609556941484688456L, 4489349249986359432L, 4608249939749729003L, 4493701489874821447L, 4609515060112054456L, 4492800342901171464L, 4610280501212794594L, -4750044056104652760L, 4609826749225349610L, 4485297514723916651L, 4608904117351278268L, -4742212645833177622L, 4608898408971333246L, 4485457674793010939L, 4603274089874086902L, 4493055961706209161L, 4605298124212404036L, 4491493508391311146L, 4606537254731877947L, -4738462493614182112L, 4604847191079940453L, 4472980125983420376L, 4607712384953205583L, -4744524094012092184L, 4607944001007308348L, -4738535791902351220L, 4608301234212123562L, 4485801197853724005L, 4608372562130995496L, 4487490405551081119L, 4607012473209768545L, 4487341054236340781L, 4610248900498452132L, 4489205765157312224L, 4608786590593038038L, 4492713692693767282L, 4604827172062847432L, 4486721279937807256L, 4609784804284334434L, -4739400629125114936L, 4609441729498485413L, 4492305372296630919L, 4607242424992329143L, 4491004681794994644L, 4607230509020726702L, 4492846028427528148L, 4607913390866230754L, 4492480344160270072L, 4608663469219526880L, 4486258895282963903L, 4605885910104340193L, -4738674894720709368L, 4609885458259288108L, 4481674004156454407L, 4609046672778589385L, 4493652991793117931L, 4607790199629889474L, -4745977063321371376L, 4607274817210611387L, 4490183942137654146L, 4607718461656391725L, 4480585020082924700L, 4609328608599749500L, 4478107504404292988L, 4609165149650395977L, 4482890379457668172L, 4601001647663017578L, 4489820335379630296L, 4601698033768686572L, 4488077533386287864L, 4600421679604826932L, 4491135094982359410L, 4609894786185766424L, 4487510398797457633L, 4607192581510017389L, 4492279412989366267L, 4607214023238809562L, 4466586647179974848L, 4609165465608376723L, 4490282733151537510L, 4599772387736780160L, 4483296421082897402L, 4604318817674453504L, 4489003822996403986L, 4607948973248029983L, -4742473332324717392L, 4607991522009945422L, 4492014590555616718L, 4605493589400549145L, 4482929136657156498L, 4607534446773702798L, -4748451092490049904L, 4601849384044454980L, 4489144371691731412L, 4608558433657341779L, 4491729776204931188L, 4607799255102296115L, 4493008040831839698L, 4609428449630554382L, 4476521145353558736L, 4606162008167341718L, -4738024312011591628L, 4600086830471247254L, 4491152456614983978L, 4608248702876627930L, 4493310901509838730L, 4605601372703585330L, 4493424592868408752L, 4605669737423686278L, 4489581435638341572L, 4607793452691376616L, -4738629953573930988L, 4603735271275454281L, -4738562945548371520L, 4601946612138003662L, 4476667253988068800L, 4606571757502774004L, 4490359633197966123L, 4601101300575842532L, -4740754059569423176L, 4607835444581363533L, -4753122194606558128L, 4607369287614836412L, 4483002077907598914L, 4609744405398485514L, 4487226722578786883L, 4607686157115012947L, 4492087042549558100L, 4602587636662023314L, 4488385071559075428L, 4605818241377394086L, -4741134427145375644L, 4603528152482554949L, 4491944498097720250L, 4607516992988080178L, -4738769016416991072L, 4605116729635755543L, 4488366459967269074L, 4606453024227037853L, 4482547303065092448L, 4600538731811723234L, 4484093993706599314L, 4608296707245968640L, 4493525148402674333L, 4607147112529752739L, 4492088083445612367L, 4599249114980916744L, 4493764526026267325L, 4607471942532977901L, 4485589704505457720L, 4608237664788268990L, -4751071981238010232L, 4601934456372269232L, 4491230127773932920L, 4599736309121293708L, 4486484271351379320L, 4606703965714940386L, -4741835509132635354L, 4610160519622096024L, 4488080504519412664L, 4610254472140106568L, 4486377376432970277L, 4605426481609119176L, 4491769759688719930L, 4609679771425004336L, 4490416239830821012L, 4605268373525483089L, 4489275019823844040L, 4603423242676479908L, 4483594206194990763L, 4604447396767817880L, 4483707187613966511L, 4603029061192917343L, 4489600271777177810L, 4604047499561012540L, 4491983052224280620L, 4600729684687175224L, 4479746834446480402L, 4605096011829843673L, -4749198169694680008L, 4606954701176976413L, 4480827250094439662L, 4608009046875736187L, 4492264487569998262L, 4604325983749558022L, -4747367724002116376L, 4608021079641984035L, 4486046674004523151L, 4604555583249690132L, 4491608939926936432L, 4602858630116578147L, 4492816830749928804L, 4604796046769864419L, 4468530054806125408L, 4603730581833778418L, 4491331763348305144L, 4602918438186852814L, 4474991269705127120L, 4608936082688935223L, 4483153039452042400L, 4602720660430936750L, 4489765374899521680L, 4607368471925915251L, 4493831852920935199L, 4609799440339249666L, 4481433428004836938L, 4605210603689623576L, 4487023528865757974L, 4600467521243386214L, 4487165536921102835L, 4601079915981643668L, 4487106453734724648L, 4606778878069196560L, -4744172094779120428L, 4607896007052930759L, -4755580830234426704L, 4600245758962581126L, 4492376829276406966L, 4601553223471802122L, 4491044158163905620L, 4609166010168907810L, 4491515937690195472L, 4606689805678500381L, 4489572661947170446L, 4602014824985207190L, -4746555076973244324L, 4608507234468492584L, 4492368349842766607L, 4609871515886775283L, 4492060341364330342L, 4599971672667014874L, 4491831080758492938L, 4604771565608261969L, 4493638232524051703L, 4608999377886435123L, 4484823091550191970L, 4603475961443645252L, 4477857019736566464L, 4603869059475257803L, -4743289881135634776L, 4605987943681795381L, 4491085245652233574L, 4603640706454800697L, 4482849513700723889L, 4602313461459413430L, -4751238839152858288L, 4606807764776619192L, -4741447366423648326L, 4605985941945786788L, 4492556639043582368L, 4609110169787543499L, 4486720664831859967L, 4605354516922013106L, 4490791959663628081L, 4600942667117748048L, -4747265926857560368L, 4602844725721111042L, -4749817858475768472L, 4609066555066690296L, 4486577943781674714L, 4605089190424112584L, 4484198691445433254L, 4608461716668045344L, 4476504749735290160L, 4609222471740898595L, 4492096085213789108L, 4609836245450172467L, 4491494844681240571L, 4610086195464462382L, 4493431320044369414L, 4608837055680733617L, 4488269761920738916L, 4603344122093322068L, 4489814705505696628L, 4600850821670791112L, 4493254662894743954L, 4604398169332236906L, 4482748269527686994L, 4603909257356991838L, 4492047555986393000L, 4607246652135134021L, 4485407410955454722L, 4605593125457950285L, 4486109355415429455L, 4608873676837560011L, 4493006799551272168L, 4609256696367225565L, -4739018310073301136L, 4608198112302223947L, 4491231452605909430L, 4606608028050009289L, 4486081330963983140L, 4608880591708257114L, 4490215851809100500L, 4604823842311787456L, -4753313712074395264L, 4606628902148905520L, 4488968643032413557L, 4609112716802849456L, 4492700186501185500L, 4610167445878006728L, 4485043487762635532L, 4603626121083513280L, 4485637215933799796L, 4603496530145854662L, -4740947456918850788L, 4609861036613079032L, -4742926783888252216L, 4602384077973170862L, 4481794651529734657L, 4603219306120583528L, 4490158973384763838L, 4608734530914227192L, 4490322140349027124L, 4608093339385762594L, -4751220275907246184L, 4609173909381257949L, 4472261494079747000L, 4610251590613596513L, 4493816615393433972L, 4609682109386486529L, 4492798942316407870L, 4606682621298299486L, 4493381995797928420L, 4607661026417295638L, 4486175543668634963L, 4600420155902455584L, -4749054009032599848L, 4605751203027479209L, 4490892961271731444L, 4608591092761248144L, 4490753202104994066L, 4607027936091198866L, 4488448277731875777L, 4601972526210935382L, 4486203362710464841L, 4608052109960149759L, 4488296472428278517L, 4607800253748032510L, 4465404327410088064L, 4606137749103012349L, 4482563814680124563L, 4605983305895504624L, -4741025406948843372L, 4602771929032392394L, -4738226638507261488L, 4605450247844877828L, -4739174728709897208L, 4607551739974694503L, -4738812024304906992L, 4606392316838281786L, 4483332354940271801L, 4606384424761718723L, 4477546810301486248L, 4607786808758958715L, 4491543083385976118L, 4608365852586447766L, 4492340077096129726L, 4605623294205930633L, 4481340516364085542L, 4604481217337342546L, 4490992424049465638L, 4605647774221107349L, -4739823487501814432L, 4607660636045008846L, -4744396352797528868L, 4601727414233391832L, 4493464698538432933L, 4609214535051903135L, 4492769825354692546L, 4608048803473357425L, 4492026146058626614L, 4600709589712137532L, 4482319234734138217L, 4606163575845172064L, 4491850919754165632L, 4608734463759489567L, 4483165648734771869L, 4603144024765484052L, 4492009955684478660L, 4605120807266416354L, 4490741858548848534L, 4606735091915028711L, -4741876545616246534L, 4602180476753942556L, 4490567766967229669L, 4607590549258089561L, 4489819164919919100L, 4600690626867884586L, 4489410815803360906L, 4609393040255300402L, 4491057614018066449L, 4608459709120276199L, 4477836555788025344L, 4605167528290781499L, 4479302996335133540L, 4603562328798560782L, 4491635393533366262L, 4606841579848922329L, 4490228494382980143L, 4604895838291733692L, 4485584901772772294L, 4608488495953132783L, 4486662708905981221L, 4604553002098531282L, 4490866833317588088L, 4606996462226692423L, 4488695715967178510L, 4605396783175334708L, 4482013838352252825L, 4607954637551021006L, 4490599079766538846L, 4603104129098512192L, -4743673613409142152L, 4601175502714481886L, -4739010905310282376L, 4603388600967270544L, 4492224521569143645L, 4599949387434413412L, 4477452042264918588L, 4609631070232930489L, 4488174839727776539L, 4607650875521712984L, 4488863415579539029L, 4606241912233770188L, 4490643744286376302L, 4604635292125121980L, -4760052618147084352L, 4606436482189183928L, 4491757416467609360L, 4607414570367615896L, 4463011703846904896L, 4600690487922611466L, 4490923962277237952L, 4609932175550644978L, 4493674275711148196L, 4609743659507738776L, 4478408783460895044L, 4608859277756334064L, 4479707854780404140L, 4600683792147157914L, 4485096571300434810L, 4607149124921889732L, 4493767117055113106L, 4609770598653143354L, 4490557184102485517L, 4599546296433792828L, 4483453013207770525L, 4605448915872223179L, 4491288422128734432L, 4600658774687143602L, 4493067154816728728L, 4609410441165075115L, 4484637800257719480L, 4609819902908167958L, 4485878383217944827L, 4607561515929744742L, 4487524484918201532L, 4607635600597956661L, 4491234667741819420L, 4608030855761684859L, -4738409639192213620L, 4609966149171960954L, 4485850085716140002L, 4606109155534496088L, 4486614005943633498L, 4609570555912347354L, -4745990318400283792L, 4609991755623163227L, 4491350199955701478L, 4605949440058356438L, -4765840730209309056L, 4607388000864056674L, -4742324701871491948L, 4602602425760168294L, 4480311247966425660L, 4602057455646631674L, -4740054349827597888L, 4608502894986433132L, 4491349627765260350L, 4608631997420391544L, 4493659621417880668L, 4602293810496413810L, -4740835450721224708L, 4606458068366200534L, 4484082667373302983L, 4609238363667053004L, 4491441750201084837L, 4609202399926192247L, -4747230084129320720L, 4603396385996512729L, 4488821239042694810L, 4603122881512458836L, 4488586752221410624L, 4601225742408400382L, 4485589505988189726L, 4606337867292974023L, -4739780265741586060L, 4605760153078977027L, -4746656275218380148L, 4604769021334396530L, 4486983912475619841L, 4605142885313292757L, 4491473753047957181L, 4599947443089551872L, 4489521135105861996L, 4608192285053221163L, 4486983681509170003L, 4606133884172629085L, -4738828075307440440L, 4608855558809048009L, 4476755569040516488L, 4602185985311344142L, 4492409598325890192L, 4605069770077416503L, 4483569307557699173L, 4604231309584255356L, 4493130222574780381L, 4605857286032147088L, 4481717830814818553L, 4608262589157766657L, 4489616300378285256L, 4608063991034504072L, -4738446210900776276L, 4600056015637726846L, 4472021954621328304L, 4600737106060164194L, 4492923061669160320L, 4606997261101804983L, 4492217192813074388L, 4601874575390032016L, 4481695829407166687L, 4608761519085598810L, 4491035904983550430L, 4610121304106076210L, 4489185870050864324L, 4607903786192500571L, 4461077156063355968L, 4608882755775946202L, 4491246543211651978L, 4606912704547906163L, -4742147159101310836L, 4609325885148971403L, 4487120937912518019L, 4600195399944355176L, 4493164632005132072L, 4608889528054426140L, 4482159852953365258L, 4609703694997085752L, -4744913595733031904L, 4600385862067123826L, 4493389327134288640L, 4608914912550456884L, -4738445256564600236L, 4609174636271630152L, 4493558599811704271L, 4608430402860106686L, 4486437174818378923L, 4607440203784588667L, 4481946944506498239L, 4607472372346022523L, 4492783123791024156L, 4604385815403941248L, 4490585035547326854L, 4609523914563183503L, 4490963814198409078L, 4606780444682278084L, 4492012323396925722L, 4610193974367341934L, 4481181805498827786L, 4607304642239412680L, 4485889225313329408L, 4606574094650715395L, 4452634829508840704L, 4607573197966429965L, 4491471214449609327L, 4607261740269803901L, 4492093120167392894L, 4604340367487440323L, 4487438243040947963L, 4608038399953382403L, 4483494767804340492L, 4609778201981553055L, 4493109449662611210L, 4608383276802983087L, 4493120402376842617L, 4602858944124086800L, 4485837547697063899L, 4605996657251729178L, -4738115178336655116L, 4602935692808711314L, -4761452109224602368L, 4608702769961356190L, 4492195741474179782L, 4607658729140702888L, -4741302963266095214L, 4606783862358329473L, 4492249473159876624L, 4606451017365696241L, 4493490258002153543L, 4608751801675804994L, 4487789724742107382L, 4606252005780029226L, 4492076080526052586L, 4607952566085176080L, 4473730211951612072L, 4605066176965873294L, 4482158316844279178L, 4604939072907842245L, 4476222631874808816L, 4608532807971198239L, 4492496719880693170L, 4604353496663709446L, -4738856591742729756L, 4607324495458138225L, 4486629734404732246L, 4609353345001164034L, 4478478053707724576L, 4608741249103279805L, 4492578503160918840L, 4604756922701055575L, 4486958637979470646L, 4607599404813922733L, 4491410401595585472L, 4607384835002255675L, 4464280796073097472L, 4610012887963176684L, -4751885691059807104L, 4604992416241375269L, 4490600063882856983L, 4606525514900429121L, 4482752834798665764L, 4609690243928886535L, -4745204899193813048L, 4609870688683722051L, 4492876314686486822L, 4609843278775002990L, -4746350994036278600L, 4604986818146732566L, 4486503324175900862L, 4609938756352436250L, 4478867368351335684L, 4607392126799086178L, 4486345987158277188L, 4609842582136181392L, -4745054326584603968L, 4601108971960237970L, 4489411450780108850L, 4601451123601683870L, 4485134222595703051L, 4601861559441372096L, 4491491944061381202L, 4599717712239989394L, 4492180484229843006L, 4600609966054231338L, 4491777520062797673L, 4607790504760760904L, 4490170155019068796L, 4608396383551692875L, 4476016112230215456L, 4601050778559129534L, 4490504296379851110L, 4607910008731799066L, 4491868086720178590L, 4601039211556956122L, 4481534855222683486L, 4607723365794326743L, -4753102848995919264L, 4609877121410470892L, 4486461890204828929L, 4602811069018463522L, 4490140508582312218L, 4608896541771204950L, -4738941933167577280L, 4608821807875812534L, 4491757532961351155L, 4609566772318618810L, 4487132864543374355L, 4607378512183859941L, 4493821450727735235L, 4607030172571045664L, 4490572115107182930L, 4608690183581669029L, 4491943795786119433L, 4610201197392706734L, 4482329350083861295L, 4600763194457586110L, -4738366732603808704L, 4603342551873130222L, 4487469796117335532L, 4608118877624863425L, -4743227263141643808L, 4610066813109603012L, 4487631066632474408L, 4603414957743125726L, -4755385039982636368L, 4609928629640383142L, -4745747113040555680L, 4607114681521678546L, 4489615953964895459L, 4608408756182106083L, 4486504342451718027L, 4609137066391972132L, 4488016010799824896L, 4599985129642748070L, 4492735927018728510L, 4610189615148194090L, 4479291995209098626L, 4607379376318328645L, 4476991283760648540L, 4604559995573335295L, 4486415378627879814L, 4599238939350868948L, 4484725344949372912L, 4610276553532521082L, -4745237597999347764L, 4607984958179697275L, -4764207510713288896L, 4609970007802227548L, 4488387713841670471L, 4600954659504680764L, -4740540207813963636L, 4608940748976243598L, 4486403719621771760L, 4602386489921271714L, 4483208240141380615L, 4608237624713863015L, 4488234845704310999L, 4606792381351348916L, -4738958020366615264L, 4609032927667795494L, 4490668900447624879L, 4599914233000083354L, 4491899972435804912L, 4606097599913119346L, 4475090894052168296L, 4605292986825767648L, 4489573829765080336L, 4609222239815359648L, 4473476541216951552L, 4604131185718674886L, 4468343741277466432L, 4607533932557245338L, 4493163059643544518L, 4603446200388672259L, 4484747520104676901L, 4608368238520115537L, 4486269435836008115L, 4607057995770226285L, 4491827457301829785L, 4604917515840141293L, 4490729272497690192L, 4603518590002799304L, 4484282046387446382L, 4607448416719082077L, -4738502795798733504L, 4601523086516729022L, 4493115712782542161L, 4607967294771431593L, 4491950640004747674L, 4604120177990970162L, 4492050808414687310L, 4604427501667570688L, 4492223697684849279L, 4605296928519885869L, -4743744178138557796L, 4606617424335109382L, 4483396749947455135L, 4610002828816474754L, 4480072135986470562L, 4608714836003738016L, 4487397477115238419L, 4600214438208703702L, 4492563559909853631L, 4610013379033637882L, 4492845859167781420L, 4608748698052862277L, -4739006694624701704L, 4609301537157817433L, 4480771178473358508L, 4606805556847087484L, 4493671384116908792L, 4607741947715126406L, 4491528567423986216L, 4607980985824814569L, 4491092448362338300L, 4609586739336627897L, 4491922539857797283L, 4605894275250771900L, 4473715474818464888L, 4608554098686076438L, 4486285669750955022L, 4603131797594737782L, 4488840911651373185L, 4610173546971603226L, 4493088153584116928L, 4599785995526988826L, 4485362934176015974L, 4604280474498105574L, 4489140860221451632L, 4609690640992967620L, 4481610021382417842L, 4608184116221555725L, 4481488764620240778L, 4608525386664823564L, 4493223968791498148L, 4608957070375160693L, 4490435171320554133L, 4602185000132742272L, 4492715168478887828L, 4602767961702663812L, 4478250342499431278L, 4601319138614320866L, -4742586233052431316L, 4602960459051137638L, 4481209080651320863L, 4603743929543832887L, 4482727517022408078L, 4601485506482328292L, 4486395895951457064L, 4600218812792276726L, 4490384043884173664L, 4606285589489257887L, 4492955138415849874L, 4607790233728445904L, 4488226252590208546L, 4609264656499610040L, 4487166256297227089L, 4600070576680860762L, 4492882302647953988L, 4603707338451102677L, 4488749439957790471L, 4608178397914816212L, -4740112821551679220L, 4603981367413200904L, -4747302790485132576L, 4609535641367306469L, 4477753653712512488L, 4605866353418711957L, -4738592410862485396L, 4605259521755888865L, 4492069601780117866L, 4608189317571344602L, 4490066087307882150L, 4608753253840584117L, 4492806427154594540L, 4608377837321649762L, 4488455415564476734L, 4608660053014874796L, 4492463362372089958L, 4603799445582585590L, -4742483393787813320L, 4608915263347487317L, -4740824123147225832L, 4610209148816884924L, -4741459113989083428L, 4607393403532489752L, 4485383686110633705L, 4604603019870212618L, 4493761395724872715L, 4601652690358900174L, 4472826937020950792L, 4609248553526172331L, 4487934008856239959L, 4607918336304987087L, 4487643981260865909L, 4608252021162736379L, 4483838725496888056L, 4609162671066558968L, 4493670564297577914L, 4607959439249029449L, 4492266178547287690L, 4610240677149008226L, 4490567901110438122L, 4608755966068827991L, -4744106098981600400L, 4610064398994806450L, 4489314475836550810L, 4602126195881609664L, 4485992411455553436L, 4607032682979665008L, 4486674270812836304L, 4608321408617283064L, 4482448319899425523L, 4601233853502624788L, 4488327634718375090L, 4602575285349643444L, 4488158707652646432L, 4605804156272644254L, 4493803092205952800L, 4601305763360575080L, 4490676199268824752L, 4608749564903241734L, 4483627952319625890L, 4609111074762779113L, 4470152393742431104L, 4600042162459928582L, 4488592476904087102L, 4605658002211711604L, 4488632126119956880L, 4606654934527345267L, 4491981649957680787L, 4607807675528403617L, 4486339190343960407L, 4604831847577221698L, 4492613375953580088L, 4606202107460275935L, 4493617143560379456L, 4605104178416899125L, 4489384456058147954L, 4603393243889897708L, 4487295854249925288L, 4602192318433574794L, 4493106820298413483L, 4610269899759817598L, 4488994852495781834L, 4609487116791575706L, 4488676243549993574L, 4610169734500576940L, 4491738768258536214L, 4603363341288731160L, 4491800197942077950L, 4602531769091478124L, 4492552697525358850L, 4608876043824384520L, 4460934578824224576L, 4604799072725652937L, 4492142933144249129L, 4602796125081786446L, 4490730097610082562L, 4606126020183616538L, -4740553706600703880L, 4610161805628229912L, 4472135258550047288L, 4606531978274577968L, 4485928906574794124L, 4602826779579541556L, 4489012495947019709L, 4609769657519088306L, 4493839699604248651L, 4610060850653006634L, -4744054227050092788L, 4601024980182695298L, 4490607900050862613L, 4607423421454718161L, 4490274791180623370L, 4600221886227226594L, 4484141043840334060L, 4606499359017239249L, 4475876303162867600L, 4601040552459203986L, 4489278573086931242L, 4607121531365431629L, 4489065616876101186L, 4610009597767231588L, 4488745064594066762L, 4608637341764164174L, 4490393012422366039L, 4605612379947053962L, 4492618206719822214L, 4608455434525033842L, 4479555334247612452L, 4603054609068909622L, 4482502008318596943L, 4603175192448626278L, 4493179583087810597L, 4600070445721015408L, 4490604886492973073L, 4609416268741484410L, 4486286569167789978L, 4607253348065593423L, 4488670443961083900L, 4607578628321249711L, 4491033433161774331L, 4603855139098327468L, 4487995963272901556L, 4609916164106152856L, -4740657082348449432L, 4604800654875326688L, 4492394494344069718L, 4603797404300923642L, 4488792666813713388L, 4602244050575066742L, 4489260368004392335L, 4604105367442821554L, -4763303882681798464L, 4605465965419530723L, 4493171660702574749L, 4607270599593796327L, -4756671973409345632L, 4605775775774891268L, 4484096482102442047L, 4603709764221964405L, 4477859681611771972L, 4601762758315811186L, 4486939953463419732L, 4605041869201431647L, 4490933381816445052L, 4600671814150903450L, 4489024588008904026L, 4609050920010945293L, 4492768807059960835L, 4600180438691030728L, -4747245822736775512L, 4603168608953233474L, 4489225714516995245L, 4610145961814786215L, -4763083363713726464L, 4602254646104432320L, -4743729190513623240L, 4607638595258966927L, 4488164271102007894L, 4599452327556352814L, 4489783033331667403L, 4603193654917379981L, 4493357261550485070L, 4603033998498058591L, 4492469185840176472L, 4606134507648717748L, 4479408380705029114L, 4609012308843610212L, 4484727147228328001L, 4606387914746463461L, 4483524841328624043L, 4609132691688466223L, -4739611411882612924L, 4601194847189877534L, 4491162387785978126L, 4600878400872930560L, 4492511711805069979L, 4609612167225748612L, -4753694471042615264L, 4610172828443688120L, -4753283713146591360L, 4600296367672173980L, -4741311211998056898L, 4609427826666971102L, 4486911216893652867L, 4607754857772076251L, 4491177859177151283L, 4608901801560971637L, 4487231146421157794L, 4609447327928257560L, 4489102586834124363L, 4604197514054392464L, -4750153500187036448L, 4606642711017328775L, 4492338723349526706L, 4605493907312422440L, 4464243914292319424L, 4608395996329611258L, 4489606169066207771L, 4608208668959023358L, 4491945327272607945L, 4608979037949940665L, 4485518742030104136L, 4601684635119369498L, 4493442624766121182L, 4603536533133266882L, 4485999751271292184L, 4604611105905601052L, -4741094230742984688L, 4604416797366131701L, -4738440245201576616L, 4606246095369387030L, -4744238700776637232L, 4604617573161591434L, 4490372691082992920L, 4609520088560099482L, 4487150934420981828L, 4608416867934043104L, 4471271422375371936L, 4601404108254367074L, 4486443636625681561L, 4607898384105340329L, 4493628281836916399L, 4607996776433667355L, 4490234148072381602L, 4603878202824004094L, 4491214328193170447L, 4607784962989576200L, -4739508500577474108L, 4603510912739348456L, 4490371756689689150L, 4607884793222289985L, 4490117587222802840L, 4608180032442290789L, 4491718926113218425L, 4605843649360698083L, 4485731372007017061L, 4601995446030777260L, 4488874738611902572L, 4608428837966654678L, 4472965423965775944L, 4607720825205937400L, -4752140322531731440L, 4609262775187512880L, 4486403594358036498L, 4609400059421490317L, 4476527933781527352L, 4609599549954266110L, 4485894098923409981L, 4603330412523890254L, 4479748552711913018L, 4609471691796206136L, 4491828053361440178L, 4608467366475665414L, -4746225328139150848L, 4602753265569758006L, -4744379386943963152L, 4608230020188335222L, 4491377588661718300L, 4609623085038319734L, 4491133015577669188L, 4602979908242078342L, -4751532447224937488L, 4605820292327950040L, -4750481917410066352L, 4610196752247688556L, 4491680639989048033L, 4607910668169720693L, 4492180711798990002L, 4607548047747894055L, -4739024942613440508L, 4603036250803313174L, 4492324823317898881L, 4603690241542405010L, 4486750368161667317L, 4608546505925652122L, 4474394289684191040L, 4607766795034025074L, 4491214936575116455L, 4600898082800100208L, -4738628353296291420L, 4606462810066365664L, -4748670394441891640L, 4605055673826878214L, -4754145534117108464L, 4603854778802697798L, 4483710781836279161L, 4608329463126227557L, -4740715278663211816L, 4606842925536237908L, 4484172207613296006L, 4607316413979339713L, -4739259882269840552L, 4605303281511280586L, 4489156307024903585L, 4608692493268941060L, -4738891976432355372L, 4600232104999941510L, -4744179177320468996L, 4604340647992313599L, 4482669040322173934L, 4607585066018361738L, 4491283627787239934L, 4609130953684102606L, 4492171770159116436L, 4609320656235217213L, -4742203307211036042L, 4607213877235835732L, 4492540879417727866L, 4606875921423992559L, 4478198400702844608L, 4603736417491867164L, 4490384995269298366L, 4609684734408589444L, 4491087072878119298L, 4608095927184421908L, 4483494157184220820L, 4605514304363011872L, 4493067506846317526L, 4609082951352681266L, 4481906577389713300L, 4601121219389353304L, 4481641245945398740L, 4609173260756356902L, -4746716445833379620L, 4600298982926664676L, -4738900567639091240L, 4608944974300958957L, 4488873117981328798L, 4604794122361792837L, 4493166747027418562L, 4606881120846229123L, 4484759927815739047L, 4608598667373713436L, -4757321465635212032L, 4602864810599857475L, 4490274710340912603L, 4609146277953710406L, -4740038963297256560L, 4599737141707857478L, 4491089890720020590L, 4601423033355075446L, -4747635928809178128L, 4600799264214515124L, -4746807702925189760L, 4604290155370688008L, 4487427388894609033L, 4607221997827809810L, 4479467131405512730L, 4605015776155440550L, -4740936303154494518L, 4609608309869456737L, 4489482324315454125L, 4607418462504398510L, -4742511920043140976L, 4608495877318728618L, 4491724941671161315L, 4607497962428328884L, -4745793868957390336L, 4604592974055795219L, 4488376379138936370L, 4610010147399166493L, 4490703756657018955L, 4604893069817004274L, 4474478170906959808L, 4603184568219968084L, 4484455895631284589L, 4610151136081376802L, 4489565545772395662L, 4608780923614747236L, -4749550986207035616L, 4609815430393604051L, 4487100582124157878L, 4608077857341637897L, 4482732525754808038L, 4608870865755281773L, 4493576834707457414L, 4607703438924824952L, -4743697114522789356L, 4608111562748186616L, 4490002610316500123L, 4601490867683305574L, -4743076363066123440L, 4606957991752260634L, 4470769886188647120L, 4609300042786622881L, 4482976117075657754L, 4608228025806828024L, 4491307977897781526L, 4607378236105153962L, 4491769893454066399L, 4608036523673227743L, 4463357114355072864L, 4604166411951404784L, 4490722545037615425L, 4608684941386736431L, -4740134212684382468L, 4604877077709186546L, -4741834380812420392L, 4608447471557142894L, 4477650747370464408L, 4606324966187197376L, 4493757576139361315L, 4609466901684061284L, 4493685244292609891L, 4609188934049760130L, 4491691603518813372L, 4604355637578993783L, -4741233633113997012L, 4607684477056998212L, 4487076465458654331L, 4604804566614263204L, 4493193312490424234L, 4609148387822082854L, 4493390480725975150L, 4608022509136408265L, 4491874082991706512L, 4602903476220267166L, 4488534931495629546L, 4602929586310971164L, 4481376914572399850L, 4608163268922036102L, 4492564657007768822L, 4607503265420505567L, 4487465443423488736L, 4603854722249573458L, 4493518874120292098L, 4608076617824555740L, 4480489618190260884L, 4607467314815099027L, 4479192639481911116L, 4608651031489345078L, 4493603693335712134L, 4608368416289029464L, 4487507877596480323L, 4610118267361757755L, 4491259290936926080L, 4608579789284780398L, 4488226114758802720L, 4608033943314047266L, -4748960182407000752L, 4608705429142227877L, 4465251069015182112L, 4605155186830641947L, 4484003023276855206L, 4603944343565704717L, 4481247043184236719L, 4600521254068685608L, 4486944238060137379L, 4605555323505309262L, -4749843018594339808L, 4606611929486265876L, 4491559652259928768L, 4602768347158794245L, -4756985504212558432L, 4602271318642831888L, 4488881796874054852L, 4607487254814060941L, 4491876432248484292L, 4607790396079329948L, 4490466858424288121L, 4606573608366369555L, 4490584653288021689L, 4605290727618440593L, 4484011268842058366L, 4609963596078392266L, 4492999763992576768L, 4602455411015437040L, 4484385560154922322L, 4609948450495941148L, 4472395938608245600L, 4603341550261750896L, 4488036962258873280L, 4608783242459661227L, 4492942027635580128L, 4606858282867524671L, 4482374727805006240L, 4607580514575768404L, -4750288358582796128L, 4605058950149596192L, 4468764318775068544L, 4606627330366294001L, 4482030920451803641L, 4607246464634695348L, -4738432568170872008L, 4608002836575428093L, 4487332507597155144L, 4605999046782618334L, -4739387632632072020L, 4600511620332022174L, -4755308338642575552L, 4608315125584538140L, -4744131614575572584L, 4610230279096986950L, 4489904255677826614L, 4608636709092998991L, 4493687763950617352L, 4607038085902274785L, 4480900209194600454L, 4605471957806140498L, 4472916348211051464L, 4604184026970126052L, -4739080387654956528L, 4603421374058090544L, 4489962827972821786L, 4609691400954253846L, -4749640970357836920L, 4609346907357129656L, 4484099693555542718L, 4605463040255329130L, 4479627189125625722L, 4599476946019130148L, 4491753054869194830L, 4607529786287330660L, 4476722346253161980L, 4609988298357160788L, -4740097402289173676L, 4599283710906288370L, 4486243446697201074L, 4609570188574031804L, 4492353530732448083L, 4609982361492104585L, -4749341275839042984L, 4607027939058405820L, 4486143076625006959L, 4607809066810873790L, -4741690695717718194L, 4605101397791666134L, 4490895291495820106L, 4607613210441544657L, 4489391412678960530L, 4608055956318978139L, -4752343463667286608L, 4609629402166594850L, 4490067923191308353L, 4606100539266528029L, 4491050572245340482L, 4605585948923932491L, 4486497236278214711L, 4602980109252998930L, 4490809351003013054L, 4610114278622758184L, 4492015759751037166L, 4604103862137699404L, 4486163397490812171L, 4608469944593342322L, 4490331644346530796L, 4610294192856807596L, 4481822008513058188L, 4606888007928176387L, 4485108949801144484L, 4600986047488925586L, 4487773145570918076L, 4608462612258037174L, 4492256863797251098L, 4600550821007687320L, 4491548952288426972L, 4610147411215245074L, -4740190266171754192L, 4600183123842219116L, -4741400967257667910L, 4607354342253355123L, 4487016177080464724L, 4608849397915208066L, 4486849110897285593L, 4603213203657490651L, 4483135229079571339L, 4605276289918595899L, 4491428183627153158L, 4609775706925801785L, 4491426821563539544L, 4609931027939864538L, -4747678853634216464L, 4607932314550827666L, 4491863446185453998L, 4609667350151681064L, -4773547217445827840L, 4606828002232872495L, 4491967042493597544L, 4607655877167979443L, 4488768432203177708L, 4605447039711873960L, 4475629848753288488L, 4605130591329739168L, 4484693438154213841L, 4608763812099168919L, 4492820040151367188L, 4601895083397907544L, 4487754968807688755L, 4608287040768154198L, 4483815969923673076L, 4608299782408180032L, -4738725423984124768L, 4600668647086352044L, -4738329683112888760L, 4607761698257193741L, 4485843639605042531L, 4609145975124826993L, 4481508695824598482L, 4603415655056285595L, 4490603000381933284L, 4607387940227271228L, 4482648150567662432L, 4609592536108995400L, 4469626064114143712L, 4606402374596885189L, 4487364379468284054L, 4607293135768288962L, 4491696386681584948L, 4607668961507527065L, -4738623552059950648L, 4608104398302586838L, -4741401082888094526L, 4606895116551491920L, -4742925453832104568L, 4608432290705722142L, 4491142538148621880L, 4609938921218495558L, 4488642008157324266L, 4608767280035329525L, -4750200701188897048L, 4600475612592077374L, 4490273312837460915L, 4607458483386223863L, 4493362040390369437L, 4605056547789224620L, 4490818144243113849L, 4603069004144651670L, 4476769915705136868L, 4607874187982315608L, 4473061421797482288L, 4607617974048936077L, 4488436476259753420L, 4606464303980564488L, 4489268915607520915L, 4608967795120831944L, 4490975953230846425L, 4610087001639230136L, 4491229651906087046L, 4607201063303170854L, 4493413509639933219L, 4600426944131585088L, -4740450257142095604L, 4603420136429811065L, -4739012726558038760L, 4605740311236440990L, 4483171069953125963L, 4608453591554144238L, 4491248854498626614L, 4609794013724585404L, -4741002029366592558L, 4606434977321394113L, 4493818637283254230L, 4606703172745376106L, -4745100746779145304L, 4607847411758145314L, 4487835694577190009L, 4609830526358999642L, 4489841205304337155L, 4608873948664621229L, -4745251420493511392L, 4609635313975305764L, 4478048319352946676L, 4601278743714121294L, 4491413946286612768L, 4604615824791517734L, 4486121815974090556L, 4605116329662748477L, -4739717188369093288L, 4606546054241542245L, 4493156082454242832L, 4609603165789843420L, 4487023145961124338L, 4605928294460686911L, 4482018001814268997L, 4604017115173676330L, 4486594407556807032L, 4604191356511393676L, 4483880133995670460L, 4608677536807857712L, 4486580845780095727L, 4603193779704537854L, 4492911610603022849L, 4608940680705090020L, 4486819466222059954L, 4599600122907710602L, 4491418948843296190L, 4609019971457915590L, -4786916057510801408L, 4599957037741171764L, -4745144182570096376L, 4606255710568299272L, -4766075611612472576L, 4609361040763283904L, 4488281969116582108L, 4609902283210108778L, 4486327996771930001L, 4607855142290627324L, -4772235206995444736L, 4604176692616921830L, -4740394882318717800L, 4603395977084873703L, 4485931218746089073L, 4607107703525365772L, 4488368793898522767L, 4608970103171214145L, 4477128895175064660L, 4609417528692429039L, 4482029645129286806L, 4609828821057650180L, 4485536414591686786L, 4606572993083691957L, 4491682619161279562L, 4607423119835363346L, 4490569403051644224L, 4608273118575232087L, -4742790417931711296L, 4608236513093676922L, 4486615975751306931L, 4605182751886580162L, 4486004788290636239L, 4608995112286973740L, 4493335515348754440L, 4608078197721133157L, 4490674803432903962L, 4599593143503647662L, 4484802966016228608L, 4609759719148238776L, 4491791528276015886L, 4604458489580569764L, 4493407489346705093L, 4607191061728467989L, 4493261479575388956L, 4607350702786514004L, 4485431346817916822L, 4606186394349113648L, 4490446659976219567L, 4604559779777434843L, 4486166808016033662L, 4601427771347607362L, -4743595680495588372L, 4607933218568603329L, 4493530004076070269L, 4608127323601466310L, 4493023332305502376L, 4606858169235433329L, 4491823897317196268L, 4604681135688027566L, -4742253798670701012L, 4604377597124582348L, 4490263210783941502L, 4608629326459220158L, 4492509309196369045L, 4607415207313488523L, 4476548360512318824L, 4605706951921757228L, 4487397815044032459L, 4609627137474021604L, 4484179612539036970L, 4602972865906509816L, -4780603816210870272L, 4603636878858702262L, 4492431641971856585L, 4609871019109825944L, 4484652866821977522L, 4609558246312979233L, 4490937863304295952L, 4610138243594513227L, -4740117356486964572L, 4602668460521489782L, -4746643080773580076L, 4607566148380237524L, -4745801535176311516L, 4609497229482811922L, 4493299191958458512L, 4607312714648499756L, 4489179949925377948L, 4607540855285797582L, -4738129179862926956L, 4600619994510756796L, 4473136472876841440L, 4601718239696071652L, 4485675291457418872L, 4607406125852880812L, 4490125557285225035L, 4608873358366618968L, 4478862978238921638L, 4610218574506935912L, 4489386274750903311L, 4609391488026261648L, 4492329234351830164L, 4604809204475191044L, -4738580875622804912L, 4606799554028397121L, 4488904022822966211L, 4607197727815196539L, -4739709859007370648L, 4610171637815295920L, 4489088564575665599L, 4607339748592337051L, 4486695135477216413L, 4602013494108477728L, 4486136081071900726L, 4608659319250692371L, 4487997177347982240L, 4600187007316641858L, 4487715707163819846L, 4609698797437105158L, 4491122252202707932L, 4609651868835622554L, 4469286744153837792L, 4609709105764172008L, 4476993590224682832L, 4607010566634456162L, 4491005149072294087L, 4604725376866121922L, -4738773687274930100L, 4607121656143385570L, -4745037306226149576L, 4601282063230086670L, -4747029564676482272L, 4605886989862709843L, 4486155321113261156L, 4603428006906053978L, 4492002213763385987L, 4605790564744429925L, 4490261539613056100L, 4608238177441249426L, -4744708582592778876L, 4599799624479903434L, -4742042352687524036L, 4599490046266496670L, 4480508299304422996L, 4608482349182170990L};

  //Generated values with parameters:
  //Count = 1000
  //values_mean = 1.000000e+08; values_stdDev = 1.000000e+08; values_ordering = Random
  //weights_mean = 1.000000e+00; weights_stdDev = 4.000000e-01; weights_ordering = Random
  private static final long generatedVar22 = 4846200736714929451L; // approx ~ 9.662275e+15
  private static final long[] generatedWeightedValues22 = {-4508003867778414272L, 4603280828722460798L, 4729707386194668682L, 4603232863145991950L, 4719169049740466704L, 4609668086339886058L, 4713511082400400800L, 4608644018257454446L, 4729344503487506029L, 4603106757679225044L, 4732424511113751442L, 4606415218993061565L, 4728386144632071768L, 4607579096263713335L, 4729511371377018054L, 4604462857033046343L, -4501747649571881880L, 4607616692087950133L, 4732502260004910798L, 4601892358639442286L, 4725509363172377170L, 4607256290892887735L, -4503367049083650620L, 4609395380423826005L, -4500071663753881224L, 4601521410561128518L, 4725438978223336009L, 4606582840927350685L, 4732213975475248736L, 4599831112609577860L, 4729329239436336950L, 4602873294185712751L, 4729682256907048662L, 4607139978839871143L, 4729027980489954949L, 4608553113807789989L, -4500193588846328832L, 4608729218438435224L, 4726653674143249767L, 4608745671691166136L, 4727273499293313747L, 4609789828581786316L, 4730718894149778070L, 4608643039008476662L, 4733334092693001994L, 4606679797419825131L, 4701176948054622528L, 4602745489433878632L, -4506395250304431036L, 4600071788757252554L, 4733129759221298296L, 4607159275334325842L, 4732838240436624481L, 4606158173450658614L, -4513484428479627376L, 4608719302227573387L, 4733200110246990891L, 4606854953070256079L, 4733056960800155641L, 4609125803719613234L, 4705272918002146912L, 4607008890695940445L, 4729632523882744211L, 4602811453702966502L, 4704717067303293408L, 4601022977950484094L, -4503372834465260580L, 4608379682835730655L, 4728551864732817237L, 4609916783876226442L, 4729724826091455194L, 4606292489156449860L, 4724354485159886707L, 4609274900896163544L, -4506526636539513736L, 4607723015251473247L, 4731322866048319468L, 4607648389654940127L, 4727918321604854730L, 4605697904653137578L, 4727991816835710804L, 4599420926244173838L, 4731304020505498413L, 4607419433230462069L, 4724865778304772246L, 4607844621912280726L, 4729824055494581565L, 4609829460816363618L, 4722225686347499292L, 4600309395544839100L, 4731913107497689990L, 4605522123517668888L, 4733309886116241146L, 4607325042822074202L, 4720200841636863770L, 4607929113277423891L, 4728264791614185941L, 4606095992482233354L, 4730741599851817048L, 4607277347520553143L, 4732800816905814107L, 4609850442537460475L, -4499646431053106916L, 4610231384802230566L, -4530320930783153920L, 4605113171731774701L, -4502080493445093212L, 4607747013102631641L, 4733250090834965691L, 4599946752807802494L, 4731588632351959524L, 4608104696773776510L, 4733352415068777882L, 4607861558336828474L, -4499515929511676760L, 4605529902186972348L, 4726765956015116321L, 4605997369827468958L, -4514031687684187088L, 4610193173985066028L, 4730440921717424648L, 4605971990312965691L, 4725003938927360554L, 4608904115518631157L, -4499073303143573632L, 4605286101402982702L, -4507055161308322308L, 4605910174092856096L, 4713930528205004344L, 4604739425745554822L, 4724555991317420275L, 4605256131587782872L, 4727058255961660922L, 4607873230921290874L, 4731527784904381823L, 4610182978415479421L, 4730106748021890566L, 4600510293603482684L, 4729730387474452108L, 4601987133456187406L, -4502302713733913772L, 4608923499014147456L, 4731066559036775862L, 4609156979433045461L, 4684474747194537472L, 4607935265558905957L, 4732564131161289446L, 4609436449206661982L, -4501007305480803860L, 4603652254563223166L, 4728244406650767562L, 4603441894615502209L, 4726262447384430405L, 4607387913064493466L, 4726275572880846108L, 4607418660239512158L, 4721672626455848733L, 4602370735206659004L, 4728825958347571477L, 4602136772845574672L, 4720998700062612368L, 4603730438035723160L, 4729837187773745393L, 4605773424903951856L, 4732364661945891598L, 4609077957829140972L, 4721055525876123516L, 4603311122457527336L, 4720404134091623192L, 4609530910146090730L, 4732213579729879630L, 4609460060724815566L, 4730838318909057162L, 4610228592797072748L, 4730511116852730053L, 4602671721101826962L, 4726642827440804880L, 4604521543065905266L, 4724793841566331236L, 4605332762660600491L, 4729880088052571431L, 4605814720232263296L, 4713923995158164992L, 4609549610900443842L, 4724985176444825608L, 4609129708876250890L, 4727735437933270461L, 4605631915607438854L, 4724446240212399814L, 4607814904852144640L, -4510540828478496240L, 4606777755777005131L, 4709014399273686544L, 4609868662978594166L, -4514060804425526160L, 4605325259278487661L, 4731921472559956172L, 4599213890638669750L, 4715995251692254760L, 4608498756391083650L, 4733258196627618293L, 4602281834429750932L, 4731718204841867362L, 4599524228718415594L, -4505648414354922632L, 4605390754606589264L, 4725677902552310282L, 4610000817649382054L, 4731693979762509458L, 4609709119143032648L, 4731931590643396248L, 4609320555264795786L, -4506812296332537284L, 4601105057871446360L, -4499372142412632324L, 4607353169641141734L, 4732484745113270505L, 4604014799108613354L, 4731540907301324471L, 4608670682541680698L, 4729933977826321287L, 4601497652198551014L, 4721109366572293306L, 4607819210620195526L, 4726964447971852924L, 4607568258072515772L, -4507245863686793184L, 4605604490824694335L, 4725636706627725694L, 4605025341949194483L, 4731478722363046054L, 4602560789713967784L, -4499834840577154980L, 4608238630273242550L, 4730369284216898912L, 4608760425539696149L, 4729115831902695855L, 4599538879908086442L, 4724501977351468172L, 4603297283381158921L, -4505004458400867868L, 4600745662281075282L, 4723595022835047936L, 4609972559260988452L, -4499509201504782092L, 4610286504118199896L, 4733049228133815905L, 4601479345989281664L, 4729817482229505353L, 4610074398569171715L, 4732814776311773355L, 4606059471696578821L, 4722457719730124930L, 4602556875642119468L, -4511004551334421864L, 4602522363791089520L, 4732114635326802620L, 4607474126652523794L, 4732384272202443100L, 4607318534996649524L, 4725950782858623509L, 4607772579550623401L, 4722964466073696045L, 4609697417083623208L, 4728870665100564691L, 4609309819480327712L, 4718121421837806700L, 4604776019693329036L, 4732345295236619499L, 4609746365585512323L, 4723430294840039432L, 4601518761947134258L, 4733011425826576824L, 4608429273668937944L, 4722875804410568159L, 4607261566159513786L, -4508059281422879252L, 4606428383875308190L, -4498771683911899358L, 4604896321002147381L, 4699674785145036800L, 4608673291335263711L, -4511093722258656112L, 4605251794935491712L, 4724366331674794591L, 4609825066517580934L, 4728913846891509471L, 4609603277852811327L, 4732334503291760388L, 4609562627260861287L, 4732625811278536478L, 4607257011649013967L, 4729721205477546539L, 4602666791865012342L, -4520215659733068384L, 4608205915215856279L, 4728871982086144202L, 4607325745036465608L, 4730164473364318004L, 4609777929194986654L, 4716298835678716356L, 4604430053822004621L, 4729752203761607843L, 4608218006821683445L, 4729447629003897079L, 4604408715908400215L, 4730274208921104142L, 4606041166378960254L, 4726527150806521460L, 4608319919161142100L, 4732284187791620804L, 4608844470111102006L, 4707680779495902512L, 4604556553083245143L, 4723881738009509202L, 4603299373979170744L, 4732457856028422076L, 4610035638950899494L, -4511163189080074792L, 4601120302406164690L, 4729348924265579330L, 4609976238132913920L, 4728681714487584335L, 4604995639038850267L, 4731803793543548302L, 4605503558125620613L, 4728723832185134512L, 4609540233729768168L, 4730585997880121406L, 4605776231138319111L, 4729944931168820067L, 4605439476325112168L, -4500797042876857164L, 4600752199372425084L, -4505228876768921608L, 4608664958005308509L, -4507263607411050516L, 4609223703781967440L, 4727818625684979624L, 4599333716124269892L, 4720521446076049738L, 4605521832011783018L, 4732821620158994343L, 4606131296138746647L, -4520144632498908704L, 4608599341644974604L, 4729961736456617203L, 4610061000756339214L, 4730609083869670922L, 4607299126376430823L, 4729177572581881022L, 4599776977987132798L, 4732650352345304350L, 4606978655459261239L, 4732551612043378360L, 4608731052693273517L, 4720023826709489224L, 4608643150530414444L, 4721098785305955782L, 4607711438160147995L, 4721447780802327887L, 4606836456784591630L, 4732142488657683936L, 4601367682088253788L, 4721874876082105865L, 4608909155651874568L, 4722078332377147247L, 4604216923848443903L, 4730512570803247535L, 4608522149540730712L, 4733324877916137484L, 4606438892897765055L, -4500041181638317240L, 4606055186278356772L, 4726657389158415801L, 4602454881439292148L, 4722279822452533521L, 4600198488887927154L, 4729921438382610074L, 4607670771006840407L, 4723354292844133265L, 4607753694209664777L, 4719961197883878436L, 4608832516947963998L, 4731809194102850802L, 4604010578572653013L, -4503322447471953940L, 4607469745131498101L, 4732265777043883727L, 4605391354417105920L, 4731950593484109820L, 4609434897504364263L, 4730193352128142447L, 4605224274080157710L, 4727088791263989625L, 4601636719420703110L, 4726502409330356680L, 4605538221624177649L, 4727332327914181825L, 4608982243443256105L, 4720779128338238318L, 4606503048138784585L, -4504851008589308048L, 4608530931963428694L, 4732922588930030569L, 4605635364760137793L, -4505698641682691840L, 4609530987875369395L, 4708511437851688976L, 4609158277543480265L, -4511752644411270952L, 4605986093394755872L, 4714286687544448848L, 4609259664574903006L, -4502239495282742472L, 4605068932227833267L, 4730024979692380868L, 4607341825678756293L, 4728102164141462973L, 4602849210728701984L, -4501683928375172196L, 4609587782646805783L, -4504035692274381656L, 4609987503557304586L, -4498738403567101570L, 4600498977951134382L, 4728352878790570418L, 4609250326431254777L, 4713596200031878992L, 4609605962392258465L, 4732028542231254662L, 4608144405097447524L, -4505601894347625404L, 4600689701848208998L, 4726245112697857287L, 4606837416429284498L, -4500900299794734356L, 4607757075336926252L, 4729050826292128781L, 4607000555851631363L, 4729901215083197126L, 4600592596971612328L, 4725562272263564364L, 4601209431134652684L, 4732675907133966917L, 4609278573855185406L, 4725906687851790591L, 4606938666933008054L, 4718649041754283640L, 4610074546709554718L, 4733243640730861317L, 4602848946646586456L, -4507470255917729028L, 4606049863045790442L, 4729063250787911484L, 4606291915484340725L, -4499548207656820452L, 4606026647620574786L, 4726375006385605268L, 4602541318939161502L, 4731167924853748602L, 4602697522763523010L, 4721211870032338497L, 4607777965249463964L, 4728852885529225330L, 4601603045295180890L, 4714902395690805584L, 4609011758798508043L, 4728054309013979994L, 4607678375925814082L, 4731403693083990696L, 4609219612213773220L, -4507694360935729468L, 4604943272036419513L, 4729457502412877382L, 4607176820955244941L, -4504433664758913232L, 4604571914069910944L, 4725976306506131401L, 4606925320796472265L, 4730759566362848397L, 4605104906787653608L, 4728383476864528548L, 4607884924082140978L, 4726824161470487815L, 4602872058153126149L, 4729695678961138890L, 4607084124096380491L, -4504119258534672956L, 4605711983162276891L, 4729180873110636782L, 4604745775668224638L, 4732799032271128738L, 4605944876344127752L, 4727966416593011267L, 4608854151443429256L, 4729793715346341891L, 4607744265434739017L, 4732090584481417630L, 4608849444622521105L, 4728827711296205143L, 4609973550396583758L, 4727382654842582688L, 4607447902940335053L, 4729369442138131248L, 4607562395877407982L, 4731562737000359702L, 4600168489029322236L, 4725062573837449999L, 4610097285970630711L, 4731200971151448808L, 4599254815137323548L, 4733098279847330508L, 4609571936555181303L, 4732931716990258561L, 4609714446912459026L, 4726858648012886120L, 4608100443798005769L, 4731136631901908146L, 4605479150750361297L, 4711426554944997136L, 4606163365907123521L, 4727141297423183057L, 4608403581931674905L, 4731637963753948548L, 4601756570394170734L, 4733283726953361878L, 4604858601317085342L, 4732295040410638910L, 4609432965892504858L, 4732172321762380035L, 4605366573311899330L, 4726014190286750350L, 4604332043481795144L, 4727307706064666608L, 4599421774098707358L, 4728809963993054216L, 4608335942651679566L, 4720773158810065904L, 4606841184546491700L, 4733083758741111036L, 4604243001933180082L, -4498958334992329420L, 4603930014060077152L, 4703907509822552192L, 4603085182645755750L, 4728690395526118683L, 4600629412030073170L, -4502874227346842756L, 4605071609184777746L, -4500591444402032532L, 4608754461191199277L, 4718943566915648728L, 4607836621179717290L, 4727980309938711832L, 4605885547970098620L, 4721714866007863638L, 4606191431313778406L, -4502329022918905708L, 4608232432318190287L, 4730789631587230617L, 4608368906261689853L, 4730213493831070676L, 4604833494730021990L, 4725637939759781344L, 4607224648046845571L, 4728366411224996274L, 4606030757912440399L, 4719875280258141309L, 4604432336284450292L, -4513604397775844912L, 4608463769404563822L, 4720073456417635558L, 4608721720059213814L, 4730287006625691542L, 4607743453502816957L, 4728875963378641711L, 4607390713066083847L, -4500664794977410548L, 4605408584319918768L, 4728615042338188304L, 4603791837956554957L, 4733013547569572491L, 4607285538207292805L, 4730289478651936729L, 4604170167851335552L, 4727302937586881639L, 4609549128376361322L, 4729142579149254478L, 4600290772216496238L, 4728196910129193776L, 4609893139244303541L, 4728564251947729477L, 4607250812920755500L, 4731199984070764571L, 4609172940208884462L, 4724633237464174900L, 4605507302888899583L, 4729149659120893136L, 4608386393968528025L, -4507635195531701544L, 4609299947838106253L, 4725943761384430511L, 4610192858273491997L, 4729691844341871870L, 4606415876806700415L, 4728935219920651378L, 4609574682347997144L, -4518116689074415200L, 4603314803463320038L, -4501423905564751012L, 4606747342407282622L, 4729433308555097933L, 4607721036046837742L, -4500994441838523008L, 4609514999910598652L, -4503935468796967952L, 4607428369952381129L, 4728858776488666033L, 4606454523177082195L, -4503597163368960200L, 4609902011767673814L, 4728050625246167258L, 4602704351686922024L, 4715913034968350352L, 4602833290390275584L, 4733101443356437928L, 4600611694277200920L, -4515388772263361536L, 4604650999408051960L, 4726839809351388886L, 4608783777163375201L, 4731742903625022879L, 4608159421861537244L, 4724998722239994729L, 4609340788463149931L, 4720084353699656404L, 4609331949723201156L, 4733280381172867710L, 4604213667258533714L, 4732430051797418883L, 4607010999519019950L, 4733019130913844558L, 4607743555830735436L, 4725530765335934481L, 4604107466183001510L, 4719317279719359852L, 4609881265458283748L, 4730462334679691888L, 4608388787083773892L, -4525425353385613952L, 4609348373847204897L, 4722316659783130725L, 4608946813819863812L, 4729731455263257314L, 4609361414129526415L, 4725986998657079273L, 4604251637565682659L, 4732937175964100739L, 4602770162492595090L, 4732846178057817634L, 4605081078325219054L, -4498929110412040352L, 4605041668765254350L, 4731501423813672656L, 4609474783851304425L, 4732039541232248589L, 4608432476418770007L, 4729073513689503588L, 4609131098759443941L, -4518719967160285184L, 4608914214232262756L, 4724402565465586935L, 4610274870695669839L, 4730023437380358344L, 4603959702869645694L, 4727600454994848058L, 4607978824294758655L, 4702172414391421024L, 4609249041042978635L, 4719739636382183690L, 4604439238590662950L, 4730276746144300074L, 4607616093861908189L, 4729575206166544392L, 4599620754177659702L, 4729552001677392706L, 4605937450795017390L, -4507335967974619856L, 4609667729922560561L, 4729744766796128456L, 4603023659413294735L, 4730956184871578687L, 4603687535282922221L, -4499123725796056440L, 4607584245603079118L, 4721315240481696488L, 4601363598549475790L, 4730853744044715122L, 4603092971241553298L, 4721493083486326423L, 4607652178348382185L, 4726219112307922671L, 4601090585780869336L, 4731481942325404848L, 4600140235912408030L, 4731944010534783851L, 4605267780413155284L, -4498767238495825290L, 4603822793343701876L, 4728059555499161822L, 4608527274192356622L, 4726627560227800911L, 4608621624049823825L, 4730835975561594929L, 4606867704813845307L, 4731764215856402660L, 4599411009109481008L, 4731810281607285376L, 4609745090972728086L, 4725986010753335410L, 4607298775205696613L, -4499066227168773594L, 4605312891479126565L, 4709291055000916736L, 4609293091291880350L, 4730371015270469832L, 4602819286056009236L, 4729919622492052152L, 4607378631006411785L, 4717017870667790060L, 4604714799421974911L, -4501484151452568168L, 4608096221904352363L, -4516266180219786672L, 4603670531417781896L, 4716613654997826540L, 4610270949742192640L, 4731048924042928986L, 4608417329084309420L, 4731322011006927522L, 4602277535749631826L, 4719836134350326370L, 4601508366397253278L, 4730196228611898306L, 4607539891476730319L, 4730158055877181546L, 4608194314976574052L, -4508570195673535552L, 4608425083311374298L, 4730537912317239550L, 4603201817101577101L, -4502367067219570268L, 4601066874519254180L, 4731331588755694248L, 4608262778492772149L, 4733293838790538245L, 4606041361297454478L, -4498867407981712788L, 4604017278549408796L, 4730245529016309632L, 4601229434347861478L, 4718892514718015268L, 4606477540329508919L, 4716388624321194080L, 4603065545151853520L, 4731067824609692236L, 4608495291005004168L, 4725777245780007146L, 4604842689461486162L, 4725265810830333375L, 4610277745879004418L, -4507136111858049692L, 4602812651486777472L, 4725142643726741859L, 4609211495603733464L, 4725852497916773595L, 4604047829423944739L, -4507523890844312644L, 4606087141863577265L, 4726548907487497282L, 4609911749243804622L, 4730715066401194724L, 4607416287852476804L, 4722851019431817379L, 4601241632302275462L, 4732716516875178967L, 4606180604248090029L, 4726506144046256070L, 4606480817862928162L, 4731882356100939002L, 4600980199494179912L, 4712256794289665056L, 4608757288627289043L, -4500532261415748968L, 4602008475994443844L, 4717915223490716356L, 4605892308781980936L, -4514380766132927152L, 4603133184309952300L, 4725770857655586230L, 4604604793373923600L, 4731147315089163926L, 4608666473348915859L, 4733082895149232329L, 4607572598496182683L, 4730845513340810898L, 4607896214758278124L, -4505042099092415872L, 4609409863700349930L, -4518858215810635744L, 4607888657473788502L, 4731354436297078357L, 4608080188721715902L, 4709035770520215200L, 4609636272316625220L, 4724956038369437288L, 4602123973585646106L, 4709529150517481520L, 4599974311658532914L, -4506954006751994688L, 4605841924601404327L, 4732950490289423307L, 4600359216341305042L, 4731997084492265070L, 4605442577117968712L, 4727415538942568116L, 4608353746769948134L, 4721683778439206253L, 4607703706727561857L, 4724820250151809209L, 4604042237692676720L, 4730456030465365438L, 4602372394395741966L, 4719992943943892123L, 4608832871796819216L, 4731256576529645469L, 4608285425505137878L, -4504339920588930280L, 4607392040710669687L, -4503282022794727584L, 4601317474951665360L, 4731799854685315330L, 4609553173148328882L, 4719711433358141962L, 4601199873874146872L, 4720365744985409588L, 4609074356749586263L, 4723644848165884654L, 4607379465242575611L, 4731648248663527744L, 4603010448783724526L, 4730127830141686052L, 4604751083259977290L, -4513173326938635904L, 4603478915084229438L, 4724891755418141329L, 4609511644478081292L, 4712530146002747968L, 4600600652989064146L, 4718631792195815504L, 4604142959045105171L, 4717834312774102632L, 4608183958006023877L, 4731106208406088901L, 4604874599212487000L, 4722929737809727786L, 4608936869858682116L, 4727749329307936050L, 4608279328013243492L, 4727498704792867922L, 4599560875542791430L, 4725003135628073477L, 4604503817997956104L, 4725275679155871916L, 4605835356414807766L, 4724513352862023159L, 4608859701597880165L, 4720472778592331978L, 4607491768231598073L, -4498848699753915088L, 4606453913850157456L, 4730623294604583599L, 4609320479820793023L, 4715354243112360392L, 4600836638281471590L, 4729945073258692912L, 4609901321907537846L, 4732140496250725187L, 4605060944445364260L, -4503742213822028720L, 4609542450102668632L, -4503290534710885308L, 4604451945165063704L, 4715991517221669072L, 4607580013087215281L, 4733299340771255620L, 4606256578842013028L, 4730192545891844988L, 4608898040030909731L, 4722610250964196259L, 4607675920606224406L, 4729880673879494948L, 4609163716965102593L, -4513966658722170496L, 4607613396184945507L, 4722143012980878755L, 4607842291209722441L, 4725566520477376562L, 4609977907845717000L, 4724909218648475481L, 4607106107173712072L, 4724659882310253028L, 4609501744499895742L, 4732597081749747960L, 4604010078640981018L, 4723858007347950920L, 4607652290347589383L, 4728634536280667642L, 4608935428759955652L, 4724622850904215150L, 4609672764347879684L, 4731039887786814958L, 4604007517398038598L, 4720600745608763310L, 4608419289108288776L, 4732587420960238874L, 4601945768982796302L, 4726533733937220771L, 4604198977274618872L, 4730197646116520882L, 4602284470178568282L, 4713000036431678144L, 4604215136868947122L, -4501839623169662824L, 4600699916701129690L, 4719859546727465497L, 4607610705320580043L, 4721580638513561883L, 4608462310118142273L, -4511643098236462304L, 4608689283747017077L, 4722938369387202999L, 4605741551347582066L, 4729652183785710025L, 4609121311235086672L, 4731744146577054979L, 4602036718739293464L, 4732136100385519431L, 4607886133191074884L, -4524471293193590208L, 4599224134842280574L, 4722106287149618839L, 4610266315024133618L, 4717093859048536916L, 4607652584099996181L, 4720203725999391522L, 4605192248317924464L, 4715621756482148688L, 4608586731223101844L, 4732686726748946730L, 4608623816765241756L, -4502267202291574508L, 4603065277978494579L, 4729923462045365380L, 4608350749988629585L, -4508705870685326816L, 4608103495907453933L, 4725808483660348284L, 4610098541403545591L, -4501552337558263496L, 4605520865831440498L, -4501420689732937688L, 4607998830740553202L, 4733065189947679548L, 4607734470092896689L, -4500388919116722356L, 4604596562126187358L, 4731762033282036606L, 4601713512186833854L, -4503514721469484924L, 4607836091127573439L, 4718899453663569956L, 4606475552518765319L, 4732076143469974217L, 4599445081747811768L, 4727347842113803524L, 4609142749439503490L, 4726337976930038153L, 4608126805350627294L, 4729291114947564537L, 4607452004730504307L, 4721757519511582960L, 4604255710315268435L, 4730862485439369882L, 4609925683403586764L, 4730187641430969168L, 4604770342692998348L, 4732045781444428249L, 4608644587443186372L, 4732293409349511679L, 4608957984937126498L, 4729712506297431902L, 4610177673450251336L, 4729824019675109192L, 4608500906529832438L, 4728930783813211324L, 4609208098291329156L, -4509076631240067800L, 4609579222661316260L, 4730138025640955186L, 4607434928152794376L, 4726747743379197843L, 4603905232858430688L, 4730346973309223212L, 4608460175678573470L, 4730260776377681960L, 4609305937679844868L, 4729145586767762815L, 4609189480754678287L, 4729694120916289133L, 4607286006279760649L, 4728260545111383084L, 4608006741160131888L, 4719760898691456572L, 4606155488740249674L, 4726959336716543467L, 4603145544173118136L, 4721593336715616814L, 4608956618626074671L, 4725435222938520354L, 4601366485032527118L, -4514916761918187904L, 4610209406398868516L, 4732208773781080771L, 4608193805569628412L, 4732505885513250398L, 4607154651009911461L, 4732544411432841600L, 4599371524277735672L, 4717779380999103864L, 4608384823444359274L, 4728549778769684561L, 4602669141979522162L, 4729904055866917683L, 4608401727308704959L, 4732675284608462541L, 4606950142435825296L, 4719800839186761936L, 4608724929941433081L, -4499027170359488414L, 4609230229359172112L, 4731083776975846716L, 4608839848860093312L, 4719574213395147336L, 4605692786437727982L, 4732826710156649202L, 4608456425736797474L, 4730501745454163632L, 4609312454330923876L, 4726595579079339151L, 4609893879793201033L, 4732898047303097329L, 4608137789450015177L, 4722196136922783171L, 4604988630538390800L, 4718270350600742584L, 4603353510114578478L, 4729208016898599188L, 4602974220793650682L, 4729250180096033396L, 4609357803548798440L, 4729809755020129177L, 4602938209937322655L, -4503822565463109420L, 4602360693863897882L, 4731967030195567316L, 4605897346663251418L, 4724982881693883255L, 4609763780861016916L, 4724432760970004487L, 4604472651051218150L, 4732387815166797095L, 4607963266333237889L, 4720171736065913079L, 4604107671343184402L, 4732703519877372747L, 4607599294656839236L, 4730656169732161576L, 4601768484972273682L, -4512095550088604680L, 4608188080541792000L, -4502623704146372232L, 4604432468791048280L, 4727456436187119227L, 4602971366098396387L, -4501231612320904652L, 4608555679460145290L, 4730010900613083632L, 4605836214664023428L, 4730733092188759688L, 4606376179257562442L, 4711375621788349080L, 4602906131968424190L, 4722984804205933454L, 4607795240491953870L, 4731599551264275724L, 4608216479582345362L, 4724625190084584384L, 4605758094392240499L, -4503993438974635392L, 4608697297098618250L, 4727706938562031006L, 4609482748814429980L, 4731909547688718992L, 4602405189564981494L, 4731136949705459984L, 4604321565703882062L, -4508187890823942624L, 4600654284376086376L, -4504398915005099056L, 4604001131204499751L, 4724367204125814608L, 4603661556613915690L, 4731451027423571674L, 4610093268203251442L, -4514024823602797696L, 4605496073195390140L, 4729589518036287514L, 4601680141487640088L, -4502357888476807952L, 4603942684390710462L, 4730646077777712002L, 4601650762320335188L, 4724582383229000670L, 4603448312341512246L, 4728679124622513194L, 4608661431646123284L, 4729647944042884238L, 4605078078269772500L, 4728464118623216770L, 4607662169495782231L, 4727058491871722978L, 4600470653132725764L, 4731168461036924172L, 4607583345399531718L, 4729991007666213856L, 4604960679104869802L, 4726137466942607224L, 4601476543813719382L, -4505719887346247116L, 4606725397361634573L, -4499657784970962160L, 4608539210654161482L, 4731054646283970697L, 4602745395324872435L, 4716165994596668448L, 4602828477840187944L, 4732099578880091462L, 4610042201944779818L, 4724814059995915570L, 4610255528818904813L, 4722778381537428411L, 4605107642376398559L, 4732103063488023082L, 4610098299610767662L, 4729786721105836278L, 4608154678584992151L, 4730955767997176032L, 4608030178961896416L, -4512300865271238296L, 4606478979095013526L, 4723721186979525020L, 4602798925931018092L, 4730211872248353182L, 4603623952987977533L, 4730053524629080910L, 4601659777829002180L, 4723796525603312644L, 4603985855308062952L, 4729440290774589896L, 4603410505291373595L, 4728529688079329256L, 4604625345657457630L, 4729147437723751636L, 4609852070530270274L, 4719790675550472840L, 4608570640252841855L, -4515141919497934656L, 4608990083165520312L, -4507909719711250968L, 4603030187202636042L, 4727711987017420213L, 4608367347321653551L, 4731395841754285992L, 4606529971289212247L, 4730561208479721206L, 4601953561209288066L, -4503459292997920964L, 4600778734417284090L, 4730524161972480760L, 4607650188196094548L, -4502277975418717496L, 4606159512561010619L, 4730545809288798433L, 4606531509848849754L, 4731521403229531426L, 4607208417664231613L, 4731228061556795250L, 4604798186369474748L, 4733174390687191638L, 4607910084487712370L, 4728241902487934274L, 4609501505070390868L, 4729264372686207874L, 4604645176386713630L, -4503045134495155372L, 4603273458462697556L, -4503742733174375260L, 4603301338152163956L, 4729506111475536340L, 4607847274267677903L, 4730531916321029164L, 4602976277693973384L, 4728730302254096201L, 4605746714444001146L, 4730771713866031104L, 4609330622882944318L, -4518401467354413728L, 4603593352325679584L, 4725886622379326539L, 4608364791488951290L, 4729806938797500514L, 4608728123351483229L, 4730485898678733202L, 4609425189166754659L, 4728933580126206624L, 4604480111839804638L, 4731540941277980555L, 4605326193867310683L, -4509650106917099600L, 4599340124356534832L, -4506682913953851484L, 4608167753233923618L, 4726171654458749386L, 4608664612223389050L, 4732410659106037892L, 4609052421633061799L, 4726264542277344540L, 4601843379988212438L, 4732639447966172941L, 4610050983326203908L, 4689446467021871104L, 4602117616176987078L, -4506783054023409520L, 4608894180539165544L, 4731900548865739165L, 4606502589363329501L, 4733068242535994818L, 4608883121269681072L, 4727793548845670385L, 4603784532801608386L, 4727683722169478682L, 4608500452061949215L, 4727973508862574076L, 4607563145672469997L, -4502364329344220832L, 4609634330824797030L, -4501254068516210224L, 4604186145354735967L, 4730098065356172360L, 4609278004260045716L, 4724588226904527869L, 4609892240886657114L, -4510078172256078432L, 4603919716780330055L, 4729399564302561993L, 4608409161910913534L, 4732959951218050105L, 4606315102515979490L, 4731843872256336187L, 4609434452737699602L, -4499543598478065024L, 4599397833231530324L, 4730812910999720590L, 4602712711799992802L, -4501301818183330588L, 4601002730704518504L, 4720830854528846534L, 4605678805289025875L, -4501858887887933620L, 4604259618940532093L, 4731484119207854489L, 4601348145419462880L, 4725214392600659926L, 4604315179349370579L, 4723332349406789041L, 4602780051194521696L, 4731879200985247362L, 4604541308516859597L, 4725919827608425714L, 4606099139483315201L, 4731080132814982796L, 4608522225896761566L, -4506392789722245848L, 4608388021564806042L, 4731251947859294890L, 4609018167747927229L, 4714617815103890272L, 4604086122585204542L, -4510241917959279432L, 4605275782529755240L, 4731853563673357510L, 4607725701633222266L, -4499589944389945280L, 4609392020951185170L, 4730182770700885848L, 4609845739470448076L, 4727909292036329899L, 4609844708484362956L, 4729444774457337026L, 4608797029397309103L, 4733309011766049776L, 4607741689306822232L, 4728820192730060144L, 4609776719994543201L, 4731451286857052323L, 4608473757238881144L, 4722636030268716793L, 4607318881560325798L, -4512033239836932392L, 4608209229367354338L, 4729065823016353693L, 4601586141281615740L, -4500685659653866204L, 4609254586979300061L, 4726722334371964454L, 4607807168240572162L, 4732906262031496197L, 4606946488650898904L, 4710425438532417520L, 4601057313313118134L, 4726594399329975545L, 4603903745875042029L, 4728003784101908936L, 4604982469914086318L, 4733335181057193650L, 4602191099714939098L, 4718542777680793952L, 4605993416540852081L, 4733326170875366257L, 4607414024147235274L, 4724027603028975836L, 4606816161563780707L, 4728122249203794972L, 4608842012162009320L, -4499932261343101608L, 4608470201109856926L, 4731056505113825464L, 4609725661388233603L, 4732166622188721778L, 4609516252475721216L, 4717928087561726104L, 4608797826375098880L, -4499009093888487984L, 4604869190632497564L, 4718928292439422196L, 4605413060260604837L, 4726787114969101773L, 4605368780485127303L, -4508030610011492700L, 4601762114506526636L, 4728896402692281184L, 4608862906080795438L, 4732371924609234106L, 4606322881701477516L, -4506785542354809388L, 4606410302823166306L, 4719995311295415946L, 4604838721357491086L, 4729476981225563434L, 4602849068616352919L, 4718013385585304584L, 4604628732025771652L, 4716411990688538464L, 4608916320826423286L, 4725410121418618242L, 4608631805518252058L, -4500826614113743228L, 4609526085788731848L, 4713378248797256816L, 4603621592690283622L, 4731061258033608412L, 4608668098463754689L, 4732781404294515944L, 4604539718238624660L, 4731683983032905746L, 4600854685472899136L, 4711363679448154808L, 4608872957941165414L, 4730756298331685594L, 4603678885338894916L, 4718542635458745932L, 4603107864930922099L, 4726221980027962625L, 4605850061184238630L, 4724001339447066745L, 4599382317457306738L, -4500491964796809248L, 4609534041557169463L, 4728905363534769388L, 4604723927201605858L, 4732699926608169513L, 4608890365489524331L, 4731520304234463790L, 4609079093099494853L, 4730107492309322748L, 4607648569825451921L, -4499902412133742956L, 4608006222628161126L, -4510047999540003664L, 4609339996383471433L, 4727578820403698163L, 4602868646951505317L, 4712131071892823024L, 4608719426491940105L, 4732650402834898665L, 4606170670101069610L, 4732540189994869225L, 4602984776359334620L, 4732323622908237091L, 4608082601963063165L, 4719200326249752556L, 4610247713035503802L, -4500145753221928748L, 4606852732288620029L, -4501498265580988360L, 4603556752167494878L, 4717866640189524096L, 4604610357793246564L, 4733203608631784189L, 4609153987396660569L, 4726910770552762891L, 4604114427883580260L, 4716101771236625320L, 4602310114735464276L, 4732153882841433009L, 4607378838876816771L, -4499442229302147348L, 4609895655647603918L, -4515023597465724192L, 4609505062141801099L, 4731194060547331983L, 4604734865493772611L, 4729319170870636729L, 4609543280214371550L, -4511221278569395088L, 4607668909962329265L, 4726948600757771286L, 4605043215110778623L, 4729960848831833198L, 4608147742631823202L, 4728868177707301248L, 4603449239632073658L, 4726731043338822830L, 4606993562736765562L, 4717603659650297840L, 4604432024638354209L, 4724759319468093824L, 4605904086130102961L, 4725267478417389945L, 4599252468587987140L, 4725748398074338859L, 4603692751848046512L, 4726216674471089621L, 4603212403155256148L, 4729127277442484560L, 4608241213784999210L, 4728957523272446862L, 4608632392998553421L, 4729369391777069792L, 4600302753103991180L, 4731011359770952736L, 4601718751172104052L, 4726276969841035825L, 4600100991954812984L, 4731588255758956296L, 4605957022646016945L, 4715474923864075500L, 4603341375955511500L, 4733265469156722878L, 4609414971444847758L, 4699162343049293120L, 4604766059463026022L, 4731755323347096828L, 4609444748660128968L, -4505928967034746644L, 4603604229762576496L, 4729461687518363279L, 4607451636819219737L, 4731922869282702774L, 4609047814697793762L, 4732218171323690950L, 4607554930198695396L, 4728900937672674143L, 4609738310019240755L, 4733162353946112666L, 4603795403148737933L, 4722592134839527336L, 4608828410204391511L, 4714927676083296952L, 4607940314659956122L, 4726119905581479828L, 4603589482992955958L, 4728369371429385686L, 4607785874037382684L, 4725673839452919372L, 4609900750420051706L, -4504458817587200588L, 4605673842899154205L, 4716630734988010076L, 4608151104486462010L, 4726060083702438708L, 4608105792056525110L, 4728694526413516558L, 4608788746661256976L, 4729327910097421838L, 4609690357760011221L, -4498760023845325494L, 4608965752264946118L, 4720760092996636314L, 4605840613781108182L, 4727685954833605839L, 4608815867487178665L, 4725421923244911955L, 4607400490375749454L, -4501942787489023064L, 4606434788618314278L, 4731524639068314556L, 4602906333885096310L, 4731487957635356500L, 4603459654713893477L, 4722990248656086538L, 4608796205951584176L, 4728389753986734716L, 4609239008770689812L, 4725600242931356654L, 4608355210805914750L, 4726135855850321189L, 4609331807171697487L, 4730774403933772410L, 4602534634244457044L, 4731258241582683744L, 4608201082925714751L, -4504755573859617888L, 4601456082747694838L, 4729752913991252102L, 4609004980954335767L, 4725763687621301647L, 4603741536242248662L, 4729849624875928103L, 4610040941119455174L, 4728680777048966879L, 4609437787126668946L, -4507312716894701904L, 4607481224060718241L, -4499431315770644820L, 4608419584133326456L, 4732441594714848970L, 4605855331547329221L, 4733311485145445449L, 4601912105604795184L, 4732185996692745500L, 4606295593431670275L, 4731019313122450682L, 4607771204238695964L, -4508417493318685376L, 4609651329333363462L, -4501061278563734008L, 4608386405724493352L, 4706032087058388544L, 4606998072503344000L, -4534609351774115328L, 4608879655573086862L, -4523274389330446848L, 4608513367228365569L, 4729958545450666302L, 4601845229028153728L, 4720737478734095539L, 4609253482351220223L, 4731830883971920837L, 4609466599400773876L, 4733168827231909558L, 4605335614043069306L, 4732570913209359850L, 4599526028037901242L, -4511768972362900144L, 4609907990902783812L, 4720747442780916190L, 4607704395834013849L, 4729297064937396369L, 4605182754948888650L, 4725464641463279255L, 4603299931173824298L, 4728583869394628948L, 4608600688081832326L, 4729779082923571778L, 4609667547228562856L, 4729444349493984037L, 4608422167208880329L, 4731563602549633721L, 4610182949299985242L, 4732213070749944641L, 4606637428101142492L, 4724681504735657867L, 4607774717846409017L, 4729626053484196333L, 4609485701282270496L, 4729693204592782975L, 4606945269259572194L, -4503981480736595212L, 4607303967106856325L, 4709392094973982720L, 4607280338795867336L, 4731358912854027369L, 4608768353361105543L, -4500692945041069020L, 4604858312146049998L, 4723746168576058081L, 4604854768585518248L, -4502868009435818648L, 4607682975587152246L, 4732493544657890366L, 4608578577254809828L, 4732264365533786173L, 4608365421762084226L, 4729210896678037931L, 4610014961775448746L, 4727314704797355729L, 4609014812969699290L, -4507274510071460844L, 4608966700732375619L, 4730914684137628276L, 4603947210700387478L, 4714960035135529416L, 4606041276111741849L, 4726830985273872223L, 4605243322039332736L, 4731894556106197632L, 4608898265950873857L, 4732655510059956293L, 4607029994990151697L, 4732521228310959966L, 4602948925511499926L, 4732846278334103429L, 4605041355194053671L, 4720440686094520374L, 4607988272756239478L, 4712389442573292496L, 4605051885951904352L, 4724994112021380570L, 4601348105929727844L, 4724388807408017579L, 4604054625405110619L, -4499065115275250040L, 4605091068171333171L, -4502584448523584292L, 4608026785337736720L, 4730599255221500821L, 4607620148624266973L, 4730512666438978518L, 4600529107280628572L, 4731806298928873832L, 4610118387045378550L, 4733338494767880003L, 4607237267878034460L, 4729398964562479006L, 4599690100279949540L, -4500937700258414888L, 4602274833900588562L, 4729991915120636952L, 4604785535491046303L, -4511745277261004576L, 4608041737661265620L, -4501229508394887876L, 4599364315708237686L, 4731303644359784659L, 4610123853791089234L, 4728982979904752976L, 4604788438861419176L, 4729597974904516988L, 4607649511710281896L, 4729518885924473456L, 4609224720120463994L, 4731521333886115072L, 4607321021009992288L, 4731822483669763180L, 4609175080629572803L, 4720851799336030985L, 4603171759871056978L, 4730986533165320038L, 4603033749329065839L, 4729741296001234676L, 4603514412023907481L, 4722758611739776117L, 4607578826420667292L, -4507873446030160684L, 4602317418954872822L, 4720459918231195753L, 4606191794343196085L, 4733355848606928514L, 4603733104069273225L, 4731938582148689104L, 4608699290281810764L, -4511790433070951616L, 4604958018188360347L, 4718032177112139144L, 4605646297658447273L, 4726957076164442904L, 4607286521565699817L, 4731304387119817389L, 4609931554216099221L, 4727477365045199118L, 4599455546448895174L, 4733094146397466791L, 4606568696826869570L, 4722837365645793778L, 4602943884716181100L, 4728910754082164488L, 4610105443450028955L, -4513660109462240624L, 4607504459588574169L, 4729863658792057164L, 4606650796688943137L, 4732684713977860843L, 4608232945429593426L, 4733230244434198304L, 4608357615766268109L, 4717016101495809896L, 4606276407065042127L, 4722016218702229405L, 4609703380702341670L, 4726882191237022742L, 4607108223764479874L, 4723952388355881615L, 4600991124114245636L, 4721147039281045152L, 4605643173756081059L, 4733082739491676548L, 4608771164968783457L, -4500115071416649020L, 4610102943088987548L, 4720649558385309134L, 4604117989168358605L, 4726781024979947711L, 4610154790318789504L, 4721317116338077432L, 4601645003699027036L, 4728826717815667858L, 4607492497866203904L, -4512058733874047104L, 4603488411395433409L, 4725139241512744192L, 4608000039275484986L, 4732958630418031203L, 4607024642329629638L, 4705792764419450688L, 4607593084410205594L, 4719457697400164108L, 4604805497430304262L, 4725866542172354202L, 4605176877335924013L, 4716215314911611504L, 4605026645545710550L, 4731908697376888460L, 4601496649634837396L, 4731922650698328628L, 4600924374134563530L, 4727153017002226373L, 4607771911648218303L, 4729840312146525312L, 4599947671216560864L, 4729157559819670327L, 4599953592597202164L, 4713075172374295632L, 4605505072835628164L, 4724304147762781536L, 4600121849782586332L, 4731389052553295902L, 4602181675279555674L, 4719823543851865170L, 4607876664908883214L, -4500886204090078364L, 4608256201280446706L, -4499004112530062642L, 4601314646327785784L, 4726956121860688220L, 4608734793575872556L, 4717071568119623572L, 4600015106375213752L, 4730852682118338650L, 4607288377754054401L, 4726326730601392256L, 4609856819574329076L, 4724724936049415890L, 4610249922884516613L, 4723920146763399284L, 4610183654386255302L, 4724895207306057399L, 4600112949476614620L, -4523302940285714048L, 4607208966305833781L, 4728200062832345368L, 4609400559281372146L, 4731790816513259730L, 4610243221433165562L, 4725676689940892468L, 4607139999782357648L, 4730794263174409236L, 4603548331024925618L, -4503507934446397200L, 4609522468850557796L, 4725761137032068913L, 4603493876670180528L, 4728922793510194720L, 4608221202175628758L, 4731604096644976626L, 4609886930820603566L, -4506313126613936240L, 4610154971873821738L, 4731303625949513718L, 4604985465697581868L, 4710932163076404296L, 4609617581274741378L, 4715922860178369868L, 4607517693651921662L, 4728626051611430766L, 4607056172874953052L, 4732446832296996104L, 4606008512319015694L, 4726002670592545904L, 4609015704392506886L, 4728168463960292170L, 4609313024195236457L, 4729605819279755600L, 4601486130212389516L, 4720368242858696957L, 4603153731406929934L, 4730547481869015574L, 4610083750047164764L, 4716963804740697544L, 4603525961647952919L, -4504445314833463768L, 4610095446514761684L, 4727764989743318052L, 4601628370684518384L, -4502937159415267284L, 4609457226190025728L, 4732784079772841591L, 4604829985987041402L, 4731727094223559422L, 4605339811903082663L, 4733127495424830072L, 4606196415556600951L, 4724441711982943191L, 4608534545614889485L, 4726514987513789470L, 4602236747453007636L, -4500357213195083064L, 4600669747596253086L, 4721985955907745037L, 4607422435167749466L, 4732116181685329204L, 4609780343950587316L, -4500034597429454428L, 4600230194468211926L, 4730595037461422752L, 4601454860216655306L, 4719421208616650420L, 4607609393794661259L, -4511914328921157888L, 4604609288646864920L, 4731996142033276960L, 4609235561161137335L, -4510050295376442760L, 4605959823437107472L, 4727778407027357832L, 4604549237825705258L, 4726341811549744758L, 4604356807405648718L, 4724286378687955659L, 4602200791590485188L, 4729980728872690842L, 4607325268456196512L, 4730080345703392432L, 4602757265196225258L, -4508679215860522104L, 4608485037483789932L, 4733277042541752001L, 4606267486661604693L, 4730240628272258890L, 4603697455753143810L, 4732238375126609366L, 4607238060826493370L, -4505655108576189768L, 4603835547180407324L, 4731615046411073128L, 4606398292529573174L, 4705299879318589664L, 4609689631158358354L, -4507524529013987840L, 4603329154667698907L, 4729994711874818432L, 4610071766467581382L, 4719162993101314324L, 4609084103080104372L, 4725904863585139381L, 4602202125423568938L, 4731863307036845404L, 4605781657955058202L, -4511827885231985968L, 4608862316457692958L, 4733189551721901778L, 4608358425933753299L, 4715322508783116536L, 4609477070934536486L, 4731612087465488342L, 4603298347343103662L, 4726525964356633204L, 4607260118203478472L, 4719875401420774124L, 4608489901963894573L, 4728920638716772400L, 4609074695640006704L, 4730294903643266734L, 4609192355255605992L, 4728700183646981442L, 4604670146923958576L, 4731819261805958694L, 4603909485315571088L, 4729402865208365037L, 4608805581786988878L, -4500792120376243040L, 4609314333062905364L, 4731176099303313599L, 4605985782627536147L, -4504702286825116480L, 4603674993277355228L, 4727427396163789207L, 4599354602991113322L, -4500151413185026688L, 4602627405624299176L, -4504689272399348980L, 4610217153918825674L, 4722656985238715876L, 4610069173871367714L, 4729025722128419061L, 4604422574453914828L, -4500228305372232828L, 4608117102696154119L, 4728573666833102662L, 4603637951009735092L, 4729028465409556245L, 4608064142700890142L, 4714552509454890944L, 4609897763973899304L, 4730099231625036732L, 4607514310530982566L, 4720022200611646406L, 4609733384823358932L, 4720070840104469247L, 4603723553809070582L, 4712701482626974016L, 4606736829064117087L, 4731428579132137444L, 4608853735561034549L, 4731311577536728566L, 4607314796233081591L, -4503331372478517512L, 4608688128325360984L, 4720325297632341808L, 4600863326220308922L, 4730337208050875873L, 4604373320758168124L, 4731702095093058857L, 4607694606517493599L, 4731013642716350563L, 4602728477422536618L, 4725510178783449062L, 4606441897569814602L, 4728646153849007091L, 4606739768063594001L, 4723844711757127392L, 4609482237764919368L, 4725509621342804282L, 4609059757263163134L, 4719497918932386536L, 4609443117104454086L, 4728665016646866387L, 4603755856091693803L};

}
