/****************************************************************************
 *
 * Copyright(c) 2005-2006, John Forkosh Associates, Inc. All rights reserved.
 * --------------------------------------------------------------------------
 * This file comprises arXivbib, which is free software. You may redistribute
 * and/or modify it under the terms of the GNU General Public License,
 * version 2 or later, as published by the Free Software Foundation.
 *      ArXivbib is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY, not even the implied warranty of MERCHANTABILITY.
 * See the GNU General Public License for specific details.
 *      By using arXivbib, you warrant that you have read, understood and
 * agreed to these terms and conditions, and that you possess the legal
 * right and ability to enter into this agreement and to use arXivbib
 * in accordance with it.
 *      Your arXivbib distribution should contain a copy of the GNU General
 * Public License.  If not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA,
 * or point your browser to  http://www.gnu.org/licenses/gpl.html
 * --------------------------------------------------------------------------
 *
 * Source:	arxivbib.c     http://www.forkosh.com/arxivbib.zip
 * Author:	John Forkosh   http://www.forkosh.com
 *
 * Purpose:   o	ArXivbib, licensed under the gpl, retrieves abstract files
 *		from arxiv.org, parses them out, and emits bibtex-readable
 *		entries.  An input file containing lines in the form
 *		   quant-ph/9812037 \\ cite=Aha:99b
 *		   math.OA/0404553 \\ cite=Kribs:04 \\ entry=book
 *		   quant-ph/0506082
 *		   etc.
 *		specifies the abstracts retrieved.
 *		Corresponding output is in the form
 *		   @article{Aha:99b,
 *		      title       = "Quantum Computation",
 *		      author      = "Dorit Aharonov",
 *		      year        =  1998,
 *		      journal     = { },
 *		         month    = "December",
 *		         eprint   = "quant-ph/9812037"  }
 *		   etc.
 *		(note that journal signals a missing but required field,
 *		i.e., the author had no Journal-ref: in his arXiv abstract)
 *	      o	Intermingled with arXiv references, your input file may
 *		also contain lines in the form
 *		   @article \\ cite=abc \\ title=whatever \\
 *			author=whomever \\ journal=wherever \\ etc
 *		which the program just reformats as a bibtex @article entry.
 *		The first line, ending with \\, signals that the next line
 *		is a continustion of the same entry.
 *	      o	arXiv entries may also be followed with \\ fields, e.g.,
 *		   quant-ph/0506082 \\ entry=book \\ ?publisher=Springer \\
 *			+comments=(not sure about publisher)
 *		combines arXiv abstract information with yours -- entry=book
 *		formats a bibtex @book entry-type.  The leading ? preceding
 *		?publisher means your information is used _only_ if the
 *		arXiv abstract contains no publisher field.  The leading +
 *		preceding +comments means your information is concatanated
 *		to any arXiv comments.  No leading character means your
 *		information replaces any information from the arXiv abstract.
 *	      o	See the Notes below for further discussion, and especially
 *		see arxivbib.html included with your distribution and also
 *		at http://www.forkosh.com/arxivbib.html
 *
 * Functions:	------ see individual functions for additional comments -----
 *		main(argc,argv)    interpret command-line args, control, etc.
 *		daemonize(void)   run as daemon (if requested with -d switch)
 *		arxivbib(bibdata)                parse an arXiv abstract file
 *		parsedate(date,bibdata)             parse date for month,year
 *		editauthors(authors)        change "a,b,c" to "a and b and c"
 *		edittitle(title)       remediate TeX-sensitive chars in title
 *		editjournal(journal,bibdata)    try to find volume,year,pages
 *		makefld(key,value)               construct "key:value" string
 *		putfld(flddata,bibdata)          store "key:value" in bibdata
 *		getindex(key)                  return bibkeys[] index for key
 *		getkey(index)                  return bibkeys[] key for index
 *		makeinfo(bibdata,infoflds)    construct "title\\authors\\etc"
 *		makebib(bibdata,file)             write bibtex-readable entry
 *		fxgets(line,maxlen,file)    fgets() but concats continuations
 *		fxputs(line,file,format)     fputs() but splits continuations
 *		findright(s,left,right)   if *s is a left delim returns right
 *		strchange(nfirst,from,to)  nfirst leading chars of from to to
 *		strreplace(string,from,to,nreplace)     change 'from' to 'to'
 *		delimreplace(string,from,to)  change matching delimiter pairs
 *		makewords(s,white)    break s into whitespace-delimited words
 *		maketokens(s,delims,left,right)           break s into tokens
 *		purge(s) remove nonprintable [n] chars produced by lynx -dump
 *		prune(s,white)         remove leading and trailing whitespace
 *		strip(s,white)            copy s with all white chars removed
 *		strlower(s)                               lowercase copy of s
 *
 * --------------------------------------------------------------------------
 * Notes:     o	Compiles and runs only under Unix and clones.  Compile as
 *		   cc arxivbib.c -o arxivbib
 *	      o	Usage: run the program either as
 *		    nohup ./arxivbib [-switches] < infile > outfile &
 *		or  ./arxivbib [-switches] -d -f infile -o outfile
 *	      o	See main() comments for command-line -switches.
 *	      o	The program sleeps 15 seconds between arXiv.org retrievals,
 *		as recommended by mjf at www-admin@arXiv.org, to avoid
 *		tripping arXiv's robot detection. This sleep delay is why
 *		nohup usage or running as daemon is suggested above.
 *	      o	As explained above, the program reads an input file like
 *		  quant-ph/0506082
 *		  hep-th/0508039
 *		  cs.LO/0508005
 *		  etc.
 *		For each line in that file it constructs a corresponding
 *		command (using the first line as an example)
 *		  lynx -dump http://arxiv.org/abs/quant-ph/0506082 > tempfile
 *		which is run via a system() call.  Then the program parses
 *		out tempfile for title, author, etc, which is reformatted
 *		and combined with the quant-ph/0506082 input, and written
 *		to an overall output file.
 * --------------------------------------------------------------------------
 * Revision History:
 * 11/30/05	J.Forkosh	Version 1.00 released.
 *
 ****************************************************************************/

/* -------------------------------------------------------------------------
standard header files
-------------------------------------------------------------------------- */
/* --- standard headers --- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
/* --- required for daemonize() --- */
#include <sys/types.h>
#include <sys/stat.h>

/* -------------------------------------------------------------------------
bibdata[] indexes for corresponding fields specified in bibkeys[] below.
-------------------------------------------------------------------------- */
/* --- first two fields are for @entry-type and \cite{key}  --- */
#define	BIBENTRY	1		/* "article", etc */
#define	CITEKEY		(BIBENTRY+1)	/* defaults to "quant-ph/0506082" */
/* --- standard entry fields from tlc2, page 765 --- */
#define	ADDRESS		(CITEKEY+1)	/* Address= */
#define	ANNOTE		(ADDRESS+1)	/* Annote= */
#define	AUTHOR		(ANNOTE+1)	/* Author= or arXiv Authors: */
#define	AUTHORS		(AUTHOR)	/* arXiv Authors: or Author= */
#define	BOOKTITLE	(AUTHOR+1)	/* Booktitle= */
#define	CHAPTER		(BOOKTITLE+1)	/* Chapter= */
#define	CROSSREF	(CHAPTER+1)	/* Crossref= */
#define	EDITION		(CROSSREF+1)	/* Edition= */
#define	EDITOR		(EDITION+1)	/* Editor= */
#define	HOWPUBLISHED	(EDITOR+1)	/* Howpublished= */
#define	INSTITUTION	(HOWPUBLISHED+1) /* Institution= */
#define	JOURNAL		(INSTITUTION+1)	/* Journal= or arXiv Journal-ref: */
#define	JOURNALREF	(JOURNAL)	/* arXiv Journal-ref: or Journal= */
#define	KEY		(JOURNAL+1)	/* Key= (not the one for \cite{}) */
#define	MONTH		(KEY+1)		/* Month= */
#define	NOTE		(MONTH+1)	/* Note= or arXiv Comments: */
#define	COMMENTS	(NOTE)		/* arXiv Comments: or Note= */
#define	NUMBER		(NOTE+1)	/* Number= */
#define	ORGANIZATION	(NUMBER+1)	/* Organization= */
#define	PAGES		(ORGANIZATION+1) /* Pages= */
#define	PUBLISHER	(PAGES+1)	/* Publisher= */
#define	SCHOOL		(PUBLISHER+1)	/* School= */
#define	SERIES		(SCHOOL+1)	/* Series= */
#define	TITLE		(SERIES+1)	/* Title= or arXiv lines after Date*/
#define	TYPE		(TITLE+1)	/* Type= */
#define	VOLUME		(TYPE+1)	/* Volume= */
#define	YEAR		(VOLUME+1)	/* Year= */
/* --- some of the more common additional fields --- */
#define	AFFILIATION	(YEAR+1)	/* Affiliation= */
#define	ABSTRACT	(AFFILIATION+1)	/* Abstract= or arXiv lines */
#define	CONTENTS	(ABSTRACT+8)	/* Contents= note room for abstract*/
#define	COPYRIGHT	(CONTENTS+1)	/* Copyright= */
#define	ISBN		(COPYRIGHT+1)	/* ISBN= */
#define	ISSN		(ISBN+1)	/* ISSN= */
#define	KEYWORDS	(ISSN+1)	/* Keywords= (not bibtex sort key=)*/
#define	LANGUAGE	(KEYWORDS+1)	/* Language= */
#define	LCCN		(LANGUAGE+1)	/* LCCN= */
#define	LOCATION	(LCCN+1)	/* Location= */
#define	MRNUMBER	(LOCATION+1)	/* MRnumber= */
#define	PRICE		(MRNUMBER+1)	/* Price= */
#define	SIZE		(PRICE+1)	/* Size= */
#define	URL		(SIZE+1)	/* Url= or just http: */
/* --- additional arXiv abstract fields from http://arxiv.org/help/prep --- */
#define	DATE		(URL+1)		/* Date: or Date...: on arXiv */
#define	REPORTNO	(DATE+1)	/* Report-no: */
#define	SUBJCLASS	(REPORTNO+1)	/* Subj-class: */
#define	DOI		(SUBJCLASS+1)	/* DOI: */
#define	MSCCLASS	(DOI+1)		/* MSC-class: */
#define	ACMCLASS	(MSCCLASS+1)	/* ACM-class: */
/* --- additional fields --- */
#define	LASTCHECKED	(ACMCLASS+1)	/* Lastchecked= */
#define	EPRINT		(LASTCHECKED+1)	/* Eprint= */
#define	FTP		(EPRINT+1)	/* ftp= or just ftp: */
/* --- new fields in data will be added after pre-defined ones above --- */
#define	HIGHINDEX	(FTP)		/* highest pre-defined index */
static	int highindex = HIGHINDEX;	/* bumped for new fields in input */

/* -------------------------------------------------------------------------
bibdata[index] where value will be stored according to key in key:value field,
e.g., title=Quantum Computation\\author=C.H.Bennett  puts the strings
Quantum Computation in bibdata[TITLE]  and  C.H.Bennett in bibdata[AUTHOR]
-------------------------------------------------------------------------- */
#define	MAXKEYS 512			/* max #keys in bibkeys[] */
#define	flddef  struct flddef_struct	/* typedef for flddef struct */
flddef {
  char	*key;				/* e.g., "Title:", "Authors:", etc */
  int	index;				/* corresponding index from above */
  } ; /* --- end-of-struct flddef --- */
static flddef bibkeys[MAXKEYS] = {
  /* -----------------------------------------------------------------------
  first two fields are for @entry-type and \cite{key}
  ------------------------------------------------------------------------ */
  /* --- bibentry is "@article" or "article", etc  --- */
  { "Entry=",		BIBENTRY },
  { "Entrytype=",	BIBENTRY },	/* synonym/alias for entry */
  { "Entry-type=",	BIBENTRY },	/* synonym/alias for entry */
  { "Style=",		BIBENTRY },	/* synonym/alias for entry */
  /* --- identifier is the \cite{key}, i.e., @article{identifier, etc} --- */
  { "Cite=",		CITEKEY },	/* key used in \cite{key} */
  { "Citekey=",		CITEKEY },	/* synonym/alias for cite */
  { "Identifier=",	CITEKEY },	/* synonym/alias for cite */
  /* -----------------------------------------------------------------------
  standard entry fields from tlc2, page 765
  ------------------------------------------------------------------------ */
  { "Address=",		ADDRESS },
  { "Annote=",		ANNOTE },
  { "Author=",		AUTHOR },
  { "Booktitle=",	BOOKTITLE },
  { "Chapter=",		CHAPTER },
  { "Crossref=",	CROSSREF },
  { "Edition=",		EDITION },
  { "Editor=",		EDITOR },
  { "Howpublished=",	HOWPUBLISHED },
  { "Institution=",	INSTITUTION },
  { "Journal=",		JOURNAL },
  { "Key=",		KEY },
  { "Month=",		MONTH },
  { "Note=",		NOTE },
  { "Number=",		NUMBER },
  { "Organization=",	ORGANIZATION },
  { "Pages=",		PAGES },
  { "Publisher=",	PUBLISHER },
  { "School=",		SCHOOL },
  { "Series=",		SERIES },
  { "Title:",		TITLE },
  { "Type=",		TYPE },
  { "Volume=",		VOLUME },
  { "Year=",		YEAR },
  /* -----------------------------------------------------------------------
  some of the more common additional fields
  ------------------------------------------------------------------------ */
  { "Affiliation=",	AFFILIATION },
  { "Abstract=",	ABSTRACT },
  { "Contents=",	CONTENTS },
  { "Copyright=",	COPYRIGHT },
  { "ISBN=",		ISBN },
  { "ISSN=",		ISSN },
  { "Keywords=",	KEYWORDS },
  { "Language=",	LANGUAGE },
  { "LCCN=",		LCCN },
  { "Location=",	LOCATION },
  { "MRnumber=",	MRNUMBER },
  { "Price=",		PRICE },
  { "Size=",		SIZE },
  { "URL=",		URL },
  /* -----------------------------------------------------------------------
  additional arXiv abstract fields from http://arxiv.org/help/prep
  ------------------------------------------------------------------------ */
  { "Date:",		DATE },
  { "Authors:",		AUTHOR },
  { "Comments:",	NOTE },
  { "Journal-ref:",	JOURNAL },
  { "Report:",		REPORTNO },
  { "Report-no:",	REPORTNO },
  { "Subj-class:",	SUBJCLASS },
  { "DOI:",		DOI },
  { "MSC-class:",	MSCCLASS },
  { "ACM-class:",	ACMCLASS },
  { "Abstract:",	ABSTRACT },
  /* -----------------------------------------------------------------------
  additional fields
  ------------------------------------------------------------------------ */
  { "Lastchecked=",	LASTCHECKED },
  { "Eprint=",		EPRINT },
  { "http:",		URL },
  { "ftp:",		FTP },
  { "Cross-ref=",	CROSSREF },
  { NULL, -1 } /* trailer signalling end-of-table */
 } ; /* --- end-of-bibkeys[] --- */

/* -------------------------------------------------------------------------
bibliography entry-types (article, book, etc -- add your own or modify these)
-------------------------------------------------------------------------- */
#define	bibentry  struct bibentry_struct  /* typedef for bibentry struct */
bibentry {
  char	*name;				/* e.g., "book" or "article", etc. */
  flddef fields[256];		/* e.g, {{"author",AUTHOR},...,{NULL,-1}} */
  } ; /* --- end-of-struct bibentry --- */
static	bibentry bibentrys[] = {
  /* --- article entry-type --- */
  { "xxxarticle",     {	/* --- article fields (required and optional) --- */
			{ "title", TITLE },
			{ "author", AUTHOR },
			{ "journal", JOURNAL },
			{ "year", YEAR },
			{ "optional", -1 }, /* optional fields... */
			{ "key", KEY },
			{ "volume", VOLUME },
			{ "number", NUMBER },
			{ "month", MONTH },
			{ "pages", PAGES },
			{ "note", NOTE },
			{ "eprint", EPRINT },
			/* --- additional (ignored) fields --- */
			{ "date", DATE },
			{ "report", REPORTNO },
			{ "Subj-class", SUBJCLASS },
			{ "DOI", DOI },
			{ "MSC-class", MSCCLASS },
			{ "ACM-class", ACMCLASS },
			{ "publisher", PUBLISHER },
			{ "crossref=", CROSSREF },
			{ "url", URL },
			{ "ftp", FTP },
			{ "abstract", ABSTRACT },
			{NULL,-1}}  },
  /* --- book entry-type --- */
  { "xxxbook",	     {	/* --- book fields (required and optional) --- */
			{ "title", TITLE },
			{ "year", YEAR },
			{ "publisher", PUBLISHER },
			{ "optional", -1 }, /* optional fields... */
			{ "author", AUTHOR }, /*author _or_ editor required*/
			{ "editor", EDITOR }, /*I made them _both_ optional*/
			{ "key", KEY },
			{ "volume", VOLUME },
			{ "number", NUMBER },
			{ "series", SERIES },
			{ "address", ADDRESS },
			{ "edition", EDITION },
			{ "month", MONTH },
			{ "note", NOTE },
			{NULL,-1}}  },
  /* --- you can use this as a "template" --- */
  { "entrytype",     {	/* --- entrytype fields (required and optional) --- */
			{ "title", TITLE },
			{ "author", AUTHOR },
			{ "optional", -1 }, /* optional fields... */
			{ "key", KEY },
			{ "type", TYPE },
			{ "note", NOTE },
			{NULL,-1}}  },
  /* --- end-of-"template" --- */
  { NULL, {{NULL,-1}} }	/* trailer signalling end-of-table */
 } ; /* --- end-of-bibentrys[] --- */

/* -------------------------------------------------------------------------
additional arxivbib information
-------------------------------------------------------------------------- */
/* --- arxivbib-specific strings --- */
#define	BIBCOMMENT "%"			/* bibtex comment char */
#define	COMMENT "#!" BIBCOMMENT		/* # or ! also signals comment */
#define	FLDDELIM "\\\\"			/* fields separated by \\ */
#define	ANDDIRECTIVES "+-"		/* concat value to existing field */
#define	ORDIRECTIVES "|\?"		/* use value only if field empty */
#define	DIRECTIVES (ANDDIRECTIVES ORDIRECTIVES) /* any directive */
/* --- user-definable #define'd compile-line parameters --- */
#ifndef	COMMAND
  #define COMMAND "lynx -dump http://arxiv.org/abs/" /* arXiv query */
#endif
#ifndef XFILE
  #define XFILE "/tmp/arxivbib.out"	/* temp file for arXiv abstract */
#endif
#ifndef	UNKNOWN
  #define UNKNOWN " "			/*value for missing required fields*/
#endif
#ifndef	UNPUBLISHED
  #define UNPUBLISHED "unpublished"	/* entry-type if no Journal-ref: */
#endif
#ifndef	MAXFLDLEN
  #define MAXFLDLEN 990			/* maximum bibtex field length */
#endif
/* --- declare char bibdata[MAXFLDS][DATAFLDLEN] --- */
#define	MAXFLDS 192			/* max # of data fields */
#define	DATAFLDLEN 2048			/* max # of chars per data field */

/* -------------------------------------------------------------------------
additional macros
-------------------------------------------------------------------------- */
/* --- some additional general-purpose macros --- */
#define	WHITESPACE " \a\b\t\n\r\f\v"
#define	GRAYSPACE "<([{.,:;_=+-*/|~!@#$%^&\?\'\"\\}])>"
#define	DIGITS "0123456789"
#define	LETTERS "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ"
#define	isthischar(thischar,accept) ( accept==NULL || thischar=='\0'? 0 : \
	( *(accept)!='\000' && strchr((accept),(thischar))!=(char *)NULL ) )
#define	isthisword(thisword,accept) ( accept==NULL || thisword==NULL? 0 : \
	(strspn(thisword,accept)==strlen(thisword)) )
#define	iswhite(thischar) isthischar((thischar),(WHITESPACE))
#define	isgray(thischar) ( iswhite(thischar) \
	|| isthischar((thischar),(GRAYSPACE)) )
/* --- control variables for testing/debugging --- */
#ifdef TEST
  #define ISTESTING 1
#else
  #define ISTESTING 0
#endif
static	int istesting = ISTESTING;	/*true to _not_ issue system() calls*/
static	int isediting = 1;		/* true to apply field-level edits */
static	int editlevel = 0;		/* edit if isediting>=editlevel */
static	int msglevel = 0;		/* debugging message level */

/* ==========================================================================
 * Function:	int main ( int argc, char *argv[] )
 * Purpose:	retrieve abstract files from arxiv.org,
 *		parse them out for bibliographic data,
 *		abd emit it as bibtex-readable entries.
 * --------------------------------------------------------------------------
 * Command-line Arguments:
 *		---------------------- usual switches -----------------------
 *		-d		Run as daemon (in which case both the -f
 *				and the -o switches must also be given).
 *		-f inputfile	Open inputfile (instead of default stdin).
 *		-o outputfile	Open outputfile (instead of default stdout).
 *		------------------- additional switches ---------------------
 *		-sn		Sleep n seconds between arXiv.org queries.
 *				Default 15, suggested by www-admin@arXiv.org,
 *				avoids tripping arXiv's robot detection.
 *				This sleep delay is why nohup usage or
 *				running as daemon is recommended above.
 *		-tn		Output format n=0 (-t0) will, for input
 *				line quant-ph/0506082 write a *single*
 *				output line of the form
 *				  quant-ph/0506082 # title \\ authors \\ etc
 *				Format n=1 (-t1) writes *multiple* output
 *				lines of the form
 *				  quant-ph/0506082 # title \\
 *					authors \\
 *					etc
 *				(See Notes below for further info.)
 *		-rn		Reformatting option simply reads input file
 *				in either -t0 or -t1 formats (single or
 *				multiple lines), and rewrites output file
 *				in format n (0=single, 1=multiple lines).
 *				No arxiv.org data is collected; it's just
 *				a reformatting run.  If you use -r0 on an
 *				already single-line input file, then the
 *				output file is an exact copy of the input.
 *				Similarly for -r1 on multiple-line input.
 * --------------------------------------------------------------------------
 * Returns:	( int )		EXIT_SUCCESS (always)
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
int	main ( int argc, char *argv[] )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
/* --- input/output --- */
FILE	*inpfile = stdin,		/* -f inputfile or default stdin */
	*outfile = stdout;		/* -o outputfile or default stdout */
char	inpline[32768], *fxgets(), *prune(), /* line from input file */
	comment[16384],			/* comment fields on line */
	outline[32768],			/* line to output file */
	*dataflds[512],			/* ptrs to \\ fields in input line */
	*makefld(),			/* construct "key:value" string */
	*strlower();			/* lowercase a string */
int	ndataflds = 0;			/* number of \\ fields found */
int	strreplace();			/* no @'s in comment fields */
int	putfld();			/* store input into bibdata[] */
static	char bibdata[MAXFLDS][DATAFLDLEN]; /* bibliographic data fields */
char	*info=NULL, *makeinfo(),	/* construct "title\\author\\etc" */
	entry[512] = "article",		/* bib entry, default article */
	unpublished[512] = UNPUBLISHED;	/* or default if no Journal-ref: */
int	fxputs(),			/* write output in format... */
	format = 2;			/* 0=1line 1=\\break >1=bibtex tab */
int	makebib();			/* format bibtex-readbale entry */
/* --- control --- */
char	*notarxiv = "@";		/* non-arXiv input line prefixes */
int	processtype = 1;		/* 1=arXiv 2=notarxiv 0=reformat */
int	argnum = 0,			/* argv[] index */
	isreformat = 0,			/* -r for reformat-only run */
	isbibentry = 1,			/* true for bibtex, >1 forces type */
	nsleepsecs = 15,		/* -s #secs between arXiv lookups */
	daemonize(), isdaemon=0;	/* -d to daemonize process */
/* --- arXiv abstract data and control --- */
char	*commandprefix = COMMAND,	/* arXiv query prefix */
	command[2048];			/* prefix + "input > abstractfile" */
int	arxivbib();			/* parse abstractfile from arXiv */
/* -------------------------------------------------------------------------
interpret command-line args
-------------------------------------------------------------------------- */
while ( argc > ++argnum )		/* check each command-line arg */
 if ( *argv[argnum] == '-' ) {		/* flush arg if not a -switch */
   char *type  = argv[argnum] + 1;	/* first char after - is switch */
   char *value = argv[argnum] + 2;	/* which may be followed by value */
   int  vallen = strlen(type) - 1;	/* #chars comprising value */
   if ( vallen < 1 )			/* just a single char after - */
    if ( argc > argnum+1 )		/* there's an argv[] after -switch */
     if ( *(argv[argnum+1]) != '-' )	/* and it's not the next -switch */
      {	value = argv[++argnum];		/* so it's the value for this one */
	vallen = strlen(value); }	/* reset val length */
   if ( vallen >= 0 )			/* we have a -switch character */
    switch ( tolower(*type) ) {		/* process each type of -switch */
      default: break;			/* flush unrecognized -switch */
      case 'r':				/* -r for reformat */
	isreformat = 1;   format = 1;	/* set flag and fall through */
      case 't':				/* -t sets output format */
	if ( vallen < 1 ) {		/* don't have any value */
	  format = 1;			/* so reset default format */
	  break; }			/* and break */
	if ( !isthisword(value,DIGITS) ) /* which isn't numeric */
	  strcpy(entry,value);		/* so it's our default bib entry */
	else {				/* have a numeric argument */
	  format = atoi(value);		/* so it's our format number */
	  isbibentry = (format==2?1:0);	/* set true for bibtex output */
	  if ( /*isbibentry &&*/ argc>argnum+1 ) /* there's another arg */
	   if ( *(argv[argnum+1]) != '-' ) /*and it's not the next -switch*/
	    { argnum++;			/* so it's our entry */
	      strcpy(entry,argv[argnum]); } } /* replace default entry */
	break;
      case 's':				/* -s sets seconds to sleep */
	if ( vallen > 0 ) nsleepsecs = atoi(value);  break;
      case 'm':				/* -m sets message level */
	if ( vallen > 0 ) msglevel = atoi(value);  break;
      case 'f':				/* -f sets input file */
	if ( vallen > 0 ) inpfile = fopen(value,"r");  break;
      case 'o':				/* -o sets output file */
	if ( vallen > 0 ) outfile = fopen(value,"w");  break;
      case 'u':				/* -u sets unpublished entry-type */
	if ( vallen > 0 ) strcpy(unpublished,value);  break;
      case 'd':				/* -d to run as daemon */
	isdaemon = 1;  break;		/* no value required */
      case 'x':				/* -x for testing (no arXiv call) */
	istesting = 1;			/* no value required */
	if ( vallen > 0 ) istesting = atoi(value);  break;
      case 'e':				/* -e for field editing */
	isediting = 0;			/* no value required */
	if ( vallen > 0 ) isediting = atoi(value);  break;
      } /* --- end-of-switch() --- */
   } /* --- end-of-if(*argv[argnum]=='-') --- */
/* --- data and validity checks --- */
while ( isthischar(*entry,"+") ) {	/* leading + forces this entry */
  strcpy(entry,entry+1);		/* squeeze out directive */
  if ( isbibentry ) isbibentry++; }	/* signal forced entry */
/* -------------------------------------------------------------------------
check files and daemonize process
-------------------------------------------------------------------------- */
if ( outfile==NULL || inpfile==NULL ) goto end_of_job; /* quit if no file */
if ( isdaemon ) {			/* user wants to run as daemon */
  if ( outfile==stdout || inpfile==stdin ) goto end_of_job;
  else if ( !daemonize() ) goto end_of_job; } /* daemonize process */
/* -------------------------------------------------------------------------
read "query strings" from inpfile till eof
-------------------------------------------------------------------------- */
while ( fxgets(inpline,16000,inpfile) != NULL ) { /* read input till eof */
  /* -----------------------------------------------------------------------
  allocations and declarations
  ------------------------------------------------------------------------ */
  char	*leftptr=inpline, *rightptr=NULL; /* ptr to delims on input line */
  int	inplen = 0;			/* input line length */
  int	iscomment = 0;			/* true just copies line to output */
  if ( msglevel>=9 ) { fprintf(stderr,"%s\n",inpline); } /* for debugging */
  /* --- also init all data fields --- */
  editlevel = 999;			/* reset editlevel (no edits) */
  for ( ndataflds=0; ndataflds<MAXFLDS; ndataflds++ ) /* for bibdata[] */
    *(bibdata[ndataflds]) = '\000';	/* init fields as empty strings */
  if ( isbibentry ) strcpy(bibdata[BIBENTRY],entry); /* set default entry */
  *comment = '\000';			/* no input comments yet */
  info = NULL;				/* no makeinfo() output string yet */
  /* -----------------------------------------------------------------------
  prune whitespace from input line and check for comment, @string, etc
  ------------------------------------------------------------------------ */
  /* --- first prune the input string --- */
  prune(inpline,NULL);			/*prune leading/trailing whitespace*/
  /* --- check for comment lines, etc --- */
  if ( isthischar(*inpline,COMMENT) ) {	/* entire line is a comment */
    if ( 0 && isbibentry ) {		/* comment on bibtex entry */
      *inpline = *(BIBCOMMENT); }	/* bibtex needs a bibcomment */
    strreplace(inpline,"@","\"at\"",0);	/* remove @'s from comments */
    iscomment = 1; }			/* and signal a comment line */
  if ( memcmp(strlower(inpline),"@string",7) == 0 /* have a string */
  ||   memcmp(strlower(inpline),"@preamble",9) == 0 ) /* or a preamble */
    iscomment = 1;			/* so just output it immediately */
  /* --- ouput @string, comment lines, etc, immediately --- */
  if ( iscomment ) {			/* have a comment, @string, etc */
    strcat(inpline,"\n");		/* restore trailing newline */
    fxputs(inpline,outfile,100);	/* emit comment */
    continue; }				/* and get next line */
  /* --- store trailing comments --- */
  if ( 0 ) {				/*false to ignore trailing comments*/
    inplen = strcspn(inpline,COMMENT);	/* #chars preceding comment char */
    strcpy(comment,inpline+inplen);	/* copy comment char and comment */
    inpline[inplen] = '\000'; }		/* replace comment char with '\0' */
  strcpy(outline,inpline);		/*init output as input minus comment*/
  prune(inpline,NULL);			/* then re-prune() input line */
  /* --- check for empty lines, comment lines, and reformat-only runs  --- */
  processtype = 1;			/* init signal for arXiv abstract */
  if ( *inpline == '\000'		/* but if just have an empty line */
  ||   isthischar(*inpline,COMMENT)	/* or just a comment line */
  ||   (isreformat && !isbibentry ) )	/* or this is just a reformat run */
    processtype = 0;			/* then don't process input line */
  /* -----------------------------------------------------------------------
  separate input line into \\-separated fields
  ------------------------------------------------------------------------ */
  ndataflds = 0;			/* no \\ data fields found yet */
  if ( processtype != 0 )		/* if we're processing this line */
   while ( 1 ) {			/* separate input into \\ fields */
    int	fldlen = 0;			/* #chars preceding next \\ */
    if ( (rightptr=strstr(leftptr,FLDDELIM)) /* look for next \\ */
    !=   NULL ) *rightptr = '\000';	/* terminate this field */
    fldlen = strlen(leftptr);		/* length of this field */
    prune(leftptr,NULL);		/* and prune it */
    dataflds[ndataflds++] = leftptr;	/* set ptr to start of this field */
    if ( msglevel>=9 ) { fprintf(stderr,"%d) \"%s\"\n",ndataflds,leftptr); }
    if ( rightptr == NULL ) break;	/* we're done after last field */
    leftptr += (fldlen+strlen(FLDDELIM)); } /* start of next field */
  /* -----------------------------------------------------------------------
  set process type (check for non-arXiv prefixes)
  ------------------------------------------------------------------------ */
  if ( processtype != 0 )		/* if we're processing this line */
   if ( isthischar(*inpline,notarxiv) )	/* have a leading @ */
    { processtype = 2;			/* so reset process type */
      if ( isbibentry ) strcpy(bibdata[BIBENTRY],inpline+1); } /*and entry*/
  if ( msglevel>=9 ) { fprintf(stderr,"processtype=%d\n",processtype); }
  /* -----------------------------------------------------------------------
  process default line through arXiv
  ------------------------------------------------------------------------ */
  if ( processtype == 1			/* no special prefix found */
  &&   istesting != 1			/* no arxiv input if exactly 1 */
  &&   !isreformat ) {			/* and not just reformatting */
    /* --- set default identifier in output data --- */
    putfld(makefld("Identifier:",dataflds[0]),bibdata);
    putfld(makefld("Eprint:",dataflds[0]),bibdata); /*default eprint:, too*/
    /* --- issue command to request arXiv abstract file --- */
    strcpy(command,commandprefix);	/* start command with prefix */
    strcat(command,dataflds[0]);	/* input is, e.g., quant-ph/0506082*/
    strcat(command," > ");		/* redirect output */
    strcat(command,XFILE);		/* to temporary file for parsing */
    if ( !istesting )			/* submit to arXiv if not testing */
     { if ( nsleepsecs > 0 ) sleep(nsleepsecs); /* throttle requests */
       system(command); }		/* request abstract file */
    /* --- get arxiv info from abstract file --- */
    editlevel = 1;			/* set editlevel for arxiv data */
    arxivbib(bibdata);			/* parse out abstract file */
    /* --- set default bibliography entry type --- */
    if ( isbibentry == 1 ) {		/* true, but entry not forced */
      if ( *(bibdata[JOURNAL]) == '\000' ) /* no Journal-ref: in abstract */
	strcpy(bibdata[BIBENTRY],unpublished); /* so it's unpublished */
      else				/* author provided Journal-ref: */
	strcpy(bibdata[BIBENTRY],entry); } /* so assume article */
    } /* --- end-of-if(processtype==1) --- */
  /* -----------------------------------------------------------------------
  finish arXiv records, and process non-arXiv input
  ------------------------------------------------------------------------ */
  if ( processtype==1 || processtype>1 ) { /* areXiv or non-arXiv input */
    /* --- allocations and declarations --- */
    int ifld=0, fld1=(processtype==1?1:1); /* dataflds[] index, 1st field */
    /* --- insert explicit input data --- */
    editlevel = 1; /*2;*/		/* set editlevel for user data */
    if ( ndataflds > fld1 )		/* have fields on input */
     for ( ifld=fld1; ifld<ndataflds; ifld++ ) /* for each field */
      putfld(dataflds[ifld],bibdata);	/* store it in bibdata */
    /* --- construct output line --- */
    info = makeinfo(bibdata,NULL);	/* all bibdata fields on info line */
    if ( *info != '\000' ) {		/* got infoflds[] */
      strcpy(outline,dataflds[0]);	/* begin with first data field */
      strcat(outline," \\\\ ");		/* followed by field separator */
      strcat(outline,info);		/* and data fields */
      if (*comment != '\000') strcat(outline," "); } /*blank before comment*/
    } /* --- end-of-if(processtype>=1) --- */
  /* -----------------------------------------------------------------------
  generate output
  ------------------------------------------------------------------------ */
  /* --- emit bibtex entry --- */
  if ( isbibentry ) {			/* bibtex-readable entry requested */
    if ( isbibentry>2 || (processtype==1 && isbibentry>1) )
      strcpy(bibdata[BIBENTRY],entry);	/* force entry */
    makebib(bibdata,outfile); }		/* write bibliography on outfile */
  /* --- or emit single-line output (i.e., non-bibtex) --- */
  if ( !isbibentry ) {			/* single-line output requested */
    strcat(outline,comment);
    strcat(outline,"\n");		/* terminate output with newline */
    fxputs(outline,outfile,format); }	/* write it to outfile */
  fflush(outfile);			/*flush stream (in case of segfault)*/
  } /* --- end-of-while(fgets) --- */
end_of_job:
  if ( inpfile!=NULL && inpfile!=stdin ) fclose(inpfile);
  if ( outfile!=NULL && outfile!=stdout) fclose(outfile);
  exit(EXIT_SUCCESS);
} /* --- end-of-function main() --- */

/* ==========================================================================
 * Function:	int daemonize ( void )
 * Purpose:	runs process as daemon
 * --------------------------------------------------------------------------
 * Arguments:	-none
 * --------------------------------------------------------------------------
 * Returns:	( int )		1=success, 0=failure
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
int	daemonize ( void )
{
/* --- allocations and declarations --- */
int	status = 0;			/* init to signal failure */
pid_t	pid, sid;			/* process, session id */
/* --- fork off the parent process --- */
if ( (pid = fork()) < 0 ) goto end_of_job; /* failed to fork */
if ( pid > 0 ) exit(EXIT_SUCCESS);	/* exit parent process */
/* --- change the file mode mask --- */
umask(0);
/* --- create a new sid for the child process --- */
if ( (sid = setsid()) < 0 ) goto end_of_job;
/* --- change the current working directory --- */
if ( chdir("/") < 0 ) goto end_of_job;
/* --- close out the standard file descriptors --- */
close(STDIN_FILENO);
close(STDOUT_FILENO);
close(STDERR_FILENO);
/* --- back to caller as daemon --- */
status = 1;				/* signal success to caller */
end_of_job:
  return status;
} /* --- end-of-function daemonize() --- */

/* ==========================================================================
 * Function:	char *arxivbib ( char bibdata[][DATAFLDLEN] )
 * Purpose:	parses an arXiv abstract file produced by a command
 *		of the form
 *		  lynx -dump http://arXiv.org/abs/quant-ph/0512001 > XFILE
 *		and returns bibliographic information for that article.
 * --------------------------------------------------------------------------
 * Arguments:	bibdata (O)	char bibdata[][DATAFLDLEN] returns
 *				null-terminated character strings for
 *				date, title, authors, etc, in the fields
 *				indexed by DATE, TITLE, AUTHOR, etc.
 * --------------------------------------------------------------------------
 * Returns:	( int )		1
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
int	arxivbib ( char bibdata[][DATAFLDLEN] )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
char	inpline[4096],			/* line from abstract file */
	*prune(), *purge(), *strlower(), /* clean up abstract file input */
	*makefld();			/* construct "key:value" string */
int	putfld(),			/* store value in bibdata[] */
	parsedate();			/* store year and month from date */
char	anystub[2048] = "\000",		/*any stub may occur after Authors:*/
	anyfld[16384] = "\000";		/* buffer to accumulate any field */
int	isstub = 0,			/* set true if stub found on line */
	isblock = 0,			/* true to signal title block, etc */
	infostep = 0;			/* "state" is stepstub[] index */
FILE	*xfile = fopen(XFILE,"r");	/* open abstract file for read */
static	char *arxivstubs[] = {		/* recognized arxiv.org stubs */
	"author", "date", "journal", "report", "comment",
	"subj",   "msc",  "acm",     "doi",     NULL } ;
/* -------------------------------------------------------------------------
read and process lines from abstract file till eof
-------------------------------------------------------------------------- */
if ( xfile == NULL ) goto end_of_job;	/* failed to open abstract file */
while ( fgets(inpline,4090,xfile) != NULL ) /* read next line till eof */
  {
  if ( msglevel>=99 ) { fprintf(stderr,"%s\n",inpline); } /*debugging output*/
  /* --- prune leading and trailing whitespace --- */
  prune(inpline,NULL);			/* remove leading,trailing space */
  /* --- check line for date stub --- */
  isstub = 0;				/* init to signal no stub on line */
  if ( infostep < 2 )			/* only check for Date */
    isstub = (strstr(inpline,"Date")==NULL?0:1); /* true if Date on line */
  /* --- process line depending on current state --- */
  switch ( infostep ) {			/* infostep is current "state" */
    default: break;			/* oops, internal error */
    case 0:				/*haven't found first Date line yet*/
      if ( !isstub )			/* still haven't found it */
	break;				/* so keep looking for it */
      infostep = 1;			/* bump state and fall through */
      *anystub = *anyfld = '\000';	/* init field buffer for next step */
    case 1:				/*we're within block of Date lines*/
      if ( isstub ) {			/* found another Date line */
	char *paren= strrchr(inpline,'('); /* last ( before file size */
	char *colon= NULL;		/* : after Date or version */
	char *year = NULL;		/* check for year in 1980-2029 */
	if (paren != NULL ) *paren = '\000'; /*don't confuse size with year*/
	if ( year == NULL ) year = strstr(inpline,"198"); /* or in 1980's */
	if ( year == NULL ) year = strstr(inpline,"199"); /* or in 1990's */
	if ( year == NULL ) year = strstr(inpline,"200"); /* or in 2000's */
	if ( year == NULL ) year = strstr(inpline,"201"); /* or in 2010's */
	if ( year == NULL ) year = strstr(inpline,"202"); /* or in 2020's */
	if ( year != NULL )		/* found year */
	  *(year+4) = '\000';		/* null-terminate date after year */
	if ( (colon = strchr(inpline,':')) /* find : after Date or version */
	==   NULL )			/* weird if no colon, so just... */
	  colon = strstr(inpline,"Date")+strlen("Date"); /*...assume Date:*/
	putfld(makefld("Date",colon+1),bibdata); /* store Date */
	parsedate(colon+1,bibdata);	/* try to store month and year */
	break; }			/* now look for a later Date line */
      /* --- found first line after Date: line(s) --- */
      infostep = 2;			/* bump state to accumulate title */
      *anystub = *anyfld = '\000';	/* init field buffer for next step */
      isblock = 0;			/* fall thru for 1st title line*/
    case 2:				/* collect title line(s) */
      if ( strlen(inpline) > 1 ) {	/* if not an empty title line */
	isblock = 1;			/* signal we started title block */
	if ( strlen(anyfld) > 0 )	/* this is a continuation line */
	  strcat(anyfld," ");		/* so precede it with a space */
	strcat(anyfld,purge(inpline)); } /* add title line to title */
      else				/* empty line */
	if ( isblock )			/* signals end of title block */
	 { putfld(makefld("Title",anyfld),bibdata); /* store Title */
	   infostep = 3;		/*bump state for author or any stub*/
	   *anystub = *anyfld = '\000';	/* init field buffer for next step */
	   isblock = 0; }		/* haven't started next block yet */
      break;
    case 3:				/* collect author till next stub: */
      /* --- first check for empty line preceding abstract text --- */
      if ( strlen(inpline) < 1 ) {	/* empty line */
	if ( isblock )			/* signals end of stub block */
	 { if ( *anyfld != '\000' )	/* have last field in block */
	    putfld(makefld(anystub,anyfld),bibdata); /* store last field */
	   infostep = 4;		/* abstract follows empty line */
	   *anystub = *anyfld = '\000';	/* init field buffer for next step */
	   isblock = 0; } }		/* haven't started next block yet */
      else {
	/* --- see if line has a leading stub: --- */
	int nonwhitelen = strcspn(inpline,WHITESPACE); /*#leading non-white*/
	int istub = 0;			/* arxivstubs[] index */
	char *arxivstub=NULL, newstub[2048]; /* stub signals end of field */
	memcpy(newstub,inpline,nonwhitelen); /* copy first token */
	newstub[nonwhitelen] = '\000';	/* and null-terminate it */
	isblock = 1;			/* signal we've started block */
	isstub = 0;			/* default signal continuation line*/
	if ( nonwhitelen > 0 )		/*should be >0 due to prior prune()*/
	 if ( newstub[nonwhitelen-1] == ':' ) /* token ends with a colon */
	  for ( istub=0; (arxivstub=arxivstubs[istub]) != NULL; istub++ )
	   if ( strstr(strlower(newstub),arxivstub) != NULL ) /*found stub*/
	    { isstub = 1;		/* so signal a new arxiv stub */
	      break; }			/* no need to look any further */
	if ( isstub )			/* found a new arxiv stub */
	 { char *fldptr = inpline+strlen(newstub); /*ptr to data after stub*/
	   if ( *anyfld != '\000' )	/* have preceding field in block */
	    putfld(makefld(anystub,anyfld),bibdata); /* store field */
	   strcpy(anystub,newstub);	/* save stub till field accumulated*/
	   strcpy(anyfld,purge(prune(fldptr,NULL))); } /*get 1st fld in line*/
	if ( !isstub )			/* accumulate continuation line */
	 { strcat(anyfld," ");		/* add whitespace to preceding line*/
	   strcat(anyfld,purge(inpline)); } } /* followed by field data */
      break;
    case 4:				/* collect abstract till blank line */
      /* --- first check for empty line preceding abstract text --- */
      if ( strlen(inpline) < 1 ) {	/* empty line */
	if ( isblock )			/* signals end of stub block */
	 { if ( *anyfld != '\000' )	/* have an abstract */
	    putfld(makefld("abstract",anyfld),bibdata); /* store abstract */
	   infostep = 999;		/* no more stubs to process*/
	   *anystub = *anyfld = '\000';	/* init field buffer for next step */
	   isblock = 0; } }		/* haven't started next block yet */
      else { /* --- collecting abstract data till nexy blank line --- */
	if ( strlen(anyfld) < 16000 ) {	/* don't accumulate too much */
	 if ( isblock ) strcat(anyfld," "); /* add space to preceding line */
	 strcat(anyfld,purge(inpline));	} /* followed by field data */
	isblock = 1; }			/* signal we've started block */
      break;
    } /* --- end-of-switch() --- */
  if ( infostep > 99 ) break;		/* check state for completion */
  } /* --- end-of-while(fgets) --- */
end_of_job:
  if ( xfile!=NULL && xfile!=stdin ) fclose(xfile); /* close abstract file */
  return ( 1 );				/* back to caller */
} /* --- end-of-function arxivbib() --- */

/* ==========================================================================
 * Function:	int parsedate ( char *date, char bibdata[][DATAFLDLEN] )
 * Purpose:	parse date and store month,year in bibdata[].
 * --------------------------------------------------------------------------
 * Arguments:	date (I)	char *ptr to null-terminated string
 *				containing date string
 *		bibdata (O)	char bibdata[][DATAFLDLEN] returns
 *				with null-terminated character strings
 *				in MONTH and YEAR fields
 * --------------------------------------------------------------------------
 * Returns:	( int )		1
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
int	parsedate ( char *date, char bibdata[][DATAFLDLEN] )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
char	*word=NULL, **words=NULL, **makewords(); /* break date into tokens */
int	iword=0, wordlen=0;		/* word=words[] index, strlen(word)*/
char	*strlower();			/* for case-insensitive check */
char	*makefld();			/* construct "key:value" string */
int	putfld();			/*store value in bibdata[] by key*/
static	char *months[] = { "January","February","March","April","May","June",
	"July","August","September","October","November","December",NULL };
static	char *years[] = { "198","199","200","201","202",NULL };
/* -------------------------------------------------------------------------
check each token comprising date for month and year
-------------------------------------------------------------------------- */
words = makewords(date,GRAYSPACE);	/* first separate date into tokens */
if ( words != NULL )			/* check for returned ptr */
 for ( iword=0; (word=words[iword])!=NULL; iword++ ) { /* check each word */
   if (msglevel>=29) { fprintf(stderr,"\tword#%d) \"%s\"\n",iword+1,word); }
   /* --- could this be a month? --- */
   if ( (wordlen=strlen(word)) >= 3 )	/* only if at least 3 chars long */
    if ( isthisword(word,LETTERS) ) {	/* and if it's all letters */
     int imonth = 0;			/* months[] index */
     for ( imonth=0; months[imonth]!=NULL; imonth++ ) /* check each month */
      if ( memcmp(strlower(months[imonth]),strlower(word),wordlen) == 0 )
       { putfld(makefld("Month:",months[imonth]),bibdata);
	 if (msglevel>=29) { fprintf(stderr,"\tmonth#%d\n",imonth+1); }
	 break; } }			/* don't bother searching further */
   /* --- could this be a year? --- */
   if ( (wordlen=strlen(word)) == 4 )	/* only if exactly 4 chars long */
    if ( isthisword(word,DIGITS) ) {	/* and if it's all digits */
     int iyear = 0;			/* years[] index */
     for ( iyear=0; years[iyear]!=NULL; iyear++ ) /*check each year prefix*/
      if ( memcmp(years[iyear],word,3) == 0 ) /* found a match */
       { putfld(makefld("Year:",word),bibdata); /* store year */
	 break; } }			/* don't bother searching further */
   } /* --- end-of-for(iword) --- */
return ( 1 );				/* back to caller */
} /* --- end-of-function parsedate() --- */

/* ==========================================================================
 * Function:	char *editauthors ( char *authors )
 * Purpose:	change "a,b,c" to "a and b and c"
 * --------------------------------------------------------------------------
 * Arguments:	authors (I)	char *ptr to null-terminated string
 *				containing original authors string
 * --------------------------------------------------------------------------
 * Returns:	( char * )	edited author string
 * --------------------------------------------------------------------------
 * Notes:     o	a static buffer containing the edited authors is returned,
 *		not a separately malloc()'ed string.
 *	      o	Commas within {}'s, ()'s, etc are _not_ replaced.
 *	      o	we handle  author, jr  (and  author, phd  etc),
 *		but  author, jr, phd  gets messed up.
 * ======================================================================= */
/* --- entry point --- */
char	*editauthors ( char *authors )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
static	char edited[4096];		/* edited author string */
char	*edittitle();			/* title edit applied to authors */
char	*token=NULL, **ptokens=NULL, **maketokens(), /* get authors */
	*delims = ",",			/* separated by commas */
	*left="{\"([<", *right="}\")]>"; /* but not commas within delims */
char	*strlower(), *strip(), *strchange(), /* check for ", Jr.", etc. */
	*nameptr = NULL,		/* ptr to John for "John, Jr." */
	*titles[] = { "jr", "iii", "phd", "esq", NULL };
int	itoken = 0;			/* ptokens[] token/author index */
*edited = '\000';			/* init edited authors */
/* -------------------------------------------------------------------------
break up original string
-------------------------------------------------------------------------- */
ptokens = maketokens(authors,delims,left,right);
/* -------------------------------------------------------------------------
construct string with "and" between all authors
-------------------------------------------------------------------------- */
if ( ptokens != NULL )			/* got author tokens */
 for ( itoken=0; (token=ptokens[itoken])!=NULL; itoken++ ) /*for each author*/
  { int	isand = (itoken>0?1:0);		/* true to put "and" before token */
    int toklen=strlen(token), nprefix=0; /* #chars in token, #before comma */
    int ititle=0, istitle=0;		/*Jr,PhD,etc titles[] index, signal*/
    /* --- remove embedded commas --- */
    if ( strcspn(token,left) < toklen )	/* have a bracket */
     while ( (nprefix=strcspn(token,delims)) < toklen ) /*with commas in it*/
      *(token+nprefix) = ' ';		/* replace next comma with space */
    /* --- check if this token is a title like Jr or PhD --- */
    if ( itoken > 0 )			/*don't check first token for title*/
     for ( ititle=0; titles[ititle]!=NULL; ititle++ ) /*check for Jr, Ph.D.*/
      if ( strcmp(strlower(strip(token,".")),titles[ititle]) == 0 )
       { char *lastname = strrchr(nameptr,' '); /* space before last name */
	 if ( lastname != NULL ) lastname++; /*ptr to 1st char of last name*/
	 else lastname = nameptr;	/*have to enclose {entire name, Jr}*/
	 isand = 0;			/* no "and" before ", Jr." */
	 strcat(edited,", ");		/* restore original comma instead */
	 strchange(0,lastname,"{");	/* opening { precedes author's name*/
	 istitle = 1;			/* signal a title token */
	 break; }			/* no need to look further */
    /* --- place "and" between authors, and tack on next author --- */
    if ( isand ) strcat(edited," and "); /* put " and " between tokens */
    nameptr = edited + strlen(edited);	/* ptr to first char of this token */
    strcat(edited,token);		/* and add in author token */
    if ( istitle ) strcat(edited,"}"); } /* closing } after title */
if(0) strcpy(edited,edittitle(edited));	/* apply title edit to authors */
return ( edited );			/* back with edited authors */
} /* --- end-of-function editauthors() --- */

/* ==========================================================================
 * Function:	char *edittitle ( char *title )
 * Purpose:	tries to remediate TeX-specific chars like & and _
 * --------------------------------------------------------------------------
 * Arguments:	title (I)	char *ptr to null-terminated string
 *				containing original title string
 * --------------------------------------------------------------------------
 * Returns:	( char * )	edited title string
 * --------------------------------------------------------------------------
 * Notes:     o	a static buffer containing the edited title is returned,
 *		not a separately malloc()'ed string.
 *	      o	The goal of this function is to edit arXiv data,
 *		particularly abstract and comments/notes (and title),
 *		so that bibtex and latex run as smoothly as possible.
 *		Getting them both to run error-free, but without drastically
 *		constraining arXiv data, appears to be impractical,
 *		so some hand-tweaking of arXivbib's output is probably
 *		unavoidable.
 *	      o	The ad hoc "rules" below are under constant development
 *		to try to approach the above goals as closely as possible.
 * ======================================================================= */
/* --- entry point --- */
char	*edittitle ( char *title )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
static	char edited[16384];		/* edited title string */
char	*token=NULL, **tokens=NULL, **maketokens(); /*break title into words*/
char	*prune(), *strchange();		/* change & to and outside math mode*/
char	*delimreplace();		/* delimiter replacement */
char	*mathchars = /*"_^";*/ "_^\\";	/* chars requiring math mode */
int	strreplace(),			/* string replacement */
	itoken = 0,			/* tokens[] index */
	ifrom = 0;			/* from[] index */
/* -------------------------------------------------------------------------
edit control strings
-------------------------------------------------------------------------- */
/* --- change from-strings to to-strings according to edit level --- */
int	nreplaces = 2,			/* up to editlevel */
	mathreplace = nreplaces;	/* index for math mode replacements*/
char	*replace[][2][16] = {		/* [editlevel][from,to][15strings] */
	  { {    "&",     "#", "\\\"", "\\\'"  /*,"{","}"*/, NULL },
	    {" and ", " No. ", "\000", "\000"   /*,"(",")"*/  } } ,
	  { {    "&",     "#", "\\\""     /*,"{","}"*/, NULL },
	    {" and ", " No. ", "\\ddot "  /*,"(",")"*/  } } ,
	  /* --- math mode replacement strings --- */
	  { {     "#", "\\\"",    "\"",  NULL },
	    { " No. ", "\\ddot ", "\'\'" } } ,
	  { {NULL}, {NULL} }
	 } ; /* ---end-of-replace --- */
char	**from=replace[0][0], **to=replace[0][1]; /* default to first */
/* --- false math mode \signals --- */
char	*frommath[] = { "\\\"", "\\\'", "\\`",  "\\emph", "\\em",
	"\\use",  NULL },
	*totext[]   = { "\000", "\000", "\000", "\\emph", "\\em",
	"use"  };
/* --- delimiter replacement strings --- */
char	*fromdelims[] = { "\"", "\"", NULL },
	*todelims[]   = { "``", "\'\'" };
/* -------------------------------------------------------------------------
initialization
-------------------------------------------------------------------------- */
strcpy(edited,title);			/* default to unedited copy */
if ( isediting < editlevel ) goto end_of_job; /* no editing */
if ( isediting >= 9 ) {			/* extreme editing */
  strreplace(edited,"\\\"","\000",0);	/* just can't deal with \"s */
  strreplace(edited,"\"","\000",0); }	/* or with unbalanced "s */
tokens = maketokens(edited,WHITESPACE,"${\"","$}\""); /*get tokens in title*/
if ( tokens == NULL ) goto end_of_job;	/* probably an internal error */
/* -------------------------------------------------------------------------
check each title word, but not within $...$'s
-------------------------------------------------------------------------- */
for ( itoken=0; (token=tokens[itoken]) != NULL; itoken++ ) { /* each token */
 int ismathmode = 1;			/* init signal as math mode token */
 from = replace[0][0];  to = replace[0][1]; /* default replacement strings */
 /* --- check current token for math mode --- */
 if ( strchr(token,'$') == NULL ) {	/* token not in math mode */
  if ( strcspn(token,mathchars) < strlen(token) ) { /* should be math mode */
   int nfalse = 0;			/* #false math signals found */
   for ( ifrom=0; frommath[ifrom]!=NULL; ifrom++ ) /* check for false math */
    nfalse += strreplace(token,frommath[ifrom],totext[ifrom],0);
   if ( nfalse > 0 ) ismathmode = 0;	/* wasn't really math mode */
   else					/* looks like real \math */
    { strchange(0,token,"$");		/* so put on a leading $ */
      strcat(token,"$"); } }		/* and a trailing $, too */
  else ismathmode = 0; }		/* reset math mode signal */
 /* --- handle math mode tokens --- */
 if ( !ismathmode ) {			/* we have a text mode token */
  if ( isediting <= nreplaces )		/* use requested from/to strings */
   { from = replace[isediting-1][0];	/* set target from strings */
     to = replace[isediting-1][1]; } }	/* and replacement to strings */
 else {					/* we have a math mode token */
  strchange(0,token,"{");		/* so put on a leading { */
  strcat(token,"}");			/* and a trailing }, too */
  from = replace[mathreplace][0];	/* set math mode from strings */
  to = replace[mathreplace][1]; }	/* and replacement to strings */
 /* --- string replacements --- */
 for ( ifrom=0; from[ifrom]!=NULL; ifrom++ ) /* replace each from string */
  strreplace(token,from[ifrom],to[ifrom],0); /* by its to equivalent */
 delimreplace(token,fromdelims,todelims); /* change "..." to ``...'' */
 strreplace(token,"\"","\000",0);	/* purge any remaining "s */
 /* --- add edited word to title --- */
 if ( itoken == 0 ) *edited = '\000';	/* init string before first token */
 else strcat(edited," ");		/* separate tokens with one space */
 strcat(edited,prune(token,NULL));	/*add pruned edited token to string*/
 } /* --- end-of-for(itoken) --- */
end_of_job:
  return ( edited );			/* back with edited title */
} /* --- end-of-function edittitle() --- */

/* ==========================================================================
 * Function:	char *editjournal (char *journal, char bibdata[][DATAFLDLEN])
 * Purpose:	tries to find volume,year,pages,etc embedded in journal
 * --------------------------------------------------------------------------
 * Arguments:	journal (I)	char *ptr to null-terminated string
 *				containing original journal string
 *		bibdata (O)	char bibdata[][DATAFLDLEN] returns
 *				with volume,year,pages,etc filled in
 *				if that information was found in journal
 * --------------------------------------------------------------------------
 * Returns:	( char * )	edited journal string
 * --------------------------------------------------------------------------
 * Notes:     o	a static buffer containing the edited title is returned,
 *		not a separately malloc()'ed string.
 *	      o	if volume,year,pages,etc is found in original journal,
 *		that information is stored where it belongs in bibdata
 *		and stripped from the edited journal string.
 *	      o	information is stored in bibdata only if it's not
 *		already populated, i.e., existing data won't be replaced
 * ======================================================================= */
/* --- entry point --- */
char	*editjournal ( char *journal, char bibdata[][DATAFLDLEN] )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
static	char edited[4096];		/* edited journal string */
char	*edittitle();			/* title edit applied to journal */
/*char	*token=NULL, **tokens=NULL, **maketokens();*/ /* break into tokens */
strcpy(edited,journal);			/* default to unedited copy */
strcpy(edited,edittitle(edited));	/* apply title edit to journal */
return ( edited );			/* back with edited journal */
} /* --- end-of-function editjournal() --- */

/* ==========================================================================
 * Function:	char *makefld ( char *key, char *value )
 * Purpose:	just returns a string consisting of "key : value"
 * --------------------------------------------------------------------------
 * Arguments:	key (I)		char *ptr to null-terminated string
 *				containing "key", e.g., "Volume"
 *		vlaue (I)	char *ptr to null-terminated string
 *				containing "value", e.g., "35"
 * --------------------------------------------------------------------------
 * Returns:	( char * )	"key : value" string
 *				or empty string for error
 * --------------------------------------------------------------------------
 * Notes:     o	the address of a static buffer is returned,
 *		not an individually malloc()'ed string
 * ======================================================================= */
/* --- entry point --- */
char	*makefld ( char *key, char *value )
{
/* --- allocations and declarations --- */
static	char fldstring[16384];		/* "key:value" buffer for caller */
char	*delim = ":";			/* check if key already contains : */
/* --- check args --- */
*fldstring = '\000';			/* init as empty string */
if ( key==NULL || value==NULL ) goto end_of_job; /* signal error */
/* --- copy key and add : if it doesn't already have one --- */
strcat(fldstring,key);			/* copy key to "key:value" string */
if ( strchr(fldstring,*delim) == NULL )	/* no : delim in key */
  strcat(fldstring,delim);		/* so add it ourselves */
/* --- add value at end of string --- */
strcat(fldstring,value);		/* add value to end of "key:value" */
end_of_job:
  return ( fldstring );			/* back to caller with "key:value" */
} /* --- end-of-function makefld() --- */

/* ==========================================================================
 * Function:	int putfld ( char *flddata, char bibdata[][DATAFLDLEN] )
 * Purpose:	parse flddata containing "key:value" and place
 *		value in corresponding bibdata[] slot.
 * --------------------------------------------------------------------------
 * Arguments:	flddata (I)	char *ptr to null-terminated string
 *				containing "key:value", e.g., "Volume:35"
 *		bibdata (O)	char bibdata[][DATAFLDLEN] returns
 *				with null-terminated character string
 *				containing value in slot for key.
 * --------------------------------------------------------------------------
 * Returns:	( int )		bibdata[] index value stored in,
 *				or -1 for error
 * --------------------------------------------------------------------------
 * Notes:     o	if the key is of the form +key or &key, then its value is
 *		concatanated with any existing data (and separated by a blank)
 *	      o	if |key or ?key, then its value is used _only_ if there's
 *		no existing value; otherwise, it's discarded
 * ======================================================================= */
/* --- entry point --- */
int	putfld ( char *flddata, char bibdata[][DATAFLDLEN] )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
int	index=(-1), getindex();		/* flds[].index with flddata key */
int	isand=0,isor=0,isdirective=0,isval=0; /*default replaces current val*/
int	keylen = 0;			/* #chars preceding delim */
char	key[2048],			/* key preceding : or = in flddata */
	*aliaskey=NULL, *getkey(),	/* preferred key for index of key */
	editbuff[16384],		/* buffer for local field edits */
	*delims = ":=",			/* delims separating key and value */
	*valptr=NULL, value[4096],	/* value following : in flddata */
	*prune(), *purge();		/* clean up string */
char	*editauthors(),			/* edit author string */
	*edittitle(),			/* edit title string */
	*editjournal();			/* edit journal string */
/* -------------------------------------------------------------------------
first separate out key and value, and check leading char of key for directive
-------------------------------------------------------------------------- */
/* --- find first delim separating key:value --- */
if ( (keylen=strcspn(flddata,delims))	/* #leading chars preceding delim */
>=  strlen(flddata) ) goto end_of_job;	/* signal error if no delim found */
else { /* --- separate key and value --- */
  memcpy(key,flddata,keylen);		/* local copy of key chars */
  key[keylen] = '\000';			/* null-terminate local key */
  strcpy(value,flddata+keylen+1);	/* local copy of value chars */
  prune(key,NULL); prune(value,NULL); }	/* prune only */
  /*purge(prune(key,NULL)); purge(prune(value,NULL)); }*/ /*prune and purge*/
if ( *key == '\000' ) goto end_of_job;	/* signal error if empty key */
/* --- check leading character of key for a directive --- */
if ( isthischar(*key,ANDDIRECTIVES) ) isand=1; /* concatanate to field */
if ( isthischar(*key,ORDIRECTIVES) )  isor=1;  /* use only if field empty */
if ( (isdirective = (isand||isor)) )	/* true if any durective found */
  { strcpy(key,key+1);			/* squeeze out directive */
    prune(key); }			/* and re-prune() key */
/* -------------------------------------------------------------------------
locate _given_ key in bibkeys[]
-------------------------------------------------------------------------- */
/* --- find index of _given_ key --- */
if ( (index = getindex(key))		/* get bibdata[] index for key */
< 0 ) goto end_of_job;			/* flush field if index negative */
if ( index >= MAXFLDS ) goto end_of_job; /* or if past end of bibdata[] */
/* -------------------------------------------------------------------------
see if bibkeys[] specifies a preferred "alias" key for this index
-------------------------------------------------------------------------- */
/* --- find _first_ key in bibkeys[] with same index as _given_ key --- */
if ( *(aliaskey = getkey(index)) != '\000' ) { /* preferred key found */
  ; }
/* -------------------------------------------------------------------------
apply field-level edits
-------------------------------------------------------------------------- */
valptr = value;				/* default to unedited string */
if ( isediting ) {			/* but we're applying field edits */
  /* --- external editing functions --- */
  if ( index == AUTHOR )     valptr = editauthors(value); /* edit authors */
  if ( index == TITLE )      valptr = edittitle(value);   /* edit title */
  if ( index == REPORTNO )   valptr = edittitle(value);   /* edit report */
  if ( index == ABSTRACT )   valptr = edittitle(value);   /* edit abstract */
  if ( index == NOTE )       valptr = edittitle(value);   /* edit comments */
  if ( index == JOURNAL )    valptr = editjournal(value,bibdata); /*journal*/
  /* --- internal edits --- */
  if ( index==URL || index==FTP )	/* check for leading http: key */
   if ( strncmp(value,"//",2) == 0 ) {	/* leading http: key stripped */
     strcpy(editbuff,			/* so start with... */
	(index==URL?"http:":"ftp:"));	/* ...http: or ftp: */
     strcat(editbuff,value);		/* concatanate trailing part of url*/
     valptr = editbuff; }		/* and use it as value */
  } /* --- end-of-if(isediting) --- */
/* -------------------------------------------------------------------------
store value in bibdata[] and return to caller
-------------------------------------------------------------------------- */
isval = (*(bibdata[index])=='\000'?0:1); /* true if value currently exists */
if ( isand ) {				/* concatanate to current data */
 if (*valptr!='\000' ) {		/* but only if we have more data */
  int maxcat = DATAFLDLEN - strlen(bibdata[index]) - 2; /*#chars remaining*/
  if ( maxcat > 1 ) {			/* room for space plus 1 data char */
   if ( isval ) strcat(bibdata[index]," "); /*need space after current value*/
   strncat(bibdata[index],valptr,maxcat); } } } /* concat additional data */
else					/* not "and" */
 if ( !isor || !isval ) {		/* not "or", or no current value */
  strncpy(bibdata[index],valptr,DATAFLDLEN); } /*just store value in bibdata*/
bibdata[index][DATAFLDLEN-1] = '\000';	/*make sure data's null-terminated*/
end_of_job:
  return ( index );			/*bibdata[] index containing value*/
} /* --- end-of-function putfld() --- */

/* ==========================================================================
 * Function:	int getindex ( char *key )
 * Purpose:	find key in bibkeys[] and return corresponding index
 *		into bibdata[]
 * --------------------------------------------------------------------------
 * Arguments:	key (I)		char *ptr to null-terminated string
 *				containing bibkeys[] key
 *				whose index is wanted
 * --------------------------------------------------------------------------
 * Returns:	( int )		bibdata[] index
 *				or -1 for error
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
int	getindex ( char *key )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
flddef	*keys = bibkeys;		/* default to use bibkeys[] */
int	ikey=0, highkey=0,		/* flds[] index, highest index */
	keylen = (key==NULL?0:strlen(key)), /* length of input key */
	index = (-1);			/* flds[].index for matching key */
int	iscase = 0;			/* signal case-(in)sensitive match */
char	keybuff[1024], *prune(),*strlower(); /* for case-insensitive match */
char	*leftdelims = "([{<";		/* terminate key at any left delim */
char	*newkey = NULL;			/* malloc() for new key, if needed */
/* -------------------------------------------------------------------------
check for leading directive character in key
-------------------------------------------------------------------------- */
if ( keylen < 1 ) goto end_of_job;	/* no input from caller */
if ( isthischar(*key,DIRECTIVES) ) key++; /* skip 1st char if a directive */
strcpy(keybuff,(iscase?key:strlower(key))); /* local copy of key */
keybuff[strcspn(keybuff,leftdelims)] = '\000'; /*terminate key at left delim*/
prune(keybuff,NULL);			/*remove leading/trailing whitespace*/
if ( (keylen = strlen(key)) < 1 ) goto end_of_job; /*no remaining key chars*/
/* -------------------------------------------------------------------------
locate key in bibkeys[]
-------------------------------------------------------------------------- */
for ( ikey=0; ; ikey++ ) {		/* run thru bibkeys[] till NULL */
  char	*keyfld = keys[ikey].key;	/* field we're currently checking */
  if ( keyfld == NULL ) break;		/* signal error if key not matched */
  highkey = ikey;			/* highest key index with data */
  /*if ( strstr((iscase?keyfld:strlower(keyfld)),*/ /* look in keyfld */
  /*(iscase?key:strlower(key))) != NULL )*/ /* for caller's key */
  if ( memcmp((iscase?keyfld:strlower(keyfld)), /* look in keyfld */
  keybuff,keylen) == 0 )		/* for caller's key */
    { index = keys[ikey].index;		/* ikey has index that matches key */
      break; }			
  } /* --- end-of-for(ikey) --- */
/* -------------------------------------------------------------------------
add new key to table
-------------------------------------------------------------------------- */
if ( index < 0 )			/* new key if lookup failed */
 if ( highkey < MAXKEYS-1		/* we have room for new key */
 &&   highindex < MAXFLDS-1 )		/* and room for its data */
  if ( (newkey = (char *)malloc(strlen(key)+8)) /* allocate key memory */
  !=  NULL )				/* and if malloc() succeeded */
   { highkey++;  highindex++;		/* so bump highkey and highindex */
     keys[highkey].key = newkey;	/* set ptr to new key's memory */
     keys[highkey+1].key = NULL;	/* and set new terminating NULL */
     strcpy(newkey,key);		/* set new key in place */
     strcat(newkey,":");		/* and give it (unnecessary) : */
     keys[highkey].index = highindex;	/* data for new key in bibdata[] */
     index = highindex; }		/* and return it to caller */
end_of_job:
  return ( index );			/* back to caller with index or -1 */
} /* --- end-of-function getindex() --- */

/* ==========================================================================
 * Function:	char *getkey ( int index )
 * Purpose:	find index in bibkeys[] and return corresponding key string
 * --------------------------------------------------------------------------
 * Arguments:	index (I)	int containing bibdata[] index
 *				whose key is wanted
 * --------------------------------------------------------------------------
 * Returns:	( char * )	key corresponding to bibdata[] index
 *				or empty string "\000" if index not found
 * --------------------------------------------------------------------------
 * Notes:     o	if bibkeys[] has several keys for the same index,
 *		the first one found is returned
 * ======================================================================= */
/* --- entry point --- */
char	*getkey ( int index )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
static	char keybuff[2048];		/* key buffer returned to caller */
flddef	*keys = bibkeys;		/* default to use bibkeys[] */
int	ikey = 0,			/* flds[] index */
	iscase = 0;			/* true for case-sensistive key */
char	*key = NULL,			/* flds[ikey].key */
	*delims = ":=",			/* terminate key at any delim */
	*strlower();			/* for case-insensitive key */
/* -------------------------------------------------------------------------
locate index in bibkeys[]
-------------------------------------------------------------------------- */
*keybuff = '\000';			/* init as empty string */
if ( index >= 0 )			/* don't even look up bad index */
 for ( ikey=0; (key = keys[ikey].key) != NULL; ikey++ ) /* run thru flds[] */
  if ( index == keys[ikey].index ) break; /* until ikey has matching index */
if ( key != NULL ) {			/* found key */
 strcpy(keybuff,(iscase?key:strlower(key))); /* make a copy for caller */
 keybuff[strcspn(key,delims)] = '\000'; } /* null-terminate key at delim */
return ( keybuff );			/* back to caller with key or "\0" */
} /* --- end-of-function getkey() --- */

/* ==========================================================================
 * Function:	char *makeinfo ( char bibdata[][DATAFLDLEN], int *infoflds )
 * Purpose:	returns a string containing "title \\ authors \\ date"
 *		fields from bibdata[], as specified in infoflds[]
 * --------------------------------------------------------------------------
 * Arguments:	bibdata (I)	char bibdata[][DATAFLDLEN] containing
 *				null-terminated character strings for
 *				date, title, authors, etc, in the fields
 *				indexed by DATE, TITLE, AUTHOR, etc.
 *		infoflds (I)	array of ints, terminated by -1 trailer,
 *				specifying fields returned in info string,
 *				or NULL to return all populated fields
 * --------------------------------------------------------------------------
 * Returns:	( char * )	string containing "title \\ authors \\ date"
 *				fields as specified in infoflds[]
 * --------------------------------------------------------------------------
 * Notes:     o	the address of a static buffer is returned,
 *		not an individually malloc()'ed string
 * ======================================================================= */
/* --- entry point --- */
char	*makeinfo ( char bibdata[][DATAFLDLEN], int *infoflds )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
static	char info[32768];		/* string returned to caller */
int	iinfo = 0,			/* infoflds[] index (if given) */
	index = 0;			/* bibdata[] index for info string */
char	*key=NULL, *getkey(),		/* key for index */
	*value = NULL;			/* value for index */
/* -------------------------------------------------------------------------
combine parsed fields into returned info string
-------------------------------------------------------------------------- */
*info = '\000';				/* init info as empty string */
for ( iinfo=0; ; iinfo++ ) {		/* infoflds[] till end-of-table */
  index = (infoflds==NULL?iinfo:infoflds[iinfo]); /* next field for info */
  if ( index < 0			/* infoflds[] trailer found */
  ||   index >= MAXFLDS ) break;	/* or end of bibdata[] found */
  key = getkey(index);			/* key for this field */
  value = bibdata[index];		/* data for this field */
  if ( *value == '\000' ) continue;	/* ignore empty slot */
  if ( *info != '\000' ) strcat(info," \\\\ "); /*first add field separator*/
  if ( *key != '\000' ) strcat(info,key); /* followed by key if we have one */
   else sprintf(info+strlen(info),"field%d",index); /* or dummy key if not */
  strcat(info," = ");			/* add = delimiter */
  strcat(info,value);			/* and value */
  } /* --- end-of-for(iinfo) --- */
return ( info );			/* back to caller with info string */
} /* --- end-of-function makeinfo() --- */

/* ==========================================================================
 * Function:	int makebib ( char bibdata[][DATAFLDLEN], FILE *file )
 * Purpose:	write a bibtex-readable entry on file containing bibdata.
 * --------------------------------------------------------------------------
 * Arguments:	bibdata (I)	char bibdata[][DATAFLDLEN] containing
 *				null-terminated character strings for
 *				date, title, authors, etc, in the fields
 *				indexed by DATE, TITLE, AUTHOR, etc.
 *		file (I)	FILE *ptr to stream on which bibtex-readable
 *				entry will be written
 * --------------------------------------------------------------------------
 * Returns:	( int )		always 1
 * --------------------------------------------------------------------------
 * Notes:     o	bibdata[BIBENTRY] should contain "article", "book", etc,
 *		specifying the bibliography entry to write
 * ======================================================================= */
/* --- entry point --- */
int	makebib ( char bibdata[][DATAFLDLEN], FILE *file )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
char	entry[512] = "article",		/* bibentry from bibdata */
	*entryname = NULL,		/* entry from bibentrys[] */
	*key=NULL, *getkey(),		/* bibdata[CITEKEY] */
	*strlower();			/* do case-insensitive match */
int	ientry = 0,			/* bibentrys[] index with entry */
	ifield = 0,			/* bibfields[] index */
	index=(-1), getindex(),		/* bibdata[] index for bibfields key*/
	valtype = (-1),			/* 0=number 1=single word, 2=other */
	keytype = 1,			/* true for non-dummy key */
	isrequired = 1;			/* true for required fields */
flddef	*bibfields = NULL;		/* fields comprising entry type */
char	keybuff[512] = "\000",		/* buffer for dummy key, if needed */
	valbuff[16384], *prune(),	/* buffer for continuation field */
	*fldkey = keybuff,		/* bibfields[ifield].key or dummy */
	*fldval = valbuff,		/* bibdata[getindex(fldkey)] data */
	*dummydata = UNKNOWN;		/* for missing but required field */
char	outline[16384],			/* buffer for output lines */
	*stub=NULL, *hdrstub="  ",	/*stub indent selected, header stub*/
	*reqstub="     ", *optstub="        ", /* required, optional stubs */
	*leftdelims[]  = {  "  ", " \"", " \"", " {" }, /*left,right delims*/
	*rightdelims[] = {  "\0",  "\"",  "\"",  "}" }; /*for valtype=0,1,2*/
int	tablen=22, maxfldlen=MAXFLDLEN,	/* continuation tab, max field len */
	format=0,  iline=0, fxputs();	/* write lines to file */
static	int entrynum = 0;		/* counter to construct dummy key */
/* -------------------------------------------------------------------------
initialization
-------------------------------------------------------------------------- */
/* --- check input --- */
if ( bibdata==NULL || file==NULL )	/* user failed to supply all input */
  goto end_of_job;			/* so quit */
/* --- get bibentry from bibdata[] --- */
if ( *(bibdata[BIBENTRY]) != '\000' )	/* bibdata has an specified entry */
  strcpy(entry,bibdata[BIBENTRY]);	/* so replace default article entry*/
if ( *entry == '@' )			/* extraneous leading @ */
  strcpy(entry,entry+1);		/* just squeeze it out */
/* --- now look up bibtype in our table --- */
for ( ientry=0; (entryname=bibentrys[ientry].name) != NULL; ientry++ ) {
 if ( *entryname == '@' ) entryname++;	/* bump past extraneous leading @ */
 if ( strcmp(strlower(entry),strlower(entryname)) == 0 ) /* found match */
  break; }				/* so quit looking */
if ( entryname != NULL )		/* found entry type */
  bibfields = bibentrys[ientry].fields;	/* fields comprising bibentry */
/* --- bump counter --- */
entrynum++;				/* count another entry */
/* --- get key (or dummy key), and emit header info with key --- */
if ( *(key = bibdata[CITEKEY])		/* get ptr to identifier/key */
==   '\000' )				/* and if it's an empty string */
  sprintf(key,"reference:%d",entrynum);	/* construct a dummy key */
sprintf(outline,"%s@%s{%s,\n",hdrstub,entry,key); /* header line */
fxputs(outline,file,0);			/* emit hedaer line */
/* -------------------------------------------------------------------------
run thru fields for this entry type
-------------------------------------------------------------------------- */
*outline = '\000';			/* init empty output line */
for ( ifield=0; ; ifield++ ) {		/* fields in bibentry or all fields*/
  /* --- use bibfields[] if we have it, or dump everything --- */
  keytype = 1;				/* default to non-dummy key */
  if ( bibfields !=  NULL ) {		/* have a bibliography entry */
   if ( (fldkey = bibfields[ifield].key) == NULL ) break; /* end-of-fields */
   index = bibfields[ifield].index;	/* get explicitly given index or -1*/
   strcpy(keybuff,fldkey);		/* local copy of bibfields[] key */
   fldkey = keybuff; }			/* and reset ptr to local buffer */
  else {				/* not a defined bibentry */
   if ( (index = ifield)		/* so just run through all our data*/
   >  highindex ) break;		/* and now we're done */
   if ( index==BIBENTRY || index==CITEKEY ) /* these are shown on header */
    continue;				/* so don't repeat them here */
   if ( index>ABSTRACT && index<ABSTRACT+8 ) /*abstract continuation field*/
    continue;				/* skip extra abstract fields */
   fldkey = keybuff;  *keybuff = '\000'; } /* point to dummy key buffer */
  /* --- look up fldkey if not explicitly given --- */
  if ( *fldkey == '\000' ) {		/* no fldkey given if string empty */
   if ( index < 0 ) continue;		/* no index, either, so skip it */
   if ( *(fldkey = getkey(index))	/* get key for this index */
   ==  '\000' ) {			/* no key found for it */
    keytype = 0;			/* signal dummy key */
    fldkey = keybuff;			/* so point to our dummy key buffer*/
    sprintf(fldkey,"field%d",index); } } /* and counstruct dummy key */
  /* --- check fldkey for bibentry directives --- */
  if ( strstr(strlower(fldkey),"optional") != NULL ) /* start optional */
     { isrequired = 0;  continue; }	/* set flag and just skip field */
  if ( strstr(strlower(fldkey),"required") != NULL ) /* start required */
     { isrequired = 1;  continue; }	/* set flag and just skip field */
  /* --- not a bibentry directive, so look up index if not given --- */
  if ( index < 0 )			/* or we can look it up */
   if ( (index = getindex(fldkey))	/* bibdata[] index for fldkey */
   < 0 ) continue;			/* skip field if lookup failed */
  /* --- check bibdata[] data field --- */
  fldval = bibdata[index];		/* ptr to data for fldkey */
  valtype = (-1);			/* re-init valtype */
  if ( *fldval == '\000' )		/* no data for this field */
   { if ( bibfields == NULL		/* just skip it if we're dumping */
     ||   !isrequired ) continue;	/* or if field not required */
     fldval = dummydata;		/* or emit our dummy data */
     valtype = 3; }			/* surrounded with {}'s */
  if ( valtype < 0 ) {			/* valtype not explicitly set */
   if ( isthisword(fldval,DIGITS) ) valtype=0; /*field is entirely numeric*/
    else if ( isthisword(fldval,LETTERS) ) valtype=1; /* or just one word */
     else valtype = 2; }		/* or default to several words */
  /* --- each output field must be limited to maximum bibtex chars --- */
  for ( iline=0; *fldval != '\000'; iline++ ) {
   char *valptr=fldval;  int vallen=strlen(fldval); /*ptr,len of val field*/
   /* --- first emit preceding buffered line --- */
   if ( *outline != '\000' )		/* have preceding output line */
    { strcat(outline,",\n");		/* so first add on comma */
      fxputs(outline,file,format); }	/* and emit it */
   /* --- limit current field to maximum bibtex field length --- */
   if ( vallen > maxfldlen ) {		/* too many chars in field */
    vallen = maxfldlen;			/* so reset to maximum */
    memcpy(valbuff,fldval,vallen);	/* copy over max chars */
    while ( vallen > 0 )		/* and back up to first white space*/
     if ( isthischar(valbuff[vallen-1],WHITESPACE) ) break; /* found it */
     else vallen--;			/* or keep looking backwards for it*/
    if ( vallen < 2 ) vallen = maxfldlen; /* no whitespace to break at */
    valbuff[vallen] = '\000';		/* null-terminate field */
    prune(valbuff,NULL);		/*remove leading/trailing whitespace*/
    valptr = valbuff; }			/* emit buffer, not entire field */
   /* -- construct key for continuation fields --- */
   if ( iline > 0 ) {			/* this is a continuation field */
    int	keylen = strlen(fldkey);	/* #key chars */
    if ( iline == 1 ) {			/* init key on 1st continuation */
     if ( fldkey != keybuff ) strcpy(keybuff,fldkey); /* copy original key */
     else				/* already have kay in keybuff */
      if ( keytype == 0 ) keybuff[keylen++] = '-'; /* add dash to dummy key*/
     fldkey = keybuff;			/* use dummy key for rest of field */
     keylen++; }			/* leave room for continuation char*/
    keybuff[keylen-1] = (" 123456789ABCDEF")[iline]; /* put char in place*/
    keybuff[keylen] = '\000'; }		/* and null-terminate key string */
   /* --- counstruct this line --- */
   stub = (isrequired?reqstub:optstub);	/* select stub for this field */
   sprintf(outline,"%s%-*s =%s%s%s",	/* fldkey = "fldval" for valtype=2 */
    stub,tablen-strlen(stub)-4,fldkey,	/* start with stub and fldkey = */
    leftdelims[valtype],valptr,rightdelims[valtype]);  /* end with {value} */
   if(msglevel>=99) { fprintf(stderr,"<%d> %s\n",iline,outline); }
   format = tablen;			/* set default format */
   if ( strcspn(valptr,WHITESPACE) == vallen ) /* no whitespace */
    format = 0;				/* don't break line at " = " */
   /* --- ready for next line --- */
   fldval += vallen;			/* past chars already written */
   fldval += strspn(fldval,WHITESPACE);	/* and any additional whitespace */
   } /* --- end-of-for(iline) --- */
  } /* --- end-of-for(ifield) --- */
/* -------------------------------------------------------------------------
emit last line and then return
-------------------------------------------------------------------------- */
if ( *outline != '\000' )		/* have last buffered output line */
  { strcat(outline,"  }\n");		/* so tack on closing } */
    fxputs(outline,file,format);	/* and emit last line of entry */
    fxputs("\n",file,0); }		/* also emit blank line separator */
end_of_job:
 return ( 1 );
} /* --- end-of-function makebib() --- */

/* ==========================================================================
 * Function:	char *fxgets ( char *line, int maxlen, FILE *file )
 * Purpose:	like fgets(), but returns concatanated lines from file
 *		as long as the last non-white chars of each line are \ or \\
 *		(which signals that the next line is a continuation line)
 * --------------------------------------------------------------------------
 * Arguments:	line (O)	char *ptr to buffer returning line(s)
 *				read from file (including \n).
 *		maxlen (I)	int containing max #chars (including
 *				terminating \n and \0) line may hold
 *		file (I)	FILE *ptr to file already opened for read.
 * --------------------------------------------------------------------------
 * Returns:	( char * )	line, same as input,
 *				or NULL for any error.
 * --------------------------------------------------------------------------
 * Notes:     o	A single \ at end-of-line signals a continuation line
 *		and an end-of-word, but not an end-of-field.
 *		So a space is inserted, but the \ is not returned in
 *		the concatanated line, e.g., the three lines
 *		  abc \
 *		  def \
 *		  ghi
 *		are returned as one line consisting of
 *		  abc def ghi
 *	      o	The \\ continuation chars signal an end-of-field, and are
 *		returned in the concatanated line, e.g., the three lines
 *		  abc \\
 *		  def \\
 *		  ghi
 *		are returned as one line consisting of
 *		  abc \\ def \\ ghi
 * ======================================================================= */
/* --- entry point --- */
char	*fxgets ( char *line, int maxlen, FILE *file )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
char	inpline[8192],			/* input buffer for fgets() */
	*prune();			/* raw input lines are prune()'d */
int	inplen = 0,			/* #chars in input line */
	len = 0,			/* total #chars so far in line */
	istrunc = 0;			/* true if input overflows line */
char	*continuation = "\\\\";		/* continuation signalled by \\ */
int	contlen = 0;			/* #continuation chars (2 for \\) */
/* -------------------------------------------------------------------------
read (and maybe concatanate) line(s) from file
-------------------------------------------------------------------------- */
if ( file == NULL			/* do nothing if no input file */
||   line == NULL ) goto end_of_job;	/* or no output buffer supplied */
*line = '\000';				/* init output buffer as empty line*/
while ( fgets(inpline,8100,file) != NULL ) /* read next line from file */
  {
  /* --- add a little whitespace if this is a continuation line --- */
  if ( contlen > 0 )			/* preceding line ended with \\ */
   if ( !istrunc ) {			/* and buffer isn't full yet */
    if ( maxlen-len-1 >= 2 )		/* still room for 2 or more chars */
      line[len++] = ' ';		/* one whitespace and >=one data */
    else				/* out of room in line buffer */
      istrunc = 1; }			/* so just set truncation flag */
  /* --- now check for continuation at end of current line --- */
  prune(inpline,NULL);			/* remove leading/trailing space */
  inplen = strlen(inpline);		/* #chars in prune()'d input line */
  contlen = 0;				/* no trailing \'s found yet */
  while ( isthischar(inpline[inplen-1],continuation) ) /* last char is \ */
   { contlen++;				/* count another \ at end-of-line */
     if ( --inplen < 1 ) break; }	/* don't back up before 1st char */
  if ( contlen > 0 ) {			/* if input line ends in \\... */
    inpline[inplen] = '\000';		/* remove the trailing \'s */
    prune(inpline,NULL);		/* re-prune() the input line, */
    inplen = strlen(inpline); }		/* and reset its shortened length */
  /* --- add current input line to caller's output line buffer --- */
  if ( !istrunc ) {			/* there's still room in buffer */
   if ( inplen > maxlen-len-1 )		/* but not enough for all the input*/
     { inplen = maxlen-len-1;		/* reset input to just fill buffer */
       istrunc = 1; }			/* and set truncation flag */
   memcpy(line+len,inpline,inplen);	/* add input line to output buffer */
   len += inplen; }			/* and bump output buffer length */
  if ( contlen < 1 ) break;		/* all done if no continuation */
  /* --- add \\ to output buffer before continuation line --- */
  if ( 1 )				/*yes, we're embedding \\ in buffer*/
   if ( !istrunc ) {			/* there's still room in buffer */
    if ( maxlen-len-1 <= contlen+1 )	/*but not enough for 1 blank and \\*/
      istrunc=1;			/* so just set truncation flag */
    else				/* enough room for 1 blank and \\ */
      if ( contlen > 1 ) {		/* have a double-\\ at end-of-line */
       line[len++] = ' ';		/* so add a blank and bump length */
       strcpy(line+len,continuation);	/* and add \\ to signal new field */
       len += strlen(continuation); } }	/* and bump length */
  } /* --- end-of-while(fgets) --- */
line[len] = '\000';			/* null-terminate output buffer */
end_of_job:
  return ( len<1? NULL : line );	/* output buffer back to caller */
} /* --- end-of-function fxgets() --- */

/* ==========================================================================
 * Function:	int fxputs ( char *line, FILE *file, int format )
 * Purpose:	like fputs(), except for the extra format arg,
 *		which controls whether or not embedded \\ continuations
 *		are written as a single line or split into several
 *		lines (see Notes below).
 * --------------------------------------------------------------------------
 * Arguments:	line (I)	char *ptr to buffer to be written
 *				to file (usually containing terminating \n).
 *		file (I)	FILE *ptr to file already opened for write.
 *		format (I)	int containing 0 to write a single line,
 *				or 1 to write separate continuation lines
 *				(see Notes below).
 * --------------------------------------------------------------------------
 * Returns:	( int )		non-negative number (as returned by fputs()),
 *				or EOF for any error.
 * --------------------------------------------------------------------------
 * Notes:     o	If format=0, line is written to file _exactly_ as input,
 *		identical to fputs().
 *	      o	If format=1, then \\ continuation chars in line signal
 *		separate lines written to file, with each continuation
 *		line preceded by a \t tab character, e.g,
 *		  abc \\ def \\ ghi
 *		is written to file as
 *		  abc \\
 *		     def \\
 *		     ghi
 *	      o	If format>1, then lines longer than 72 (maxcols=72) columns
 *		are split on word boundaries, and continuation lines are
 *		indented by format columns.
 * ======================================================================= */
/* --- entry point --- */
int	fxputs ( char *line, FILE *file, int format )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
char	outline[16384],			/* buffer for continuation line */
	*prune(),			/* continuation lines are prune()'d*/
	*strchange();			/* space at start of continuation */
char	*lptr = line,			/* ptr to 1st char in current line */
	*continuation = "\\\\",		/*continuation line signalled by \\*/
	*cptr = line;			/* ptr to \\ found in input line */
int	maxcols = 72;			/* max line length */
int	outlen = 0,			/*#chars in outline (including \\) */
	contlen = strlen(continuation),	/*#chars in continuation (2 for \\)*/
	iscont = 0;			/*true if writing continuation line*/
int	status = EOF;			/* fputs() status (init for error) */
/* -------------------------------------------------------------------------
check input
-------------------------------------------------------------------------- */
if ( file == NULL			/* do nothing if no output file */
||   line == NULL ) goto end_of_job;	/* or no input line supplied */
/* -------------------------------------------------------------------------
format>=2 breaks long lines on word boundaries
-------------------------------------------------------------------------- */
if ( format >= 2 ) {			/* output format >1 requested */
 int addcont = (format>=100?1:0);	/* true to add \ at before break */
 if ( addcont ) format -= 100;		/* remove addcont signal */
 outlen = strlen(lptr);			/* initial #chars to output */
 while ( outlen > 0 ) {			/* still have chars to output */
   /* --- figure out how much we can print on this line --- */
   int	stublen = (iscont?format:0);	/* #cols for continuation line stub*/
   if ( stublen+outlen > maxcols+4 ) {	/* too much output for this line */
    while ( stublen+outlen > maxcols ) { /* remove one trailing word */
     while ( --outlen > 0 )		/* search backwards for whitespace */
      if ( isthischar(lptr[outlen],WHITESPACE) ) break; /*found whitespace */
     if ( outlen < 1 )			/* unable to break output at space */
      { outlen = strlen(lptr);  break; } /*so just dump it all on this line*/
     } } /* --- end-of-while(stublen+outlen>maxcols) --- */
   while ( outlen > 0 )			/* check for trailing whitespace */
    if ( isthischar(lptr[outlen-1],WHITESPACE) ) outlen--; /*strip it off*/
    else break;				/* but don't strip non-white chars */
   /* --- construct and print current line --- */
   if ( stublen > 0 ) memset(outline,' ',stublen); /* init stub as blanks */
   memcpy(outline+stublen,lptr,outlen);	/* place output chars after stub */
   outline[stublen+outlen] = '\000';	/* null-terminate output line */
   if ( outlen > 0 ) {			/* don't print empty line */
    if ( !addcont ) strcat(outline,"\n"); /* add a \n newline at the end */
    if ( (status=fputs(outline,file))	/* and write line */
    ==   EOF ) goto end_of_job; }	/* abort if write failed */
   /* --- ready to begin next line --- */
   lptr += outlen;			/* start next line after this one */
   lptr += strspn(lptr,WHITESPACE);	/* but skip any leading whitespace */
   outlen = strlen(lptr);		/* remaining #chars to output */
   if ( addcont ) {			/* see if we need \ continuation */
    strcpy(outline,(outlen>0?" \\\n":"\n")); /*set continuation or just \n */
    if ( (status=fputs(outline,file))	/* and write it */
    ==   EOF ) goto end_of_job; }	/* abort if write failed */
   iscont = 1;				/* set continuation flag */
   } /* --- end-of-while(outlen>0) --- */
 goto end_of_job;			/* all done */
 } /* --- end-of-if(format>=2) --- */
/* -------------------------------------------------------------------------
format=1 breaks embedded \\ continuation signals into separate lines
-------------------------------------------------------------------------- */
if ( format == 1 ) {			/* output format 1 requested */
 while ( cptr != NULL ) {		/* breaks after last \\ written */
   /* --- get entire rest of line or up to and including the next \\ --- */
   cptr = strstr(lptr,continuation);	/* look for next \\ in input line */
   outlen = (cptr==NULL? strlen(lptr):	/* write rest of line if no \\ */
	((int)(cptr-lptr))+contlen);	/* or up to an including next \\ */
   memcpy(outline,lptr,outlen);		/* local copy of line to be written*/
   outline[outlen] = '\000';		/* null-terminate it, */
   prune(outline,NULL);			/* prune() it, */
   strcat(outline,"\n");		/* and add a \n newline at the end */
   /* --- precede continuation lines with space --- */
   if ( iscont )			/* this is a continuation line */
    strchange(0,outline,"    ");	/* so precede it with 4 spaces */
   /* --- write constructed line --- */
   if ( 1 )				/* yes, we're breaking long lines */
     status = fxputs(outline,file,(iscont?106:100)); /*with \ continuations*/
   else					/* or we're not breaking long lines */
     status = fputs(outline,file);	/* just write entire line */
   if ( status == EOF ) goto end_of_job; /* abort if write failed */
   /* --- push line ptr past \\ just written (ignored after last \\) --- */
   lptr = cptr+contlen;			/* pick up after \\ just written */
   iscont = 1;				/* and set continuation flag */
   } /* --- end-of-while() --- */
 goto end_of_job;			/* all done */
 } /* --- end-of-if(format==1) --- */
/* -------------------------------------------------------------------------
default format=0 just issues a single fputs() to write entire line
-------------------------------------------------------------------------- */
if ( 1 || format==0 )			/* format 0, or trap any default */
  { status = fputs(lptr,file);		/* just write entire line */
    goto end_of_job; }			/* and we're done */
end_of_job:
  return ( status );			/* back with fputs() status */
} /* --- end-of-function fxputs() --- */

/* ==========================================================================
 * Function:	char *findright ( char *s, char *left, char *right )
 * Purpose:	If *s (leading char of s) is a delim in left,
 *		returns a ptr to the matching right delim in s.
 * --------------------------------------------------------------------------
 * Arguments:	s (I)		char *ptr to null-terminated string
 *				whose leading char is one of the delims
 *				in left (otherwise, findright returns NULL)
 *		left (I)	char *ptr to null-terminated string
 *				containing left delim chars, e.g., "([{<"
 *		right (I)	char *ptr to null-terminated string
 *				containing matching right delim chars
 *				in corresponding left order, e.g., ")]}>"
 * --------------------------------------------------------------------------
 * Returns:	( char * )	ptr within s to matching right delim,
 *				e.g., if s="(abc)..." then findright=s+4.
 *				NULL is returned if no matching right
 *				is found, or if *s is not in left.
 * --------------------------------------------------------------------------
 * Notes:     o	In cases like |...| where right=left, no nesting is
 *		possible, i.e., the very next | will always be interpreted
 *		as the closing right delim matching the initial |.
 * ======================================================================= */
/* --- entry point --- */
char	*findright ( char *s, char *left, char *right )
{
/* --- allocations and declarations --- */
char	leftdelim = (s==NULL?'\000':*s), /* leading char of s is left delim*/
	rightdelim = '\000',		/* corresponding right delim */
	*rightptr = NULL,		/* ptr to matching right delim in s*/
	schar = '\000';			/* current char from s */
int	nestlevel = 1;			/* ((...(...)...)) nesting level */
/* --- first determine corresponding right delim we're looking for --- */
if ( leftdelim == '\000' ) goto end_of_job; /* have no left delim to match */
if ( left==NULL || right==NULL ) goto end_of_job; /*or nothing to match with*/
if ( (rightptr=strchr(left,leftdelim))	/* look for leftdelim in left */
==   NULL ) goto end_of_job;		/* failed to find leftdelim in left*/
rightdelim = right[(int)(rightptr-left)]; /* corresponding delim in right */
/* --- now look for matching right delim in s --- */
while ( (schar = *(++s)) != '\000' ) {	/* keep looking till end of string */
  if ( schar == rightdelim )		/* found a right delim */
    if ( --nestlevel <= 0 )		/* and it matches original left */
      {	rightptr = s;			/* so set ptr */
	goto end_of_job; }		/* and return it to caller */
  if ( schar == leftdelim )		/* found another left delim */
    nestlevel++;			/* so bump nesting level */
  } /* --- end-of-while() --- */
rightptr = NULL;			/* no matching right delim found */
end_of_job:
  return ( rightptr );			/* back with ptr to right delim */
} /* --- end-of-function findright() --- */

/* ==========================================================================
 * Function:	char *strchange ( int nfirst, char *from, char *to )
 * Purpose:	Changes the nfirst leading chars of `from` to `to`.
 *		For example, to change char x[99]="12345678" to "123ABC5678"
 *		call strchange(1,x+3,"ABC")
 * --------------------------------------------------------------------------
 * Arguments:	nfirst (I)	int containing #leading chars of `from`
 *				that will be replace by `to`
 *		from (I/O)	char * to null-terminated string whose nfirst
 *				leading chars will be replaced by `to`
 *		to (I)		char * to null-terminated string that will
 *				replace the nfirst leading chars of `from`
 * --------------------------------------------------------------------------
 * Returns:	( char * )	ptr to first char of input `from`
 *				or NULL for any error.
 * --------------------------------------------------------------------------
 * Notes:     o	If strlen(to)>nfirst, from must have memory past its null
 *		(i.e., we don't do a realloc)
 * ======================================================================= */
/* --- entry point --- */
char	*strchange ( int nfirst, char *from, char *to )
{
/* -------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	tolen = (to==NULL?0:strlen(to)), /* #chars in replacement string */
	nshift = abs(tolen-nfirst);	/*need to shift from left or right*/
/* -------------------------------------------------------------------------
shift from left or right to accommodate replacement of its nfirst chars by to
-------------------------------------------------------------------------- */
if ( tolen < nfirst )			/* shift left is easy */
  strcpy(from,from+nshift);		/* because memory doesn't overlap */
if ( tolen > nfirst )			/* need more room at start of from */
  { char *pfrom = from+strlen(from);	/* ptr to null terminating from */
    for ( ; pfrom>=from; pfrom-- )	/* shift all chars including null */
      *(pfrom+nshift) = *pfrom; }	/* shift chars nshift places right */
/* -------------------------------------------------------------------------
from has exactly the right number of free leading chars, so just put to there
-------------------------------------------------------------------------- */
if ( tolen != 0 )			/* make sure to not empty or null */
  memcpy(from,to,tolen);		/* chars moved into place */
return ( from );			/* changed string back to caller */
} /* --- end-of-function strchange() --- */

/* ==========================================================================
 * Function:	strreplace (char *string, char *from, char *to, int nreplace)
 * Purpose:	Changes the first nreplace occurrences of 'from' to 'to'
 *		in string, or all occurrences if nreplace=0.
 * --------------------------------------------------------------------------
 * Arguments:	string (I/0)	char *ptr to null-terminated string in which
 *				occurrence of 'from' will be replaced by 'to'
 *		from (I)	char * to null-terminated string
 *				to be replaced by 'to'
 *		to (I)		char *ptr to null-terminated string that will
 *				replace 'from'
 *		nreplace (I)	int containing (maximum) number of
 *				replacements, or 0 to replace all.
 * --------------------------------------------------------------------------
 * Returns:	( int )		number of replacements performed,
 *				or 0 for no replacements or -1 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
int	strreplace ( char *string, char *from, char *to, int nreplace )
{
/* -------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
int	fromlen = (from==NULL?0:strlen(from)), /* #chars to be replaced */
	tolen = (to==NULL?0:strlen(to)); /* #chars in replacement string */
char	*pfrom = (char *)NULL,		/*ptr to 1st char of from in string*/
	*pstring = string,		/*ptr past previously replaced from*/
	*strchange();			/* change 'from' to 'to' */
int	nreps = 0;			/* #replacements returned to caller*/
/* -------------------------------------------------------------------------
repace occurrences of 'from' in string to 'to'
-------------------------------------------------------------------------- */
if ( string == (char *)NULL		/* no input string */
||   (fromlen<1 && nreplace<=0) )	/* replacing empty string forever */
  nreps = (-1);				/* so signal error */
else					/* args okay */
  while (nreplace<1 || nreps<nreplace)	/* up to #replacements requested */
    {
    if ( fromlen > 0 )			/* have 'from' string */
      pfrom = strstr(pstring,from);	/*ptr to 1st char of from in string*/
    else  pfrom = pstring;		/*or empty from at start of string*/
    if ( pfrom == (char *)NULL ) break;	/*no more from's, so back to caller*/
    if ( strchange(fromlen,pfrom,to)	/* leading 'from' changed to 'to' */
    ==   (char *)NULL ) { nreps=(-1); break; } /* signal error to caller */
    nreps++;				/* count another replacement */
    pstring = pfrom+tolen;		/* pick up search after 'to' */
    if ( *pstring == '\000' ) break;	/* but quit at end of string */
    } /* --- end-of-while() --- */
return ( nreps );			/* #replacements back to caller */
} /* --- end-of-function strreplace() --- */

/* ==========================================================================
 * Function:	delimreplace ( char *string, char **from, char **to )
 * Purpose:	Changes matching pairs of from delimiters to to delimiters;
 *		unmatched left delimiters...
 * --------------------------------------------------------------------------
 * Arguments:	string (I/0)	char *ptr to null-terminated string in which
 *				matching from delimters are replaced by to's.
 *		from (I)	char *[] to array of null-terminated strings,
 *				where from[0],from[1] are the first pair
 *				of left,right delimiters to be replaced,
 *				from[2],from[3] the second pair, etc, till
 *				a NULL (see Notes below)
 *		to (I)		char *[] to array of null-terminated strings,
 *				where to[0],to[1] replaces from[0],from[1],
 *				etc.  (Make sure to[] is populated with
 *				as many strings as from[])
 * --------------------------------------------------------------------------
 * Returns:	( string )	same as input
 * --------------------------------------------------------------------------
 * Notes:     o	Only the first char of the from delimiters is used,
 *		but the entire to string replaces it, e.g., you can replace
 *		"..." by ``...'' using from[0]=from[1]="\"" and to[0]="``",
 *		to[1]="\'\'".  But the inverse replacement is impossible
 *		because delimreplace() would only use the first "`" and "\'".
 * ======================================================================= */
/* --- entry point --- */
char	*delimreplace ( char *string, char *from[], char *to[] )
{
/* -------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
char	*findright(),			/* find from[1] given from[0], etc */
	*strchange(),			/* change from[0] to to[0], etc */
	*s = string,			/* start at beginning of string */
	*left=NULL, *right=NULL;	/* left and right from delims */
int	ifrom = 0;			/* from[] index */
/* -------------------------------------------------------------------------
process one from[i],from[i+1] delimiter pair at a time
-------------------------------------------------------------------------- */
if ( string!=NULL && from!=NULL && to!=NULL )  /* check caller's input */
 for ( ifrom=0; (left=from[ifrom])!=NULL; ifrom += 2 ) { /*get from[] pairs*/
  if ( (right = from[ifrom+1]) == NULL ) break; /* must be in pairs */
  /* --- replace all left,right pairs in string --- */
  s = string;				/* restart at beginning of string */
  while ( *s != '\000' ) {		/* and keep going till its end */
   char	*found = findright(s,left,right); /* check for a left,right pair */
   if ( found == NULL ) s++;		/* no pair, so move to next char */
   else {				/* found a delim pair to replace */
    char *toleft=to[ifrom], *toright=to[ifrom+1]; /* replacement strings */
    strchange(1,found,toright);		/* replace right delim first */
    strchange(1,s,toleft);		/* then left delim */
    s = found + (toleft==NULL?0:strlen(toleft)) /* bump past new left */
      + (toright==NULL?0:strlen(toright)) - 1; } /* and right, plus 1 char */
   } /* --- end-of-while(*s!=NULL) --- */
  } /* --- end-of-for(ifrom) --- */
return ( string );			/* back with edited string */
} /* --- end-of-function delimreplace() --- */

/* ==========================================================================
 * Function:	char **makewords ( char *s, char *white )
 * Purpose:	breaks s into individual whitespace-delimited words,
 *		either delimited by standard whitespace or by any
 *		character in user-supplied white.
 * --------------------------------------------------------------------------
 * Arguments:	s (I)		char *ptr to null-terminated string
 *				whose decomposition into words is wanted
 *		white (I)	char *ptr to null-terminated string
 *				containing chars to also be considered
 *				whitespace, e.g., ".,()=", or NULL.
 * --------------------------------------------------------------------------
 * Returns:	( char ** )	array of ptrs to null-terminated strings,
 *				each comprising a word of s, with a NULL
 *				ptr after the last.
 * --------------------------------------------------------------------------
 * Notes:     o	ptrs into a static buffer are returned,
 *		not individually malloc()'ed words.
 * ======================================================================= */
/* --- entry point --- */
char	**makewords ( char *s, char *white )
{
/* --- allocations and declarations --- */
static	char words[1024][768];		/* word buffer */
static	char *pwords[1024];		/*words[0],...,words[nwords-1],NULL*/
char	whitespace[512];		/* chars considered whitespace */
int	nwords = 0,			/* number of words in s */
	wordlen = 0;			/* #chars in current word */
/* --- init whitespace --- */
strcpy(whitespace,WHITESPACE);		/* standard whitespace */
if ( white != NULL )			/* caller has some more */
  strcat(whitespace,white);		/* so they're whitespace, too */
/* --- separate s into individual whitespace-delimited words --- */
if ( s != NULL )			/* make sure we're given input */
 while ( nwords < 1020 ) {		/* don't overflow word buffer */
   s += strspn(s,whitespace);		/* skip any leading whitespace */
   wordlen = strcspn(s,whitespace);	/* #chars preceding next whitespace*/
   if ( wordlen < 1 ) break;		/* end-of-string */
   memcpy(words[nwords],s,wordlen);	/* copy chars comprising word */
   words[nwords][wordlen] = '\000';	/* and null-terminate it */
   pwords[nwords] = words[nwords];	/* set ptr to this word */
   s += wordlen;			/* push s past this word */
   nwords++; }				/* and bump word count */
pwords[nwords] = NULL;			/* NULL ptr signals end-of-list */
return ( pwords );			/*back to caller with list of words*/
} /* --- end-of-function makewords() --- */

/* ==========================================================================
 * Function:	char **maketokens(char *s,char *delims,char *left,char *right)
 * Purpose:	breaks s into individual tokens separated by delims,
 *		but not broken within left...right brackets.
 * --------------------------------------------------------------------------
 * Arguments:	s (I)		char *ptr to null-terminated string
 *				whose decomposition into tokens is wanted
 *		delims (I)	char *ptr to null-terminated string
 *				containing chars to be considered
 *				as token terminators
 *		left (I)	char *ptr to null-terminated string
 *				containing left bracket chars, e.g., "([{<"
 *				within which tokens will not be terminated
 *		right (I)	char *ptr to (null-terminated) string
 *				containing matching right bracket chars
 *				in corresponding left order, e.g., ")]}>"
 * --------------------------------------------------------------------------
 * Returns:	( char ** )	array of ptrs to null-terminated strings,
 *				each comprising a token of s, with a NULL
 *				ptr after the last.
 * --------------------------------------------------------------------------
 * Notes:     o	ptrs into a static buffer are returned,
 *		not individually malloc()'ed tokens.
 *	      o left and right may be passed as NULL's
 *		to ignore that functionality
 *	      o	delims within left/right {}'s, ()'s, etc
 *		do not signal token breaks
 * ======================================================================= */
/* --- entry point --- */
char	**maketokens ( char *s, char *delims, char *left, char *right )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
static	char tokens[2048][1024];	/* tokens buffer */
static	char *ptokens[2048];		/* tokens[0],...,[ntokens-1],NULL */
int	ntokens = 0;			/* number of tokens in s */
char	*startptr = s,			/* tokens start at beginning of s */
	*endptr = s,			/* and will end on a delim */
	*rightptr=NULL, *findright();	/* but not within left/right */
char	*prune();			/* prune completed tokens */
/* -------------------------------------------------------------------------
separate s into individual tokens
-------------------------------------------------------------------------- */
if ( s != NULL				/* make sure caller supplied input */
&&   delims != NULL ) {			/* and delims */
 while ( ntokens < 2040 )		/* don't overflow token buffer */
  if ( (rightptr=findright(endptr,left,right)) /* look for matching right */
  !=   NULL )				/* found one */
   endptr = rightptr+1;			/* so just set ptr past right delim*/
  else					/* or else see if we have a delim */
   if ( !isthischar(*endptr,delims)	/* nope, this char isn't a delim */
   && *endptr != '\000' )		/* and we haven't finished string */
    endptr++;				/* so just set ptr to next char */
   else {				/* have a token-terminating delim */
    int tokenlen = (int)(endptr-startptr); /* #chars in current token */
    memcpy(tokens[ntokens],startptr,tokenlen); /*copy chars comprising token*/
    tokens[ntokens][tokenlen] = '\000';	/* and null-terminate it */
    prune(tokens[ntokens],NULL);	/* prune completed token */
    ptokens[ntokens] = tokens[ntokens];	/* finally, set ptr to this token */
    ntokens++;				/* and bump token count */
    if ( *endptr == '\000' ) break;	/* done after finishing string */
    startptr = ++endptr;		/* start new token after delim */
    } /* --- end-of-if/else(!isthischar) --- */
 } /* --- end-of-if(s) --- */
ptokens[ntokens] = NULL;		/* NULL ptr signals end-of-list */
return ( ptokens );			/* back to caller with tokens in s */
} /* --- end-of-function maketokens() --- */

/* ==========================================================================
 * Function:	char *purge ( char *s )
 * Purpose:	First removes all sequences of the form [nnn] from s,
 *		where nnn is a 1- to 3-digit decimal number.
 *		Then converts or removes any non-printable chars from s.
 * --------------------------------------------------------------------------
 * Arguments:	s (I/O)		char *ptr to null-terminated string
 *				to be purged.
 * --------------------------------------------------------------------------
 * Returns:	( char * )	s, same as input
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
char	*purge ( char *s )
{
/* -------------------------------------------------------------------------
allocations and declarations
-------------------------------------------------------------------------- */
char	*sptr = s,			/* start at beginning of string */
	*lbrack = NULL,			/* ptr to 1st [ after sptr */
	*rbrack = NULL,			/* ptr to 1st ] after lbrack */
	*endptr=NULL;			/* ptr to 1st non-digit between [] */
int	len = 0,			/* #chars between [] */
	val = 0;			/* numeric value between [] */
/* -------------------------------------------------------------------------
remove all [nnn] sequences from s
-------------------------------------------------------------------------- */
if ( s == NULL ) goto end_of_job;	/* do nothing if no input supplied */
while ( (lbrack=strchr(sptr,'[')) != NULL ) {  /* found an opening left [ */
  if ( (rbrack=strchr(lbrack+1,']'))	/* look for matching right ] */
  ==   NULL )  break;			/* finished if no match found */
  /* --- check for 1- to 3-character field between [] --- */
  len = ((int)(rbrack-lbrack)) - 1;	/* #chars between [] */
  if ( len<1 || len>3 )			/* too short/long for us to purge */
   { sptr=rbrack+1;			/* so set ptr just after right ] */
     continue; }			/* and resume search */
  /* --- check for numeric value between [] --- */
  val = ((int)(strtol(lbrack+1,&endptr,10))); /* span digits */
  if ( endptr!=rbrack )			/* invalid number between []'s */
   { sptr=rbrack+1;			/* so set ptr just after right ] */
     continue; }			/* and resume search after [stuff] */
  /* --- remove [nnn] from string --- */
  strcpy(lbrack,rbrack+1);		/* squeeze out [nnn] bracket */
  sptr = lbrack;			/* and resume search */
  } /* --- end-of-while() --- */
end_of_job:
  return ( s );				/* purged s back to caller */
} /* --- end-of-function purge() --- */

/* ==========================================================================
 * Function:	char *prune ( char *s, char *white )
 * Purpose:	removes all leading and trailing isspace() chars
 *		from the front and back of s (embedded space unchanged),
 *		as well as any leading/trailing chars in white
 * --------------------------------------------------------------------------
 * Arguments:	s (I/O)		char *ptr to null-terminated string
 *				to be pruned.
 *		white (I)	char *ptr to null-terminated string
 *				containing _additional_ chars to be
 *				pruned, e.g., ".,()", or NULL.
 * --------------------------------------------------------------------------
 * Returns:	( char * )	s, same as input
 * --------------------------------------------------------------------------
 * Notes:     o
 * ======================================================================= */
/* --- entry point --- */
char	*prune ( char *s, char *white )
{
/* --- initial length of input string --- */
int	len = (s==NULL?0:strlen(s));	/* length of input string */
if ( len < 1 ) goto end_of_job;		/* do nothing if no input supplied */
/* --- first eliminate trailing space --- */
while ( isspace(s[len-1])		/* last char of s is space */
|| isthischar(s[len-1],white) )		/* or in user-supplied white */
  { s[--len] = '\000';			/* remove it and decrement len */
    if ( len < 1 ) break; }		/* stop if no remaining chars */
/* --- now eliminate leading space --- */
while ( isspace(*s)			/* first char of s is space */
|| isthischar(*s,white) )		/* or in user-supplied white */
  strcpy(s,s+1);			/* so squeeze it out */
end_of_job:
  return ( s );				/* pruned s back to caller */
} /* --- end-of-function prune() --- */

/* ==========================================================================
 * Function:	char *strip ( char *s, char *white )
 * Purpose:	makes a copy of s with all chars in white removed
 * --------------------------------------------------------------------------
 * Arguments:	s (I/O)		char *ptr to null-terminated string
 *				to be pruned.
 *		white (I)	char *ptr to null-terminated string
 *				containing _additional_ chars to be
 *				pruned, e.g., ".,()", or NULL.
 * --------------------------------------------------------------------------
 * Returns:	( char * )	s, same as input
 * --------------------------------------------------------------------------
 * Notes:     o	a static buffer is returned,
 *		not an indiviually malloc()'ed copy
 * ======================================================================= */
/* --- entry point --- */
char	*strip ( char *s, char *white )
{
/* --- allocations and declarations --- */
static	char buff[4096];		/*stripped copy of s back to caller*/
char	*pstrip = buff;			/* ptr past last char of copy */
/* --- copy all non-white chars from s to buff --- */
if ( s != NULL )			/* check that caller supplied input*/
 for ( pstrip=buff; *s!='\000'; s++ )	/* till end of input string */
  if ( !isthischar(*s,white) )		/* have a non-whitespace char */
   *pstrip++ = *s;			/* so copy it and bump ptr */
*pstrip = '\000';			/* null-terminate stripped copy */
return ( buff );			/* stripped copy back to caller */
} /* --- end-of-function strip() --- */

/* ==========================================================================
 * Function:	char *strlower ( char *s )
 * Purpose:	returns a lowercase copy of s (which is unchanged)
 * --------------------------------------------------------------------------
 * Arguments:	s (I)		char *ptr to null-terminated string
 *				for which a lowercase copy is wanted
 * --------------------------------------------------------------------------
 * Returns:	( char * )	lowercase copy of s, or NULL for error
 * --------------------------------------------------------------------------
 * Notes:     o	the address of a static buffer is returned,
 *		not an individually malloc()'ed copy.
 * ======================================================================= */
/* --- entry point --- */
char	*strlower ( char *s )
{
/* --- allocations and declarations --- */
static	char slow[4][4096];		/* buffers for lowercase copy of s */
static	int  ibuff = 999;		/* index of current buffer */
char	*pslow = slow[0];		/* current char in copy */
/* --- choose buffer --- */
if ( ++ibuff >= 4 ) ibuff = 0;		/* index of wrap-around buffer */
pslow = slow[ibuff];			/* ptr to current buffer */
/* --- lowercase s --- */
if ( s != NULL )			/* check input */
 while ( *s != '\000' )			/* till end-of-string */
  { *pslow++ = (isalpha(*s)? tolower(*s) : *s); /*lowercase each alpha char*/
    s++; }				/* and move on to next char */
*pslow = '\000';			/* null-terminate returned string */
return ( slow[ibuff] );			/* back to caller with lowercase s */
} /* --- end-of-function strlower() --- */
/* --- end-of-file arxivbib.c --- */
