/*
 * Copyright  2001-2004 The Apache Software Foundation
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

package org.apache.tools.ant.taskdefs;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.condition.Condition;
import org.apache.tools.ant.taskdefs.condition.ConditionBase;

/**
 * Task to set a property conditionally using &lt;uptodate&gt;, &lt;available&gt;,
 * and many other supported conditions.
 *
 * <p>This task supports boolean logic as well as pluggable conditions
 * to decide, whether a property should be set.</p>
 *
 * <p>This task does not extend Task to take advantage of
 * ConditionBase.</p>
 *
 * @author Stefan Bodewig
 * @version $Revision: 1.16.2.3 $
 *
 * @since Ant 1.4
 *
 * @ant.task category="control"
 */
public class ConditionTask extends ConditionBase {

    private String property = null;
    private String value = "true";

    /**
     * The name of the property to set. Required.
     * @param p the name of the property
     * @since Ant 1.4
     */
    public void setProperty(String p) {
        property = p;
    }

    /**
     * The value for the property to set, if condition evaluates to true.
     * Defaults to "true".
     * @param v the value of the property
     * @since Ant 1.4
     */
    public void setValue(String v) {
        value = v;
    }

    /**
     * See whether our nested condition holds and set the property.
     *
     * @since Ant 1.4
     * @exception BuildException if an error occurs
     */
    public void execute() throws BuildException {
        if (countConditions() > 1) {
            throw new BuildException("You must not nest more than one "
                + "condition into <condition>");
        }
        if (countConditions() < 1) {
            throw new BuildException("You must nest a condition into "
                + "<condition>");
        }
        if (property == null) {
            throw new BuildException("The property attribute is required.");
        }

        Condition c = (Condition) getConditions().nextElement();
        if (c.eval()) {
            log("Condition true; setting " + property + " to " + value,
                Project.MSG_DEBUG);
            getProject().setNewProperty(property, value);
        } else {
            log("Condition false; not setting " + property,
                Project.MSG_DEBUG);
        }
    }
}
